# coding: utf-8
from __future__ import print_function

import collections
import itertools
import os
import pickle
from datetime import datetime, timedelta

import click
import matplotlib.pyplot as plt
import nanny_rpc_client
import numpy as np
import pandas as pd
import six
import vcr
from infra.awacs.proto import api_pb2, api_stub
from six.moves import range
from six.moves import zip


DEFAULT_AWACS_RPC_URL = 'https://awacs.yandex-team.ru/api/'
CASSETTES_DIR = './cassettes'
DEFAULT_DAYS = 90
DEFAULT_MODE = 'common'

stat_vcr = vcr.VCR(
    cassette_library_dir=CASSETTES_DIR,
    record_mode=os.getenv('VCR_RECORD_MODE', default='new_episodes'),
    filter_headers=['authorization'],
    match_on=['method', 'scheme', 'host', 'port', 'path', 'query', 'raw_body'],
)


def get_load_statistics_entry(awacs_rpc, dt):
    with stat_vcr.use_cassette(str(dt.date()) + '.yml'):
        statistics_stub = api_stub.AwacsStatisticsServiceStub(awacs_rpc)
        req_pb = api_pb2.GetLoadStatisticsEntryRequest()
        req_pb.start.FromDatetime(dt)
        resp_pb = statistics_stub.get_load_statistics_entry(req_pb)
        return resp_pb.entry


def get_usage_statistics_entry(awacs_rpc, dt):
    with stat_vcr.use_cassette(str(dt.date()) + '.yml'):
        statistics_stub = api_stub.AwacsStatisticsServiceStub(awacs_rpc)
        req_pb = api_pb2.GetUsageStatisticsEntryRequest()
        req_pb.start.FromDatetime(dt)
        resp_pb = statistics_stub.get_usage_statistics_entry(req_pb)
        return resp_pb.entry


@click.group()
@click.option('--awacs-token', envvar='AWACS_TOKEN', required=True)
@click.option('--awacs-rpc-url', envvar='AWACS_RPC_URL', default=DEFAULT_AWACS_RPC_URL)
@click.pass_context
def cli(ctx, awacs_rpc_url, awacs_token):
    awacs_rpc = nanny_rpc_client.RetryingRpcClient(rpc_url=awacs_rpc_url, oauth_token=awacs_token)
    ctx.obj = awacs_rpc


def pie_to_counter(pie_pb):
    rv = collections.Counter()
    rv.update(pie_pb.slices)
    return rv


def count_counter_to_hist(c):
    rv = collections.Counter()
    for lines, n in six.iteritems(c):
        rv[int(lines) / 10 * 10] += n
    return rv


def plot():
    ts = pd.Series(np.random.randn(1000),
                   index=pd.date_range('1/1/2000', periods=1000))
    ts = ts.cumsum()
    plt.savefig('./privet.png')


def find_pie_by_id(pie_pbs, id):
    for pie_pb in pie_pbs:
        if pie_pb.id == id:
            return pie_pb


NS_UPSTREAMS_HREF_TEMPLATE = 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/upstreams/list/'


@cli.command('show_load_statistics')
@click.pass_obj
@click.option('--days', default=DEFAULT_DAYS)
def show_load_statistics(awacs_rpc, days):
    end = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)
    start = end - timedelta(days=days)

    dts = pd.date_range(start=start, end=end, freq='D')
    for i, dt in enumerate(dts):
        data_pb = get_load_statistics_entry(awacs_rpc, dt).rps_statistics
        print('Processing', dt.date())
        print(data_pb)


@cli.command('ls_today_load_statistics')
@click.pass_obj
def ls_today_load_statistics(awacs_rpc):
    dt = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)
    data_pb = get_load_statistics_entry(awacs_rpc, dt).date_statistics
    max_ = 0
    avg_ = 0
    # with open('./theemnsids.txt', 'r') as f:
    #    theem_ns_ids = {line.strip() for line in f.readlines()}
    for namespace_id, entry_pb in sorted(six.iteritems(data_pb.by_namespace)):
        # if namespace_id not in theem_ns_ids:
        #    continue
        print(namespace_id, int(entry_pb.max + 1))
        # max_ += entry_pb.max
        # avg_ += entry_pb.average
    # print('Max', max_)
    # print('Avg', avg_)


@cli.command('ls_today_usage_statistics')
@click.pass_obj
def ls_today_usage_statistics(awacs_rpc):
    dt = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)
    data_pb = get_usage_statistics_entry(awacs_rpc, dt).date_statistics
    # Just an example:
    pie_pb = find_pie_by_id(data_pb.total.balancer_pies, 'l7_macro_versions')
    print('Balancer counts by l7_macro version')
    for k, v in sorted(pie_to_counter(pie_pb).items()):
        print('{}: {}'.format(k, v))


@cli.command('plot_load_statistics')
@click.pass_obj
@click.option('--days', default=DEFAULT_DAYS)
def plot_load_statistics(awacs_rpc, days):
    end = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)
    start = end - timedelta(days=days)

    avg_rpss = []
    max_rpss = []
    exclusion_dates = []
    dts = pd.date_range(start=start, end=end, freq='D')
    for i, dt in enumerate(dts):
        try:
            data_pb = get_load_statistics_entry(awacs_rpc, dt).date_statistics
        except nanny_rpc_client.exceptions.NotFoundError:
            exclusion_dates.append(dt)
            print(dt, 'not found')
            continue
        total_avg_rps = 0
        total_max_sum_rps = 0
        for _, ns_pb in data_pb.by_namespace.items():
            total_avg_rps += ns_pb.average
            total_max_sum_rps += ns_pb.max
        x = int(total_avg_rps) / 1000
        avg_rpss.append(x)
        y = int(total_max_sum_rps) / 1000
        max_rpss.append(y)
        print('total_avg_rps', x)
        print('total_max_rps', y)
    dts = dts.drop(exclusion_dates)

    lines_columns = collections.OrderedDict([
        ('Daily sum of per-namespace 24h-averages', avg_rpss),
        ('Daily sum of per-namespace 24h-maximums', max_rpss),
    ])
    x = np.array(lines_columns.values())
    columns = lines_columns.keys()
    df = pd.DataFrame(data=np.transpose(x), index=dts, columns=columns)
    ax = df.plot(title='RPS on awacs managed balancers',
                 figsize=(16, 10),
                 yticks=np.arange(start=0, stop=max(max_rpss) + 10, step=1000),
                 grid=True)
    ax.set_xlabel('Date')
    ax.set_ylabel('KRPS')
    ax.legend(loc='upper left')
    plt.show()
    plt.savefig('./rps.png')


@cli.command()
@click.pass_obj
@click.option('--days', default=DEFAULT_DAYS)
@click.option('--mode', default=DEFAULT_MODE)
@click.option('--exclude-taxi', is_flag=True, default=False)
def plot(awacs_rpc, days, mode, exclude_taxi):
    end = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)
    start = end - timedelta(days=days)

    daily_dts = pd.date_range(start=start, end=end, freq='D')
    total_l7_ns_counts = []
    total_l7_ns_without_taxi_counts = []
    total_ns_counts = []
    total_l3_counts = []
    yp_lite_ns_counts = []
    gencfg_ns_counts = []
    unistat_ns_counts = []
    alerting_enabled_counts = []
    sd_ns_counts = []
    tlem_ns_counts = []
    is_not_tlem_ns_counts = []
    uem_ns_counts = []
    is_not_uem_ns_counts = []
    qs_ns_counts = []
    cert_ns_counts = []
    domain_ns_counts = []
    theem_ns_counts = []
    is_not_theem_ns_counts = []
    theem_wo_compat_ns_counts = []
    theem_ns_compat_opts_counts = []
    top_level_compat_ns_counts = []
    upstream_compat_ns_counts = []
    with_compat_ns_counts = []
    tlem_ns_w_sslv3_counts = []
    balancer2_gpb_counts = []
    balancer2_wo_arl_gpb_counts = []
    balancer2_wo_arl_and_wm_gpb_counts = []
    total_instances_counts = []
    sd_only_l7_namespace_counts = []
    yp_es_only_l7_namespace_counts = []
    total_backends_counts = []
    gencfg_groups_backends_counts = []
    nanny_snapshots_backends_counts = []
    manual_backends_counts = []
    yp_endpoint_sets_backends_counts = []
    yp_endpoint_sets_sd_backends_counts = []


    easy_upstream_counts = []
    total_upstreams_counts = []
    ns_w_at_least_one_easy_upstream_counts = []
    ns_w_at_least_one_easy_domain_counts = []

    arl_affected_ns_counts = []
    old_gencfg_namespace_ids = set()
    curr_gencfg_namespace_ids = set()

    curr_theem_ns_ids = set()
    first_ns_ids = None
    last_ns_ids = None

    ignored_infixes = (
        'maps.yandex.net',
        '.taxi.tst.yandex.net', '.taxi.yandex.net',
        '.lavka.tst.yandex.net', '.lavka.yandex.net',
        '.eda.tst.yandex.net', '.eda.yandex.net'
    )

    exclusion_dates = []

    def calc_stats(exclude_taxi):
        for i, dt in enumerate(daily_dts):
            curr_theem_ns_ids = set()
            try:
                data_pb = get_usage_statistics_entry(awacs_rpc, dt).date_statistics
            except nanny_rpc_client.exceptions.NotFoundError:
                exclusion_dates.append(dt)
                print(dt, 'not found')
                continue
            print('Processing', dt.date())

            arl_affected_ns_count = 0
            total_l3_count = data_pb.total.l3_balancers_count
            total_l7_ns_count = 0
            total_l7_ns_without_taxi_count = 0
            total_ns_count = 0
            yp_lite_ns_count = 0
            gencfg_ns_count = 0
            unistat_ns_count = 0
            sd_ns_count = 0
            tlem_ns_count = 0
            is_not_tlem_ns_count = 0
            uem_ns_count = 0
            is_not_uem_ns_count = 0
            theem_ns_count = 0
            is_not_theem_ns_count = 0
            theem_wo_compat_ns_count = 0
            theem_ns_compat_opts_count = 0
            tlem_ns_w_sslv3_count = 0
            qs_ns_count = 0
            top_level_compat_ns_count = 0
            upstream_compat_ns_count = 0
            with_compat_ns_count = 0
            cert_ns_count = 0
            domain_ns_count = 0
            total_domains_count = 0
            easy_upstream_count = 0
            total_upstreams_count = 0
            ns_w_at_least_one_easy_upstream_count = 0
            ns_w_at_least_one_easy_domain_count = 0
            total_instances_count = 0
            sd_only_l7_namespace_count = 0
            yp_es_only_l7_namespace_count = 0

            total_backends_count = 0
            gencfg_groups_backends_count = 0
            nanny_snapshots_backends_count = 0
            manual_backends_count = 0
            yp_endpoint_sets_backends_count = 0
            yp_endpoint_sets_sd_backends_count = 0

            balancer2_gpb_ns_count = 0
            balancer2_wo_arl_gpb_ns_count = 0
            balancer2_wo_arl_and_wm_gpb_ns_count = 0

            ns_with_l3_only_count = 0
            ns_with_l7 = 0

            if i == 0:
                first_ns_ids = set(data_pb.by_namespace)
            last_ns_ids = set(data_pb.by_namespace)
            for n_id, n_pb in six.iteritems(data_pb.by_namespace):
                is_ignored_ns = any(infx in n_id for infx in ignored_infixes)

                if n_id in ('s.yandex-team.ru', 'tap-testing'):
                    continue
                if exclude_taxi and is_ignored_ns:
                    continue
                if not n_pb.balancers_count and n_pb.l3_balancers_count:
                    ns_with_l3_only_count += 1
                if n_pb.balancers_count == 0:
                    continue
                else:
                    ns_with_l7 += 1
                total_l7_ns_count += 1

                if not is_ignored_ns:
                    total_l7_ns_without_taxi_count += 1

                if n_pb.certs_count > 0:
                    cert_ns_count += 1

                if hasattr(n_pb, 'domains_count') and n_pb.domains_count > 0:
                    domain_ns_count += 1
                    total_domains_count += n_pb.domains_count

                instances_count_pie = find_pie_by_id(n_pb.balancer_pies, 'instances_count')
                for k, v in six.iteritems(instances_count_pie.slices):
                    total_instances_count += int(k) * v

                yp_lite_balancers_count = find_pie_by_id(n_pb.balancer_pies, 'platform').slices.get('YP_LITE', 0)
                gencfg_balancers_count = find_pie_by_id(n_pb.balancer_pies, 'platform').slices.get('GENCFG', 0)
                is_yp_lite_ns = n_pb.balancers_count == yp_lite_balancers_count
                is_gencfg_ns = n_pb.balancers_count == gencfg_balancers_count
                if is_yp_lite_ns:
                    yp_lite_ns_count += 1
                elif is_gencfg_ns:
                    gencfg_ns_count += 1
                    if i == 0:
                        old_gencfg_namespace_ids.add(n_id)
                    elif i == len(daily_dts) - 1:
                        curr_gencfg_namespace_ids.add(n_id)
                else:
                    pass

                unistat_balancers_count = find_pie_by_id(n_pb.balancer_pies, 'is_unistat_enabled').slices.get('True', 0)
                is_unistat_ns = n_pb.balancers_count == unistat_balancers_count
                if is_unistat_ns:
                    unistat_ns_count += 1

                def pie_to_sum(pie):
                    rv = 0
                    for k, v in six.iteritems(pie.slices):
                        rv += int(k) * v
                    return rv

                def find_pie_sum_by_id(balancer_pies, pie_name):
                    pie = find_pie_by_id(balancer_pies, pie_name)
                    if pie is not None:
                        return pie_to_sum(pie)
                    else:
                        return 0

                gencfg_groups_backends = find_pie_sum_by_id(n_pb.balancer_pies, 'gencfg_groups_backends_count')
                nanny_snapshots_backends = find_pie_sum_by_id(n_pb.balancer_pies, 'nanny_snapshots_backends_count')
                manual_backends = find_pie_sum_by_id(n_pb.balancer_pies, 'manual_backends_count')
                yp_endpoint_sets_backends = find_pie_sum_by_id(n_pb.balancer_pies, 'yp_endpoint_sets_backends_count')
                yp_endpoint_sets_sd_backends = find_pie_sum_by_id(n_pb.balancer_pies, 'sd_backends_count')

                total_backends = (gencfg_groups_backends + nanny_snapshots_backends +
                                      manual_backends + yp_endpoint_sets_backends_count +
                                  yp_endpoint_sets_sd_backends)
                non_sd_l7_backends = (gencfg_groups_backends + nanny_snapshots_backends +
                                      manual_backends + yp_endpoint_sets_backends_count)
                sd_l7_backends = yp_endpoint_sets_sd_backends

                total_backends_count += total_backends
                gencfg_groups_backends_count += gencfg_groups_backends
                nanny_snapshots_backends_count += nanny_snapshots_backends
                manual_backends_count += manual_backends
                yp_endpoint_sets_backends_count += yp_endpoint_sets_backends
                yp_endpoint_sets_sd_backends_count += yp_endpoint_sets_sd_backends
                if non_sd_l7_backends == 0 and sd_l7_backends > 0:
                    sd_only_l7_namespace_count += 1
                if total_backends == (yp_endpoint_sets_backends + yp_endpoint_sets_sd_backends):
                    yp_es_only_l7_namespace_count += 1

                balancers_without_sd_backends_count = find_pie_by_id(n_pb.balancer_pies, 'sd_backends_count').slices.get(
                    '0', 0)
                is_sd_ns = n_pb.balancers_count > balancers_without_sd_backends_count
                if is_sd_ns:
                    sd_ns_count += 1

                tlem_balancers_count = find_pie_by_id(n_pb.balancer_pies, 'is_easy_mode_enabled').slices.get('True', 0)
                is_tlem_ns = n_pb.balancers_count == tlem_balancers_count
                balancer_compat_options_count = 0
                if is_tlem_ns:
                    tlem_ns_count += 1

                    compat_options_count_pie_pb = find_pie_by_id(n_pb.balancer_pies, 'compat_options_count')
                    if compat_options_count_pie_pb is not None:
                        for k, v in six.iteritems(compat_options_count_pie_pb.slices):
                            balancer_compat_options_count += v
                        if compat_options_count_pie_pb.slices.get('https.enable_sslv3', 0) > 0:
                            tlem_ns_w_sslv3_count += 1
                else:
                    is_not_tlem_ns_count += 1

                uem_upstreams_count = find_pie_by_id(n_pb.upstream_pies, 'is_easy_mode_enabled').slices.get('True', 0)
                is_uem_ns = n_pb.upstreams_count == uem_upstreams_count

                if is_uem_ns:
                    uem_ns_count += 1
                else:
                    is_not_uem_ns_count += 1

                is_theem_ns = is_tlem_ns and is_uem_ns
                total_upstreams_count += n_pb.upstreams_count
                easy_upstream_count += uem_upstreams_count
                ns_w_at_least_one_easy_upstream_count += bool(uem_upstreams_count)
                ns_w_at_least_one_easy_domain_count += bool(n_pb.domains_count)

                v_ = 0
                compat_options_count_pie_pb = find_pie_by_id(n_pb.balancer_pies, 'compat_options_count')
                if compat_options_count_pie_pb is not None:
                    for k, v in six.iteritems(compat_options_count_pie_pb.slices):
                        v_ += v
                top_level_compat_ns_count += v_

                compat_options_count_pie_pb = find_pie_by_id(n_pb.upstream_pies, 'compat_options_count')
                uses_compat_method = False

                w_ = 0
                if compat_options_count_pie_pb:
                    for k, w in six.iteritems(compat_options_count_pie_pb.slices):
                        if is_theem_ns:
                            if k in ('by_dc_scheme.balancer.method', 'flat_scheme.balancer.method') and w > 0:
                                uses_compat_method = True
                                w_ -= w
                        w_ += w
                upstream_compat_ns_count += w_

                compat_options_count = v_ + w_
                if is_theem_ns:
                    if compat_options_count == 0:
                        theem_wo_compat_ns_count += 1
                    theem_ns_compat_opts_count += compat_options_count

                    theem_ns_count += 1
                    curr_theem_ns_ids.add(n_id)  # + (' *' if uses_compat_method else ''))
                else:
                    is_not_theem_ns_count += 1

                if balancer_compat_options_count > 0:  # compat_options_count > 0:
                    with_compat_ns_count += 1

                qs_pie = find_pie_by_id(n_pb.balancer_pies, 'is_quick_start_mode_enabled')
                qs_balancers_count = qs_pie.slices.get('True', 0) if qs_pie else 0
                is_qs_ns = n_pb.balancers_count == qs_balancers_count
                if is_qs_ns:
                    qs_ns_count += 1

                if n_id == 's.yandex-team.ru' or n_id == 'yappy.yandex.ru':
                    continue

                arl_pie = find_pie_by_id(n_pb.balancer_pies, 'balancer2_modules_without_arl_gpb_count')
                if arl_pie:
                    for k, v in six.iteritems(arl_pie.slices):
                        balancer2_wo_arl_gpb_ns_count += int(k) * v
                arl_pie = find_pie_by_id(n_pb.upstream_pies, 'balancer2_modules_without_arl_gpb_count')
                if arl_pie:
                    for k, v in six.iteritems(arl_pie.slices):
                        balancer2_wo_arl_gpb_ns_count += int(k) * v

                balancer2_wo_arl_and_wm_gpb_ns_diff = 0
                arl_pie = find_pie_by_id(n_pb.balancer_pies, 'balancer2_modules_without_arl_and_watermark_gpb_count')
                if arl_pie:
                    for k, v in six.iteritems(arl_pie.slices):
                        balancer2_wo_arl_and_wm_gpb_ns_diff += int(k) * v
                arl_pie = find_pie_by_id(n_pb.upstream_pies, 'balancer2_modules_without_arl_and_watermark_gpb_count')
                if arl_pie:
                    for k, v in six.iteritems(arl_pie.slices):
                        balancer2_wo_arl_and_wm_gpb_ns_diff += int(k) * v

                balancer2_wo_arl_and_wm_gpb_ns_count += balancer2_wo_arl_and_wm_gpb_ns_diff
                arl_affected = balancer2_wo_arl_and_wm_gpb_ns_diff > 1
                if arl_affected:
                    arl_affected_ns_count += 1
                    # print('*', n_id, NS_UPSTREAMS_HREF_TEMPLATE.format(n_id))

                arl_pie = find_pie_by_id(n_pb.balancer_pies, 'balancer2_modules_total_gpb_count')
                if arl_pie:
                    for k, v in six.iteritems(arl_pie.slices):
                        balancer2_gpb_ns_count += int(k) * v
                arl_pie = find_pie_by_id(n_pb.upstream_pies, 'balancer2_modules_total_gpb_count')
                if arl_pie:
                    for k, v in six.iteritems(arl_pie.slices):
                        balancer2_gpb_ns_count += int(k) * v
            """
            print(dt, 'balancer2_wo_arl_gpb_ns_count', balancer2_wo_arl_gpb_ns_count)
            print(dt, 'balancer2_wo_arl_and_wm_gpb_ns_count', balancer2_wo_arl_and_wm_gpb_ns_count)
            print(dt, 'balancer2_gpb_ns_count', balancer2_gpb_ns_count)
            print(dt, 'arl_affected_ns_count', arl_affected_ns_count)
            print(dt, 'theem_ns_count', theem_ns_count)
            print(dt, 'theem_wo_compat_ns_count', theem_wo_compat_ns_count)
            print(dt, 'theem_ns_compat_opts_count', theem_ns_compat_opts_count)
            print(dt, 'top_level_compat_ns_count', top_level_compat_ns_count)
            print(dt, 'upstream_compat_ns_count', upstream_compat_ns_count)
            print(dt, 'domain_ns_count', domain_ns_count)
            print(dt, 'total_domains_count', total_domains_count)
            print('---')
            print(dt, 'total_ns_count', total_l7_ns_count)
            print(dt, 'total_l3_count', total_l3_count)
            print(dt, 'ns_with_l3_only_count', ns_with_l3_only_count)
            """

            yp_lite_ns_counts.append(yp_lite_ns_count)
            gencfg_ns_counts.append(gencfg_ns_count)
            unistat_ns_counts.append(unistat_ns_count)
            total_l7_ns_counts.append(total_l7_ns_count)
            total_l7_ns_without_taxi_counts.append(total_l7_ns_without_taxi_count)
            total_ns_counts.append(total_l7_ns_count + ns_with_l3_only_count)
            total_l3_counts.append(total_l3_count)
            sd_ns_counts.append(sd_ns_count)
            tlem_ns_counts.append(tlem_ns_count)
            is_not_tlem_ns_counts.append(is_not_tlem_ns_count)
            uem_ns_counts.append(uem_ns_count)
            is_not_uem_ns_counts.append(is_not_uem_ns_count)
            theem_ns_counts.append(theem_ns_count)
            is_not_theem_ns_counts.append(is_not_theem_ns_count)
            theem_wo_compat_ns_counts.append(theem_wo_compat_ns_count)
            qs_ns_counts.append(qs_ns_count)
            cert_ns_counts.append(cert_ns_count)
            domain_ns_counts.append(domain_ns_count)
            tlem_ns_w_sslv3_counts.append(tlem_ns_w_sslv3_count)
            theem_ns_compat_opts_counts.append(theem_ns_compat_opts_count / 10)
            with_compat_ns_counts.append(with_compat_ns_count)
            top_level_compat_ns_counts.append(top_level_compat_ns_count / 10)
            upstream_compat_ns_counts.append(upstream_compat_ns_count / 10)
            balancer2_wo_arl_gpb_counts.append(balancer2_wo_arl_gpb_ns_count)
            balancer2_wo_arl_and_wm_gpb_counts.append(balancer2_wo_arl_and_wm_gpb_ns_count)
            balancer2_gpb_counts.append(balancer2_gpb_ns_count)
            arl_affected_ns_counts.append(arl_affected_ns_count)
            easy_upstream_counts.append(easy_upstream_count)
            total_upstreams_counts.append(total_upstreams_count)
            ns_w_at_least_one_easy_upstream_counts.append(ns_w_at_least_one_easy_upstream_count)
            ns_w_at_least_one_easy_domain_counts.append(ns_w_at_least_one_easy_domain_count)
            alerting_enabled_pies = find_pie_by_id(data_pb.namespace_pies, 'is_alerting_enabled')
            alerting_enabled_counts.append(alerting_enabled_pies.slices.get('True', 0) if alerting_enabled_pies else 0)
            total_instances_counts.append(total_instances_count)
            sd_only_l7_namespace_counts.append(sd_only_l7_namespace_count)
            yp_es_only_l7_namespace_counts.append(yp_es_only_l7_namespace_count)
            total_backends_counts.append(total_backends_count)
            gencfg_groups_backends_counts.append(gencfg_groups_backends_count)
            nanny_snapshots_backends_counts.append(nanny_snapshots_backends_count)
            manual_backends_counts.append(manual_backends_count)
            yp_endpoint_sets_backends_counts.append(yp_endpoint_sets_backends_count)
            yp_endpoint_sets_sd_backends_counts.append(yp_endpoint_sets_sd_backends_count)

    daily_dts = daily_dts.drop(exclusion_dates)
    if mode == 'xxx':
        calc_stats(exclude_taxi)
        def window(iterable, size):
            shiftedStarts = [itertools.islice(iterable, s, None) for s in range(size)]
            return zip(*shiftedStarts)

        def line_format(label):
            month = label.month_name()[:3]
            if month == 'Jan':
                month += '\n' + str(label.year)
            return month

        with open('bydate.pkl', 'r') as f:
            by_date = pickle.load(f)

        year_ago = (datetime.utcnow() - timedelta(days=660)).date()
        start = datetime.utcnow().date()
        while start >= year_ago:
            if start not in by_date:
                by_date[start] = set()
            start -= timedelta(days=1)

        # by_date = {k: v for k, v in by_date.items() if k >= year_ago}

        daily_items = sorted((date, value) for date, value in by_date.items())
        weekly_items = []
        for items in window(sorted(by_date.items()), 7):
            value = set(itertools.chain.from_iterable([logins for _, logins in items]))
            date = items[-1][0]
            weekly_items.append((date, value))
        monthly_items = []
        for items in window(sorted(by_date.items()), 28):
            value = set(itertools.chain.from_iterable([logins for _, logins in items]))
            date = items[-1][0]
            monthly_items.append((date, value))
        new_users_monthly_items = []

        seen_users = set()
        # for items in list(grouplen(sorted(by_date.items()), 28)):
        for items in window(sorted(by_date.items()), 28):
            date = items[0][0]
            if date.day == 1:
                users = set(itertools.chain.from_iterable([logins for _, logins in items]))
                print(date)
                print(users)
                print(users - seen_users)
                print('---')
                if date >= year_ago:
                    new_users_monthly_items.append((date, users - seen_users))
                seen_users.update(users)

        daily_dts = pd.DatetimeIndex([item[0] for item in daily_items])
        weekly_dts = pd.DatetimeIndex([item[0] for item in weekly_items])
        monthly_dts = pd.DatetimeIndex([item[0] for item in monthly_items])
        new_users_monthly_dts = pd.DatetimeIndex([item[0] for item in new_users_monthly_items])

        lines_columns = collections.OrderedDict([
            # ('daily unique editors', [len(item[1]) for item in daily_items]),
            # ('weekly unique editors', [len(item[1]) for item in weekly_items]),
            ('monthly never seen before editors', [len(item[1]) for item in new_users_monthly_items]),
            # ('monthly unique editors', [len(item[1]) for item in monthly_items]),
        ])
        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x),
                          # index=weekly_dts,
                          index=new_users_monthly_dts,
                          # index=monthly_dts,
                          columns=columns)
        ax = df.plot(title='awacs monthly never seen before editors',
                     figsize=(16, 10),
                     yticks=np.arange(start=0, stop=75, step=5),
                     grid=True,
                     kind='bar')
        # marker='.', linestyle='none')
        ax.set_xticklabels(map(lambda x: line_format(x), df.index))
        ax.set_xlabel('Date')
        ax.set_ylabel('Users count')
        ax.legend(loc='upper left')
        ax.tick_params(top=False,
                       left=True,
                       right=True,
                       bottom=True)
        plt.show()
        plt.grid(True)
        plt.savefig('./monthly-never-seen-before-editors.png')
    elif mode == 'sdftw':
        calc_stats(exclude_taxi)
        lines_columns = collections.OrderedDict([
            ('Total L7 namespaces', total_l7_ns_counts),
            ('YP endpoint sets-only L7 namespaces', yp_es_only_l7_namespace_counts),
            ('SD-only L7 namespaces', sd_only_l7_namespace_counts),
        ])
        for x, y in zip(sd_only_l7_namespace_counts, yp_es_only_l7_namespace_counts):
            print(float(x) / y * 100, )
        print()
        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='awacs SD-only L7-namespaces count',
                     figsize=(16, 10),
                     yticks=np.arange(start=0,
                                      stop=max(sd_only_l7_namespace_counts + total_l7_ns_counts) + 100,
                                      step=100),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Namespaces count')
        ax.legend(loc='upper left')
        ax.tick_params(top=False,
                       left=True,
                       right=True,
                       bottom=True)
        plt.show()
        plt.savefig('./sd-only-l7-namespaces-count.png')
    elif mode == 'backends_count':
        calc_stats(exclude_taxi)

        lines_columns = collections.OrderedDict([
            # ('Total backends', total_backends_counts),
            ("Gencfg groups backends", gencfg_groups_backends_counts),
            ("Nanny backends", nanny_snapshots_backends_counts),
            ("Manual backends", manual_backends_counts),
            ('YP endpoint sets backends', yp_endpoint_sets_backends_counts),
            ('YP endpoint sets SD backends', yp_endpoint_sets_sd_backends_counts),
        ])

        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)

        stop = max(gencfg_groups_backends_counts +
                   nanny_snapshots_backends_counts +
                   manual_backends_counts +
                   yp_endpoint_sets_backends_counts +
                   yp_endpoint_sets_sd_backends_counts)
        step = 1000

        ax = df.plot(title='awacs backends count',
                     figsize=(16, 10),
                     yticks=np.arange(start=0,
                                      stop=stop + 4000,
                                      step=step),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Backends count')
        ax.legend(loc='upper left')
        ax.tick_params(top=False,
                       left=True,
                       right=True,
                       bottom=True)

        plt.ticklabel_format(style='plain', axis='y')
        plt.show()
        plt.savefig('./sd-backends-count.png')
    elif mode == 'instances_count':
        calc_stats(exclude_taxi)

        lines_columns = collections.OrderedDict([
            ('instances_count', total_instances_counts),
        ])
        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='awacs L7 balancer instances count',
                     figsize=(16, 10),
                     yticks=np.arange(start=0, stop=max(total_instances_counts) + 1000, step=1000),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Instances count')
        ax.legend(loc='upper left')
        ax.tick_params(top=False,
                       left=True,
                       right=True,
                       bottom=True)
        plt.show()
        plt.savefig('./instances-count.png')
    elif mode == 'l7_upstream_macro':
        calc_stats(exclude_taxi)

        pkl_path = './l7_upstream_macro.pkl'
        if True:
            lines_columns = collections.OrderedDict([
                ('Total upstreams', total_upstreams_counts),
                ('Easy upstreams', easy_upstream_counts),
                ('Total namespaces', total_l7_ns_counts),
                ('Namespaces w/ at least one easy upstream', ns_w_at_least_one_easy_upstream_counts),
                ('Namespaces w/ at least one domain', ns_w_at_least_one_easy_domain_counts),
            ])

            with open(pkl_path, 'wb') as f:
                pickle.dump(lines_columns, f)
        else:
            with open(pkl_path, 'rb') as f:
                lines_columns = pickle.load(f)

        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='l7_upstream_macro adoption',
                     figsize=(16, 10),
                     yticks=np.arange(start=0,
                                      stop=max(easy_upstream_counts +
                                               ns_w_at_least_one_easy_upstream_counts +
                                               total_l7_ns_counts +
                                               total_upstreams_counts) + 500,
                                      step=500),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Namespaces count')
        ax.legend(loc='upper left')
        plt.show()
        plt.savefig('./l7_upstream_macro-without-taxi.png' if exclude_taxi else './l7_upstream_macro-with-taxi.png')

    elif mode == 'short':
        calc_stats(exclude_taxi)

        lines_columns = collections.OrderedDict([
            ('Total', total_ns_counts),
            ('Total w/ L7', total_l7_ns_counts),
            ('YP.lite-powered', yp_lite_ns_counts),
            ('Gencfg-powered', gencfg_ns_counts),
            ('w/ certificates', cert_ns_counts),
            ('w/ domains', domain_ns_counts),
            #('Alerting enabled', alerting_enabled_counts),
            ('Full easy-mode', theem_ns_counts),
        ])
        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='awacs feature adoption statistics',
                     figsize=(16, 10),
                     yticks=np.arange(start=0, stop=max(total_l7_ns_counts + total_l3_counts) + 200, step=100),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Namespaces count')
        ax.legend(loc='upper left')
        ax.tick_params(top=False,
                       left=True,
                       right=True,
                       bottom=True)
        plt.show()
        plt.savefig('./awacs-short-statistics.png')
    elif mode == 'alerting':
        calc_stats(exclude_taxi)

        lines_columns = collections.OrderedDict([
            ('Total', total_ns_counts),
            ('Total L7', total_l7_ns_counts),
            ('Alerting enabled', alerting_enabled_counts),
        ])
        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='awacs alerting adoption statistics',
                     figsize=(16, 10),
                     yticks=np.arange(start=0, stop=max(total_ns_counts), step=50),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Namespaces count')
        ax.legend(loc='upper left')
        plt.show()
        plt.savefig('./awacs-alerting-adoption-statistics.png')
    elif mode == 'easymodisation':
        calc_stats(exclude_taxi)

        lines_columns = collections.OrderedDict([
            ('Absolute easy mode', theem_ns_counts),
            ('Top-level easy mode', tlem_ns_counts),
            ('Top-level non-easy mode', is_not_tlem_ns_counts),
            ('Upstream easy mode', uem_ns_counts),
            ('Upstream non-easy mode', is_not_uem_ns_counts),
        ])

        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='awacs easymodisation',
                     figsize=(18, 12),
                     yticks=np.arange(start=0, stop=max(total_l7_ns_counts), step=100),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Namespaces count')
        ax.legend(loc='upper left')
        plt.show()
        plt.savefig('./awacs-easymodisation-without-taxi.png' if exclude_taxi else './awacs-easymodisation-with-taxi.png')
    elif mode == 'easymodisation-2':
        calc_stats(exclude_taxi)

        lines_columns = collections.OrderedDict([
            ('Easy mode namespaces', theem_ns_counts),
            ('Non easy mode namespace (at least one balancer or upstream is not in easy mode)', is_not_theem_ns_counts),
        ])

        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='awacs easy mode statistics (taxi and maps are not included)',
                     figsize=(18, 12),
                     yticks=np.arange(start=0, stop=max(max(theem_ns_counts), max(is_not_theem_ns_counts)) + 100,
                                      step=100),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Namespaces count')
        ax.legend(loc='upper left')
        plt.show()
        plt.savefig('./awacs-easymodisation-2.png')
    elif mode == 'common':
        calc_stats(exclude_taxi)

        lines_columns = collections.OrderedDict([
            ('Total', total_l7_ns_counts),
            ('YP.lite-powered', yp_lite_ns_counts),
            ('Gencfg-powered', gencfg_ns_counts),
            ('w/ unistat enabled', unistat_ns_counts),
            # ('w/ at least one SD backend',  sd_ns_counts),
            ('w/ certificates', cert_ns_counts),
            ('w/ domains', domain_ns_counts),
            ('Top-level easy-mode', tlem_ns_counts),
            ('Full easy-mode', theem_ns_counts),
            ('Namespaces w/ at least one top-level compat option', with_compat_ns_counts),
            ('Top-level compat options (total options count divided by 10)', top_level_compat_ns_counts),
            ('Upstream compat options (total options count divided by 10)', upstream_compat_ns_counts),
            # ('Top-level easy-mode w/ SSLv3', tlem_ns_w_sslv3_counts),
            ('Alerting enabled', alerting_enabled_counts),
        ])

        x = np.array(lines_columns.values())
        columns = lines_columns.keys()
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='awacs feature adoption statistics',
                     figsize=(18, 12),
                     yticks=np.arange(start=0, stop=max(total_l7_ns_counts), step=100),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Namespaces count')
        ax.legend(loc='upper left')
        plt.show()
        plt.savefig('./awacs-feature-adoption-statistics.png')
    elif mode == 'theem':
        calc_stats(exclude_taxi)

        x = np.array([
            theem_ns_counts,
            theem_wo_compat_ns_counts,
            theem_ns_compat_opts_counts,
        ])
        columns = [
            'Namespaces in the easy mode',
            'Namespaces in the easy mode w/o compat',
            'Compat options (total options count divided by 10)',
        ]
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='awacs easy mode statistics',
                     figsize=(8, 16),
                     yticks=np.arange(start=0, stop=900, step=25),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Namespaces count')
        plt.show()
        plt.savefig('./awacs-theem-statistics.png')
        print('Current the EM ns ids')
        print('\n'.join(sorted(curr_theem_ns_ids)))
    elif mode == 'arl':
        calc_stats(exclude_taxi)

        x = np.array([
            balancer2_wo_arl_gpb_counts,
            balancer2_wo_arl_and_wm_gpb_counts,
            balancer2_gpb_counts,
            arl_affected_ns_counts,
        ])
        columns = [
            'Terminal balancer2 without ARL count',
            'Terminal balancer2 without ARL and wm policy count',
            'Terminal balancer2 count',
            'Number of affected namespaces',
        ]
        df = pd.DataFrame(data=np.transpose(x), index=daily_dts, columns=columns)
        ax = df.plot(title='arl',
                     figsize=(16, 15),
                     yticks=np.arange(start=0, stop=4000, step=1000),
                     grid=True)
        ax.set_xlabel('Date')
        ax.set_ylabel('Namespaces count')
        plt.show()
        plt.savefig('./balancer2-counts.png')
    else:
        raise AssertionError('unknown mode {}'.format(mode))

    plt.clf()
    plt.close()

    # click.echo('gencfg namespaces created in the last {} days: {}'.format(
    #    days, sorted(curr_gencfg_namespace_ids - old_gencfg_namespace_ids)))
    # click.echo('===')
    # xs = sorted(id for id in last_ns_ids - first_ns_ids if 'taxi' not in id)
    # click.echo('\n'.join(xs))
    # click.echo('count: {}'.format(len(xs)))


cli.add_command(plot, name='plot')

if __name__ == '__main__':
    cli()
