# coding: utf-8
import json
import os.path


class Op(object):
    def __init__(self, op_id, data):
        self.id = op_id
        self.data = data

    @classmethod
    def get_op_dir_path(cls, op_id):
        return os.path.join('./opcache/', op_id)

    @classmethod
    def load(cls, op_id):
        op_dir_path = cls.get_op_dir_path(op_id)
        op_data_path = os.path.join(op_dir_path, 'data.json')
        if os.path.exists(op_data_path):
            with open(op_data_path, 'r') as f:
                return cls(op_id, json.load(f))
        else:
            op = cls(op_id, {})
            op.save()
            return op

    def save(self):
        op_dir_path = self.get_op_dir_path(self.id)
        if not os.path.exists(op_dir_path):
            os.makedirs(op_dir_path)
        op_data_path = os.path.join(op_dir_path, 'data.json')
        with open(op_data_path, 'w') as f:
            json.dump(self.data, f, indent=2, sort_keys=True)


class Playbook(object):
    DEFAULT_AWACS_PLAYBOOKS_PATH = './playbooks/'

    def __init__(self, playbook_id, playbook, index):
        self.playbook_id = playbook_id
        self.playbook = playbook
        self.index = index

    @classmethod
    def load(cls, playbook_id):
        playbooks_path = os.getenv('AWACS_PLAYBOOKS_PATH', default=cls.DEFAULT_AWACS_PLAYBOOKS_PATH)

        playbook_path = os.path.join(playbooks_path, playbook_id + '-playbook.json')
        with open(playbook_path, 'r') as f:
            playbook_data = json.load(f)

        index_path = os.path.join(playbooks_path, playbook_id + '-index.json')
        with open(index_path, 'r') as f:
            index_data = json.load(f)

        return cls(playbook_id, playbook_data, index_data)

    def list_stage_namespace_ids(self, stage):
        if stage not in self.playbook:
            raise ValueError('Stage "{}" is not found in playbook {}'.format(stage, self.playbook_id))
        return sorted(self.playbook[stage])

    def list_stage_balancer_full_ids(self, stage, location):
        if stage not in self.playbook:
            raise ValueError('Stage "{}" is not found in playbook {}'.format(stage, self.playbook_id))
        namespace_ids = self.playbook[stage]
        full_balancer_ids = []
        for namespace_id in sorted(namespace_ids):
            namespace_index_data = sorted(self.index.get(namespace_id, {}).items())
            for location_, balancers in namespace_index_data:
                for (balancer_id, _) in balancers:
                    if location.lower() == location_.lower():
                        full_balancer_ids.append((namespace_id, balancer_id))
        return full_balancer_ids


class NamespaceSelector(object):
    def __init__(self, expr, resolver):
        """
        :type expr: six.text_type
        :type resolver:  callable
        """
        self.expr = expr
        self.resolver = resolver

    @classmethod
    def parse(cls, expr):
        """
        :type expr: six.text_type
        :rtype: Selector
        """
        head, tail = expr.split('@', 1)
        if head == 'playbook':
            stage = tail

            def resolve(app):
                """
                :type app: App
                """
                return set(app.playbook.list_stage_namespace_ids(stage=stage))
        elif head == 'ns':
            namespace_ids = tail.split(',')

            def resolve(app):
                """
                :type app: App
                """
                return set(namespace_ids)
        elif head == 'txt':
            path = tail
            with open(path, 'r') as f:
                namespace_ids = f.read().splitlines()

            def resolve(app):
                """
                :type app: App
                """
                return set(namespace_ids)
        else:
            raise ValueError()
        return cls(expr, resolve)


class Selector(object):
    def __init__(self, expr, resolver):
        """
        :type expr: six.text_type
        :type resolver:  callable
        """
        self.expr = expr
        self.resolver = resolver

    @classmethod
    def parse(cls, expr):
        """
        :type expr: six.text_type
        :rtype: Selector
        """
        head, tail = expr.split('@', 1)
        if head == 'playbook':
            stage, location = tail.split(':', 1)

            def resolve(app):
                """
                :type app: App
                """
                return set(app.playbook.list_stage_balancer_full_ids(stage=stage, location=location))
        elif head == 'ns':
            namespace_ids = tail.split(',')

            def resolve(app):
                """
                :type app: App
                """
                return {(balancer_pb.meta.namespace_id, balancer_pb.meta.id)
                        for balancer_pb in app.awacs_client.iter_all_balancers(namespace_id_in=namespace_ids)}
        elif head == 'b':
            flat_balancer_ids = tail.split(',')
            full_balancer_ids = []
            for flat_id in flat_balancer_ids:
                full_balancer_ids.append(tuple(flat_id.split(':', 1)))

            def resolve(app):
                """
                :type app: App
                """
                return set(full_balancer_ids)
        else:
            raise ValueError()
        return cls(expr, resolve)


class App(object):
    def __init__(self, awacs_client, nanny_client=None, l3mgr_client=None, op=None, playbook=None):
        """
        :type awacs_client: infra.awacs.tools.awacstoolslib.awacsclient.AwacsClient
        :type nanny_client: Optional[infra.awacs.tools.awacstoolslib.nannyclient.NannyClient]
        :type l3mgr_client: awacs.lib.l3mgrclient.L3MgrClient
        :type op: Optional[Op]
        :type playbook: Optional[Playbook]
        """
        self.awacs_client = awacs_client
        self._l3mgr_client = l3mgr_client
        self._nanny_client = nanny_client
        self._op = op
        self._playbook = playbook

    def is_nanny_client_configured(self):
        return self._nanny_client is not None

    @property
    def nanny_client(self):
        """
        :rtype: infra.awacs.tools.awacstoolslib.nannyclient.NannyClient
        """
        if not self.is_nanny_client_configured():
            raise RuntimeError('Nanny client is not configured')
        return self._nanny_client

    def is_l3mgr_client_configured(self):
        return self._l3mgr_client is not None

    @property
    def l3mgr_client(self):
        """
        :rtype: awacs.lib.l3mgrclient.L3MgrClient
        """
        if not self.is_l3mgr_client_configured():
            raise RuntimeError('l3mgr client is not configured')
        return self._l3mgr_client

    def is_op_configured(self):
        return self._op is not None

    @property
    def op(self):
        """
        :rtype: Op
        """
        if not self.is_op_configured():
            raise RuntimeError('Op is not configured')
        return self._op

    def is_playbook_configured(self):
        return self._playbook is not None

    @property
    def playbook(self):
        """
        :rtype: Playbook
        """
        if not self.is_playbook_configured():
            raise RuntimeError('Playbook is not configured')
        return self._playbook
