import re
import signal

import click
import six
import yaml
from awacs import yamlparser
from awacs.lib import l3mgrclient
from awacs.lib.yamlparser.wrappers_util import dump_tlem_pb, pb_to_dict, AwacsYamlDumper
from awacs.model.components import SemanticVersion
from awacs.yamlparser import dump
from infra.awacs.proto import modules_pb2

from .app import App, Op, Playbook
from .awacsclient import AwacsClient
from .const import DEFAULT_NANNY_URL, DEFAULT_AWACS_RPC_URL, DEFAULT_YP_LITE_UI_URL, DEFAULT_L3MGR_API_URL
from .nannyclient import NannyClient
from six.moves import input


def clone_pb(pb):
    cloned_pb = type(pb)()
    cloned_pb.CopyFrom(pb)
    return cloned_pb


def clone_pb_dict(d):
    return {k: clone_pb(pb) for k, pb in six.iteritems(d)}


def is_l7_macro_version_gte(balancer_spec_pb, to_version):
    assert balancer_spec_pb.yandex_balancer.config.HasField('l7_macro')
    assert yamlparser.parse(
        modules_pb2.Holder,
        dump_tlem_pb(balancer_spec_pb.yandex_balancer.config)) == balancer_spec_pb.yandex_balancer.config
    l7_macro_pb = clone_pb(balancer_spec_pb.yandex_balancer.config.l7_macro)

    from_version = l7_macro_pb.version

    return SemanticVersion.parse(from_version).to_key() >= SemanticVersion.parse(to_version).to_key()


def is_instance_macro_version_gte(balancer_spec_pb, to_version):
    assert balancer_spec_pb.yandex_balancer.config.HasField('instance_macro')
    assert yamlparser.parse(
        modules_pb2.Holder,
        dump(balancer_spec_pb.yandex_balancer.config)) == balancer_spec_pb.yandex_balancer.config
    instance_macro_pb = clone_pb(balancer_spec_pb.yandex_balancer.config.instance_macro)

    from_version = instance_macro_pb.version

    return SemanticVersion.parse(from_version).to_key() >= SemanticVersion.parse(to_version).to_key()


def update_l7_macro(balancer_spec_pb, f):
    assert balancer_spec_pb.yandex_balancer.config.HasField('l7_macro')
    assert '#' not in balancer_spec_pb.yandex_balancer.yaml
    assert yamlparser.parse(
        modules_pb2.Holder,
        dump_tlem_pb(balancer_spec_pb.yandex_balancer.config)) == balancer_spec_pb.yandex_balancer.config
    l7_macro_pb = clone_pb(balancer_spec_pb.yandex_balancer.config.l7_macro)

    updated, message = f(l7_macro_pb)
    if updated:
        balancer_spec_pb.yandex_balancer.config.Clear()
        balancer_spec_pb.yandex_balancer.config.l7_macro.CopyFrom(l7_macro_pb)
        balancer_spec_pb.yandex_balancer.yaml = dump_tlem_pb(balancer_spec_pb.yandex_balancer.config)

        assert yamlparser.parse(modules_pb2.Holder,
                                balancer_spec_pb.yandex_balancer.yaml) == balancer_spec_pb.yandex_balancer.config
    return updated, message


def update_l7_macro_version(balancer_spec_pb, to_version):
    assert balancer_spec_pb.yandex_balancer.config.HasField('l7_macro')
    assert '#' not in balancer_spec_pb.yandex_balancer.yaml
    assert yamlparser.parse(
        modules_pb2.Holder,
        dump_tlem_pb(balancer_spec_pb.yandex_balancer.config)) == balancer_spec_pb.yandex_balancer.config
    l7_macro_pb = clone_pb(balancer_spec_pb.yandex_balancer.config.l7_macro)

    from_version = l7_macro_pb.version
    assert from_version

    if SemanticVersion.parse(from_version).to_key() < SemanticVersion.parse(to_version).to_key():
        l7_macro_pb.version = to_version
        balancer_spec_pb.yandex_balancer.config.Clear()
        balancer_spec_pb.yandex_balancer.config.l7_macro.CopyFrom(l7_macro_pb)
        balancer_spec_pb.yandex_balancer.yaml = dump_tlem_pb(balancer_spec_pb.yandex_balancer.config)

        assert yamlparser.parse(modules_pb2.Holder,
                                balancer_spec_pb.yandex_balancer.yaml) == balancer_spec_pb.yandex_balancer.config
        return True, 'updated l7_macro.version from {} to {}'.format(from_version, to_version)
    else:
        return False, 'l7_macro.version is already {}, which is {} or newer'.format(from_version, to_version)


def update_l7_macro_version_preserving_comments(balancer_spec_pb, to_version):
    assert balancer_spec_pb.yandex_balancer.config.HasField('l7_macro')
    assert yamlparser.parse(
        modules_pb2.Holder,
        dump_tlem_pb(balancer_spec_pb.yandex_balancer.config)) == balancer_spec_pb.yandex_balancer.config
    l7_macro_pb = clone_pb(balancer_spec_pb.yandex_balancer.config.l7_macro)

    from_version = l7_macro_pb.version
    assert from_version

    if SemanticVersion.parse(from_version).to_key() < SemanticVersion.parse(to_version).to_key():
        yml = balancer_spec_pb.yandex_balancer.yaml
        lines = yml.splitlines()
        s = 0
        if lines[s] == '---':
            s += 1
        assert u'l7_macro:' in lines[s], u'Unexpected lines[{}]: {!r}'.format(s, lines[:s + 1])
        m = re.search('^\s+', lines[s + 1])
        assert m
        indent = m.group()

        i = s + 1
        m = None
        while i < len(lines):
            m = re.match(u'^\s+version: (.+)$', lines[i])
            if m:
                lines[i] = indent + u'version: ' + to_version
                break
            i += 1

        if m is None:
            lines.insert(s + 1, indent + u'version: ' + to_version)
        updated_yml = u'\n'.join(lines)

        # make sure we've only changed the version field
        curr_yml_pb = yamlparser.parse(modules_pb2.Holder, yml)
        updated_yml_pb = yamlparser.parse(modules_pb2.Holder, updated_yml)
        curr_yml_pb.l7_macro.version = updated_yml_pb.l7_macro.version = u'XXX'
        assert curr_yml_pb == updated_yml_pb

        balancer_spec_pb.yandex_balancer.yaml = updated_yml
        return True, u'updated l7_macro.version from {} to {}'.format(from_version, to_version)
    else:
        return False, u'l7_macro.version is already {}, which is {} or newer'.format(from_version, to_version)


def update_instance_macro_version(balancer_spec_pb, to_version):
    assert balancer_spec_pb.yandex_balancer.config.HasField('instance_macro')
    assert yamlparser.parse(
        modules_pb2.Holder,
        yaml.dump(pb_to_dict(balancer_spec_pb.yandex_balancer.config),
                  Dumper=AwacsYamlDumper)) == balancer_spec_pb.yandex_balancer.config
    instance_macro_pb = clone_pb(balancer_spec_pb.yandex_balancer.config.instance_macro)

    from_version = instance_macro_pb.version or u'0.0.1'
    if SemanticVersion.parse(from_version).to_key() < SemanticVersion.parse(to_version).to_key():
        yml = balancer_spec_pb.yandex_balancer.yaml
        lines = yml.splitlines()
        s = 0
        if lines[s] == u'---':
            s += 1
        assert u'instance_macro:' in lines[s], u'Unexpected lines[{}]: {!r}'.format(s, lines[:s + 1])
        m = re.search('^\s+', lines[s + 1])
        assert m
        indent = m.group()

        m = re.match(u'^\s+version: (.+)$', lines[s + 1])
        if m:
            lines[s + 1] = indent + u'version: ' + to_version
        else:
            lines.insert(s + 1, indent + u'version: ' + to_version)
        updated_yml = u'\n'.join(lines)

        # make sure we've only changed the version field
        curr_yml_pb = yamlparser.parse(modules_pb2.Holder, yml)
        updated_yml_pb = yamlparser.parse(modules_pb2.Holder, updated_yml)
        curr_yml_pb.instance_macro.version = updated_yml_pb.instance_macro.version = u'XXX'
        assert curr_yml_pb == updated_yml_pb

        balancer_spec_pb.yandex_balancer.yaml = updated_yml
        return True, u'updated instance_macro.version from {} to {}'.format(from_version, to_version)
    else:
        return False, u'instance_macro.version is already {}, which is {} or newer'.format(from_version, to_version)


def update_instancectl_conf_version(balancer_spec_pb, to_version, to_pushclient_version):
    assert to_pushclient_version.endswith('pushclient')
    instancectl_conf_pb = balancer_spec_pb.components.instancectl_conf
    curr_version = instancectl_conf_pb.version

    if curr_version.endswith('pushclient'):
        new_version = to_pushclient_version
    else:
        new_version = to_version

    if SemanticVersion.parse(curr_version).to_key() < SemanticVersion.parse(new_version).to_key():
        instancectl_conf_pb.version = new_version
        return True, 'update instancectl.conf version from {} to {}'.format(curr_version, new_version)
    else:
        return False, 'instancectl.conf is already {}, which is {} or newer'.format(curr_version, new_version)


def update_awacslet_version(balancer_spec_pb, to_version, to_pushclient_version):
    assert to_pushclient_version.endswith('pushclient')
    awacslet_pb = balancer_spec_pb.components.awacslet
    curr_version = awacslet_pb.version

    if curr_version.endswith('pushclient'):
        new_version = to_pushclient_version
    else:
        new_version = to_version

    if SemanticVersion.parse(curr_version).to_key() < SemanticVersion.parse(new_version).to_key():
        awacslet_pb.version = new_version
        return True, 'update awacslet version from {} to {}'.format(curr_version, new_version)
    else:
        return False, 'awacslet is already {}, which is {} or newer'.format(curr_version, new_version)


def add_or_update_endpoint_root_certs_version(balancer_spec_pb, to_version):
    endpoint_root_certs_pb = balancer_spec_pb.components.endpoint_root_certs
    from_version = endpoint_root_certs_pb.version

    if endpoint_root_certs_pb.state == endpoint_root_certs_pb.SET:
        if SemanticVersion.parse(from_version).to_key() < SemanticVersion.parse(to_version).to_key():
            endpoint_root_certs_pb.version = to_version
            return True, 'update endpoint root certs version from {} to {}'.format(from_version, to_version)
        else:
            return False, 'endpoint root certs version is already {}, which is {} or newer'.format(from_version,
                                                                                                   to_version)
    elif endpoint_root_certs_pb.state == endpoint_root_certs_pb.UNKNOWN:
        endpoint_root_certs_pb.state = endpoint_root_certs_pb.SET
        endpoint_root_certs_pb.version = to_version
        return True, 'add endpoint root certs version {}'.format(to_version)
    else:
        return False, 'endpoint root certs is REMOVED'


def update_base_layer_version(balancer_spec_pb, to_version, expected_versions=None):
    base_layer_pb = balancer_spec_pb.components.base_layer
    curr_version = base_layer_pb.version

    if expected_versions is not None and curr_version not in expected_versions:
        return False, 'base layer is {}, not in {}'.format(curr_version, sorted(list(expected_versions)))

    assert base_layer_pb.state == base_layer_pb.SET
    if curr_version != to_version:
        base_layer_pb.version = to_version
        return True, 'update base layer version from {} to {}'.format(curr_version, to_version)

    return False, 'base layer is already {}'.format(to_version)


def update_instancectl_version(instancectl_pb, new_version, expected_versions=None):
    prev_version = instancectl_pb.version
    if expected_versions is not None and instancectl_pb.version not in expected_versions:
        return False, 'instancectl is {}, not in {}'.format(prev_version, expected_versions)

    assert instancectl_pb.state == instancectl_pb.SET
    if instancectl_pb.version != new_version:
        instancectl_pb.version = new_version
        return True, 'update instancectl version from {} to {}'.format(prev_version, new_version)

    return False, 'instancectl is already {}'.format(new_version)


def update_shawshank_version(shawshank_pb, new_version, expected_versions=None):
    prev_version = shawshank_pb.version
    if expected_versions is not None and shawshank_pb.version not in expected_versions:
        return False, 'shawshank is {}, not in {}'.format(prev_version, expected_versions)

    assert shawshank_pb.state == shawshank_pb.SET
    if shawshank_pb.version != new_version:
        shawshank_pb.version = new_version
        return True, 'update shawshank version from {} to {}'.format(prev_version, new_version)

    return False, 'shawshank is already {}'.format(new_version)


def update_juggler_checks_bundle_version(juggler_checks_bundle_pb, new_version, expected_versions=None):
    prev_version = juggler_checks_bundle_pb.version
    if expected_versions is not None and juggler_checks_bundle_pb.version not in expected_versions:
        return False, 'juggler checks bundle is {}, not in {}'.format(prev_version, expected_versions)

    assert juggler_checks_bundle_pb.state == juggler_checks_bundle_pb.SET
    if juggler_checks_bundle_pb.version != new_version:
        juggler_checks_bundle_pb.version = new_version
        return True, 'update juggler checks bundle version from {} to {}'.format(prev_version, new_version)

    return False, 'juggler checks bundle is already {}'.format(new_version)


def update_pginx_version(pginx_binary_pb, new_version, expected_versions=None):
    prev_version = pginx_binary_pb.version
    if expected_versions is not None and pginx_binary_pb.version not in expected_versions:
        return False, 'pginx is {}, not in {}'.format(prev_version, expected_versions)

    assert pginx_binary_pb.state == pginx_binary_pb.SET
    if pginx_binary_pb.version != new_version:
        pginx_binary_pb.version = new_version
        return True, 'update pginx version from {} to {}'.format(prev_version, new_version)

    return False, 'pginx is already {}'.format(new_version)


def create_awacs_namespace_href(namespace_id, nanny_url=DEFAULT_NANNY_URL):
    return nanny_url + 'ui/#/awacs/namespaces/list/{0}/show/'.format(namespace_id)


def create_awacs_balancer_href(namespace_id, balancer_id, nanny_url=DEFAULT_NANNY_URL):
    return nanny_url + 'ui/#/awacs/namespaces/list/{0}/balancers/list/{1}/show/'.format(
        namespace_id, balancer_id)


def create_awacs_l3_balancer_href(namespace_id, l3_balancer_id, nanny_url=DEFAULT_NANNY_URL):
    return nanny_url + 'ui/#/awacs/namespaces/list/{0}/l3-balancers/list/{1}/show/'.format(
        namespace_id, l3_balancer_id)


def create_awacs_upstream_href(namespace_id, upstream_id, nanny_url=DEFAULT_NANNY_URL):
    return nanny_url + 'ui/#/awacs/namespaces/list/{0}/upstreams/list/{1}/show/'.format(
        namespace_id, upstream_id)


def create_awacs_backend_href(namespace_id, backend_id, nanny_url=DEFAULT_NANNY_URL):
    return nanny_url + 'ui/#/awacs/namespaces/list/{0}/backends/list/{1}/show/'.format(
        namespace_id, backend_id)


def create_awacs_backends_list_href(namespace_id, nanny_url=DEFAULT_NANNY_URL):
    return nanny_url + 'ui/#/awacs/namespaces/list/{0}/backends/list/'.format(namespace_id)


def raw_input_w_timeout(timeout=30):
    class AlarmException(Exception):
        pass

    def alarm_handler(signum, frame):
        raise AlarmException

    signal.signal(signal.SIGALRM, alarm_handler)
    signal.alarm(timeout)
    try:
        text = input()
        signal.alarm(0)
        return text
    except AlarmException:
        return None
    finally:
        signal.signal(signal.SIGALRM, signal.SIG_IGN)


def wait_for_confirmation(message, confirm_automatically_after=-1):
    prompt = '{}, continue? [y/n/abort] '.format(message)
    if confirm_automatically_after > 0:
        prompt += '(would confirm automatically after {} seconds)'.format(confirm_automatically_after)
    click.echo(prompt, nl=False)
    while 1:
        if confirm_automatically_after > 0:
            inp = raw_input_w_timeout(timeout=confirm_automatically_after)
            if inp is None:
                click.echo('\nConfirmed automatically...')
                return True
        else:
            inp = input()
        if inp not in ('y', 'n', 'abort'):
            continue
        if inp == 'y':
            click.echo()
            return True
        elif inp == 'n':
            click.echo()
            return False
        else:
            raise click.Abort()


@click.group()
@click.option('--awacs-token', envvar='AWACS_TOKEN', required=True)
@click.option('--awacs-rpc-url', envvar='AWACS_RPC_URL', default=DEFAULT_AWACS_RPC_URL)
@click.option('--nanny-token', envvar='NANNY_TOKEN', default=None)
@click.option('--nanny-url', envvar='NANNY_URL', default=DEFAULT_NANNY_URL)
@click.option('--l3mgr-token', envvar='L3MGR_TOKEN', default=None)
@click.option('--l3mgr-url', envvar='L3MGR_URL', default=DEFAULT_L3MGR_API_URL)
@click.option('--yp-lite-ui-url', envvar='YP_LITE_UI_URL', default=DEFAULT_YP_LITE_UI_URL)
@click.option('--op-id', envvar='OP_ID')
@click.option('--playbook-id', envvar='PLAYBOOK_ID')
@click.pass_context
def cli(ctx, awacs_token, awacs_rpc_url, nanny_token, nanny_url, l3mgr_token, l3mgr_url, yp_lite_ui_url,
        op_id=None, playbook_id=None):
    awacs_client = AwacsClient(awacs_rpc_url=awacs_rpc_url, awacs_token=awacs_token)
    nanny_client = None
    if nanny_token is not None:
        nanny_client = NannyClient(nanny_url=nanny_url,
                                   yp_lite_ui_url=yp_lite_ui_url,
                                   oauth_token=nanny_token)
    l3mgr_client = None
    if l3mgr_token is not None:
        l3mgr_client = l3mgrclient.L3MgrClient(url=l3mgr_url,
                                               token=l3mgr_token)
        from awacs.model.l3_balancer import l3mgr
        l3mgr.ServiceConfig._client = l3mgr_client
        l3mgr.Service._client = l3mgr_client
    op = None
    if op_id is not None:
        op = Op.load(op_id)
    playbook = None
    if playbook_id is not None:
        playbook = Playbook.load(playbook_id)
    ctx.obj = App(awacs_client=awacs_client,
                  nanny_client=nanny_client,
                  l3mgr_client=l3mgr_client,
                  op=op,
                  playbook=playbook)
