# coding: utf-8
import re
import sys
import time
from datetime import datetime, timedelta

import click
import requests
import six
import yaml
from awacs import yamlparser
from awacs.lib.yamlparser.wrappers_util import dump_tlem_pb
from infra.awacs.proto import model_pb2, modules_pb2

from infra.awacs.tools.awacstoolslib.util import (
    clone_pb, create_awacs_namespace_href, create_awacs_balancer_href, create_awacs_l3_balancer_href,
    cli
)
from six.moves import input


def find_option(cfg, option):
    if option not in cfg:
        return False
    idx1 = cfg.find(option)
    idx2 = cfg[idx1:].find('\n') + idx1
    return True if 'true' in cfg[idx1:idx2].lower() else False


def delete_option(cfg, option):
    if option not in cfg:
        return cfg
    idx1 = cfg[:cfg.find(option)].rfind('\n') + 1
    idx2 = cfg[idx1:].find('\n') + idx1
    return cfg[:idx1] + cfg[idx2 + 1:]


def get_location(balancer_pb):
    if balancer_pb.meta.location.HasField('gencfg_dc'):
        return balancer_pb.meta.location.gencfg_dc
    elif balancer_pb.meta.location.HasField('yp_cluster'):
        return balancer_pb.meta.location.yp_cluster
    return None


def bump_l7_macro_version_balancer(d, namespace_id, balancer_id, to_version):
    balancer_pb = d.get_balancer(namespace_id, balancer_id)
    meta_pb = balancer_pb.meta
    spec_pb = balancer_pb.spec
    if not spec_pb.yandex_balancer.config.HasField('l7_macro'):
        return False, 'is not l7_macro'
    if '#' in spec_pb.yandex_balancer.yaml:
        return False, 'yaml probably contains comments'
    l7_macro_pb = clone_pb(spec_pb.yandex_balancer.config.l7_macro)

    assert yamlparser.parse(modules_pb2.Holder,
                            dump_tlem_pb(spec_pb.yandex_balancer.config)) == spec_pb.yandex_balancer.config

    from_version = l7_macro_pb.version
    l7_macro_pb.version = to_version
    spec_pb.yandex_balancer.config.Clear()
    spec_pb.yandex_balancer.config.l7_macro.CopyFrom(l7_macro_pb)
    spec_pb.yandex_balancer.yaml = dump_tlem_pb(spec_pb.yandex_balancer.config)
    d.update_balancer(
        namespace_id, balancer_id, meta_pb.version, spec_pb,
        comment='SWATOPS-246: bump version to {}'.format(to_version))
    return True, 'bumped version from {} to {}'.format(from_version, to_version)


@cli.command('remove_invalid_inactive_upstreams_and_backends')
@click.pass_obj
@click.option('--namespace-id', required=True)
def remove_invalid_inactive_upstreams_and_backends(app, namespace_id):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    upstream_pbs = d.list_upstreams(namespace_id)
    for upstream_pb in upstream_pbs:
        if upstream_pb.spec.deleted:
            continue
        if len(upstream_pb.statuses) != 1:
            continue
        rev_pb = upstream_pb.statuses[0]
        assert rev_pb.id == upstream_pb.meta.version

        validated_statuses = set()
        for cond_pb in rev_pb.validated.values():
            validated_statuses.add(cond_pb.status)

        active_statuses = set()
        for cond_pb in rev_pb.active.values():
            active_statuses.add(cond_pb.status)

        if validated_statuses == {'False'} and active_statuses == {'False'}:
            d.remove_upstream(namespace_id, upstream_pb.meta.id, upstream_pb.meta.version)
            print('upstream', upstream_pb.meta.id, 'removed')

    backend_pbs = d.list_backends(namespace_id)
    for backend_pb in backend_pbs:
        if backend_pb.spec.deleted:
            continue
        if len(backend_pb.statuses) != 1:
            continue
        if len(backend_pb.l3_statuses) != 1:
            continue
        if backend_pb.l3_statuses[0].validated:
            continue
        rev_pb = backend_pb.statuses[0]
        assert rev_pb.id == backend_pb.meta.version

        validated_statuses = set()
        for cond_pb in rev_pb.validated.values():
            validated_statuses.add(cond_pb.status)

        active_statuses = set()
        for cond_pb in rev_pb.active.values():
            active_statuses.add(cond_pb.status)

        if validated_statuses == {'False'} and active_statuses == {'False'}:
            d.remove_backend(namespace_id, backend_pb.meta.id, backend_pb.meta.version)
            print('backend', backend_pb.meta.id, 'removed')


@cli.command('remove_unused_backends')
@click.pass_obj
@click.option('--namespace-id', required=True)
def remove_unused_backends(app, namespace_id):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client

    backend_pbs = d.list_backends(namespace_id)
    for backend_pb in backend_pbs:
        if backend_pb.spec.deleted:
            continue
        if 'EDUCATION' not in backend_pb.meta.id:
            continue
        if len(backend_pb.statuses) != 1:
            continue
        if len(backend_pb.l3_statuses) != 1:
            continue
        if len(backend_pb.dns_record_statuses) != 1:
            continue
        if backend_pb.statuses[0].validated:
            continue
        if backend_pb.l3_statuses[0].validated:
            continue
        if backend_pb.dns_record_statuses[0].validated:
            continue
        too_old = datetime.utcnow() - backend_pb.meta.mtime.ToDatetime() > timedelta(days=28)
        resolver_status = backend_pb.resolver_status.last_attempt.succeeded.status
        if not too_old:
            continue
        if resolver_status != 'False':
            continue
        print('remove backend', backend_pb.meta.id, '?', 'old', too_old, 'resolver_status', resolver_status)

        # yn = raw_input()
        # while yn not in ('y', 'n'):
        #    yn = raw_input()
        # if yn == 'n':
        #    continue

        d.remove_backend(namespace_id, backend_pb.meta.id, backend_pb.meta.version)
        print('backend', backend_pb.meta.id, 'removed')

    # if validated_statuses == {'False'} and active_statuses == {'False'}:
    #    d.remove_backend(namespace_id, backend_pb.meta.id, backend_pb.meta.version)
    #    print 'backend', backend_pb.meta.id, 'removed'


@cli.command('bump_l7_macro_version')
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--to-version', required=True)
@click.option('--balancer-id')
def bump_l7_macro_version(app, namespace_id, to_version, balancer_id=None):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    if balancer_id is None:
        balancer_ids = d.list_balancer_ids(namespace_id)
    else:
        balancer_ids = [balancer_id]
    for balancer_id in balancer_ids:
        try:
            ok, msg = bump_l7_macro_version_balancer(d, namespace_id, balancer_id, to_version)
            if ok:
                click.echo('OK   {}: {}'.format(create_awacs_balancer_href(
                    namespace_id, balancer_id, d.nanny_url), msg))
                time.sleep(3)
            else:
                click.echo('FAIL {}: {}'.format(create_awacs_balancer_href(
                    namespace_id, balancer_id, d.nanny_url), msg))
        except Exception as e:
            click.echo('FAIL {}:{}'.format(create_awacs_balancer_href(
                namespace_id, balancer_id, d.nanny_url), six.text_type(e)))
            raise


def zerodiff_balancer(d, namespace_id, balancer_id):
    balancer_pb = d.get_balancer(namespace_id, balancer_id)
    if balancer_pb.meta.mtime.ToDatetime() > datetime.utcnow() - timedelta(hours=6):
        return False
    meta_pb = balancer_pb.meta
    spec_pb = balancer_pb.spec
    spec_pb.yandex_balancer.yaml += '\n'
    d.update_balancer(
        namespace_id, balancer_id, meta_pb.version, spec_pb,
        comment='https://st.yandex-team.ru/SWAT-6219')
    return True


@cli.command()
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--balancer-id')
def zerodiff(app, namespace_id, balancer_id=None):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    if balancer_id is None:
        balancer_ids = d.list_balancer_ids(namespace_id)
    else:
        balancer_ids = [balancer_id]
    for balancer_id in balancer_ids:
        try:
            ok = zerodiff_balancer(d, namespace_id, balancer_id)
            if ok:
                click.echo('zerodiffed {}'.format(create_awacs_balancer_href(namespace_id, balancer_id, d.nanny_url)))
                time.sleep(3)
        except Exception:
            click.echo('failed to zerodiff {}:{}'.format(namespace_id, balancer_id))
            raise


def set_balancer_location(d, namespace_id, balancer_id, yp_cluster):
    balancer_pb = d.get_balancer(namespace_id, balancer_id)
    meta_pb = balancer_pb.meta
    spec_pb = balancer_pb.spec
    meta_pb.location.type = model_pb2.BalancerMeta.Location.YP_CLUSTER
    meta_pb.location.yp_cluster = yp_cluster
    d.update_balancer(
        namespace_id, balancer_id, meta_pb.version,
        location_pb=meta_pb.location,
        comment='https://st.yandex-team.ru/SWAT-6219')
    return True


@cli.command('set_location')
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--balancer-id', required=True)
@click.option('--location', required=True)
def set_location(app, namespace_id, balancer_id, yp_cluster):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    if balancer_id is None:
        balancer_ids = d.list_balancer_ids(namespace_id)
    else:
        balancer_ids = [balancer_id]
    for balancer_id in balancer_ids:
        try:
            ok = set_balancer_location(d, namespace_id, balancer_id, yp_cluster)
            if ok:
                click.echo('OK {}'.format(create_awacs_balancer_href(namespace_id, balancer_id, d.nanny_url)))
                time.sleep(3)
            else:
                click.echo('FAIL on {}'.format(create_awacs_balancer_href(namespace_id, balancer_id, d.nanny_url)))
        except Exception:
            click.echo('failed to set location {}:{}'.format(namespace_id, balancer_id))
            raise


@cli.command('set_locations')
@click.pass_obj
def set_locations(d):
    if not d.nanny_token:
        print('"--nanny-token" option is mandatory option for this command')
        sys.exit(0)
    namespace_ids = d.list_namespace_ids()
    with click.progressbar(namespace_ids, label='processing namespaces') as bar:
        for namespace_id in bar:
            for balancer_pb in d.list_balancers(namespace_id):
                balancer_id = balancer_pb.meta.id
                service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
                service = d.get_service_info_attrs(service_id)
                yp_cluster = service['content'].get('yp_cluster')
                service_type = service['content'].get('type')
                if service_type == 'AWACS_BALANCER' and yp_cluster and not balancer_pb.meta.HasField('location'):
                    try:
                        ok = set_balancer_location(d, namespace_id, balancer_id, yp_cluster)
                        if ok:
                            click.echo(
                                'OK {}'.format(create_awacs_balancer_href(namespace_id, balancer_id, d.nanny_url)))
                            time.sleep(3)
                        else:
                            click.echo(
                                'FAIL on {}'.format(create_awacs_balancer_href(namespace_id, balancer_id, d.nanny_url)))
                    except Exception:
                        click.echo('failed to set location {}:{}'.format(namespace_id, balancer_id))
                        raise


@cli.command('bump_l3_ctl_version')
@click.pass_obj
@click.option('--just-stats', required=True, default=False)
def bump_l3_ctl_version(d, just_stats=False):
    namespace_ids = d.list_namespace_ids()
    x = 0
    y = 0
    for namespace_id in namespace_ids:
        for l3_balancer_pb in d.list_l3_balancers(namespace_id):
            l3_balancer_id = l3_balancer_pb.meta.id
            x += 1
            if l3_balancer_pb.spec.incomplete:
                continue
            if l3_balancer_pb.spec.ctl_version != 1:
                l3_balancer_pb.spec.ctl_version = 1
                if just_stats:
                    y += 1
                else:
                    d.update_l3_balancer(
                        namespace_id, l3_balancer_id, l3_balancer_pb.meta.version, l3_balancer_pb.spec,
                        comment='SWATOPS-122: use new ctl')
                    click.echo(
                        'UPDATED {}'.format(create_awacs_l3_balancer_href(namespace_id, l3_balancer_id, d.nanny_url)))
                    time.sleep(5)
            else:
                if not just_stats:
                    click.echo(
                        'SKIPPED {}'.format(create_awacs_l3_balancer_href(namespace_id, l3_balancer_id, d.nanny_url)))
    print(float(y) / x * 100, '%')


@cli.command('bump_alerting_version')
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--to-version', required=True)
@click.option('--no-act', is_flag=True)
def bump_alerting_version(app, namespace_id, to_version, no_act=False):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    try:
        namespace_pb = d.get_namespace(namespace_id)
    except Exception as e:
        click.echo('SKIPPED: {}: {}'.format(namespace_id, e))
        return

    meta_pb = namespace_pb.meta
    spec_pb = namespace_pb.spec
    if spec_pb.deleted or spec_pb.incomplete:
        click.echo('SKIPPED: {}: not applicable'.format(namespace_id))
        return

    if not spec_pb.HasField("alerting") or not spec_pb.alerting.version:
        click.echo("SKIPPED: {}: alerting is not configured".format(namespace_id))
        return

    click.echo("UPDATING: {}: {} -> {}".format(namespace_id, spec_pb.alerting.version, to_version))

    if not no_act:
        spec_pb.alerting.version = to_version
        try:
            d.update_namespace(
                namespace_id=namespace_id,
                auth_pb=meta_pb.auth,
                spec_pb=spec_pb,
                meta_version=meta_pb.version,
                comment='Update alerting to version {}'.format(to_version)
            )
        except Exception as e:
            click.echo(
                'FAIL: {} : {}'.format(
                    create_awacs_namespace_href(namespace_id, d.nanny_url),
                    e,
                )
            )
            raise

    for balancer_id in d.list_balancer_ids(namespace_id):
        balancer_pb = d.get_balancer(namespace_id, balancer_id)
        click.echo('SERVICE_NEED_UPDATE: {}'.format(balancer_pb.spec.config_transport.nanny_static_file.service_id))

    click.echo('OK: {}'.format(namespace_id))


@cli.command('add_sd_to_full_mode_balancers')
@click.pass_obj
def add_sd_to_full_mode_balancers(app):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    BLACKLIST = [
        's.yandex-team.ru',
    ]

    def is_blacklisted(ns_id):
        return ns_id in BLACKLIST

    namespace_ids = d.list_namespace_ids()
    DC = 'XDC'
    approved_until_i = 0
    for i, namespace_id in enumerate(namespace_ids):
        if is_blacklisted(namespace_id):
            continue
        balancer_pbs = d.list_balancers(namespace_id)

        for balancer_pb in balancer_pbs:
            balancer_id = balancer_pb.meta.id
            spec_pb = balancer_pb.spec
            if spec_pb.incomplete:
                continue
            if spec_pb.yandex_balancer.mode != spec_pb.yandex_balancer.FULL_MODE:
                continue
            config_pb = spec_pb.yandex_balancer.config
            yaml = spec_pb.yandex_balancer.yaml
            assert config_pb.HasField('instance_macro'), namespace_id + '/' + balancer_id
            assert config_pb.instance_macro.HasField('unistat')
            if 'sd:' in yaml:
                assert config_pb.instance_macro.HasField('sd')
                continue
            else:
                assert not config_pb.instance_macro.HasField('sd')
            if (balancer_pb.meta.location.yp_cluster or
                balancer_pb.meta.location.gencfg_dc or
                'XDC').lower() != DC.lower():
                continue

            if i > approved_until_i:
                print(namespace_id, balancer_id, 'is to be fixed')
                print('fix it? y/n')
                yn = input()
                while yn not in ('y', 'n'):
                    yn = input()
                if yn == 'n':
                    continue
                approved_until_i = i + 300

            lines = yaml.split('\n')
            s = 0
            if lines[s] == '---':
                s += 1
            assert 'instance_macro:' in lines[s], namespace_id + '/' + balancer_id
            m = re.search('^\s+', lines[s + 1])
            assert m, namespace_id + '/' + balancer_id
            indent = m.group()
            lines.insert(s + 1, indent + 'sd: {}')
            updated_yaml = '\n'.join(lines)

            spec_pb.yandex_balancer.yaml = updated_yaml
            try:
                d.update_balancer(namespace_id=namespace_id,
                                  balancer_id=balancer_id,
                                  version=balancer_pb.meta.version,
                                  spec_pb=spec_pb,
                                  comment='SWATOPS-269: enable sd')
            except Exception:
                print('failed to update', namespace_id, balancer_id)
            else:
                print('updated https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/balancers/list/{}/'.format(
                    namespace_id, balancer_id))


@cli.command('fix_count_backends')
@click.pass_obj
def fix_count_backends(app):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    BLACKLIST = [
        's.yandex-team.ru',
    ]
    BLACKLIST_SUFFIXES = ('eda.tst.yandex.net', 'eda.yandex.net', 'taxi.yandex.net', 'taxi.tst.yandex.net',
                          'lavka.tst.yandex.net', 'lavka.yandex.net')

    def is_blacklisted(ns_id):
        return ns_id in BLACKLIST or ns_id.endswith(BLACKLIST_SUFFIXES)

    namespace_ids = d.list_namespace_ids()
    for namespace_id in namespace_ids:
        # if is_blacklisted(namespace_id):
        #    continue
        upstream_pbs = d.list_upstreams(namespace_id)

        maybe_affected = False
        for upstream_pb in upstream_pbs:
            yaml = upstream_pb.spec.yandex_balancer.yaml
            if ('count_backends(true)' in yaml):
                print(upstream_pb.meta.id)
                maybe_affected = True

        if not maybe_affected:
            continue

        print(namespace_id, 'is affected')
        print('fix it? y/n')
        yn = input()
        while yn not in ('y', 'n'):
            yn = input()
        if yn == 'n':
            continue

        for upstream_pb in upstream_pbs:
            spec_pb = upstream_pb.spec
            yaml = spec_pb.yandex_balancer.yaml
            if (
                'count_backends(true)' not in yaml
            ):
                continue
            spec_pb.yandex_balancer.yaml = spec_pb.yandex_balancer.yaml.replace(
                'count_backends(true)',
                'count_backends()')

            d.update_upstream(namespace_id=upstream_pb.meta.namespace_id, upstream_id=upstream_pb.meta.id,
                              version=upstream_pb.meta.version, spec_pb=spec_pb,
                              comment='SWAT-6294: fix count_backends usage')
            print('* updated {}/{}'.format(upstream_pb.meta.namespace_id, upstream_pb.meta.id))


@cli.command('migrate_backends_to_sd')
@click.pass_obj
@click.option('--namespace-id', required=True)
def migrate_backends_to_sd(app, namespace_id):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    for backend_pb in d.list_backends(namespace_id):
        if backend_pb.meta.id == 'rtc_balancer_frontend-autobetas_sas_vla':
            continue
        if backend_pb.spec.deleted:
            continue
        if backend_pb.spec.selector.type == backend_pb.spec.selector.YP_ENDPOINT_SETS:
            print(backend_pb.meta.id)
            print('OK?')
            if input() == 'y':
                backend_pb.spec.selector.type = backend_pb.spec.selector.YP_ENDPOINT_SETS_SD
                d.update_backend(namespace_id=backend_pb.meta.namespace_id, backend_id=backend_pb.meta.id,
                                 version=backend_pb.meta.version, spec_pb=backend_pb.spec,
                                 comment='SWATOPS-270: use SD')
                print('updated')


@cli.command('migrate_backends_to_sd2')
@click.pass_obj
@click.option('--namespace-id', required=True)
def migrate_backends_to_sd2(app, namespace_id):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    rps_data = d.get_rps_data()
    affected_ns_ids = set()
    for i, namespace_id in enumerate([namespace_id]):
        adjacent_backends = set()
        graph = d.get_inclusion_graph(namespace_id)
        for node in graph:
            if node['type'] == 'upstream':
                included_backend_ids = set()
                for flat_id in node['included_backend_ids']:
                    backend_ns_id, backend_id = flat_id.split('/')
                    if backend_ns_id != namespace_id:
                        continue
                    included_backend_ids.add(backend_id)
                adjacent_backends.add(frozenset(included_backend_ids))

        max_rps = rps_data.get(namespace_id, 0)
        l7_backend_ids = set()
        should_namespace_be_migrated = False
        affected_backend_ids = set()
        backend_pbs = d.list_backends(namespace_id)
        for backend_pb in backend_pbs:
            backend_id = backend_pb.meta.id
            if backend_pb.spec.deleted:
                continue
            if backend_pb.resolver_status.last_attempt.succeeded.status != 'True':
                continue
            used_in_dns = False
            for rev_pb in backend_pb.dns_record_statuses:
                for _, cond_pb in rev_pb.validated.items():
                    if cond_pb.status == 'True':
                        used_in_dns = True
            used_in_l3 = False
            for rev_pb in backend_pb.l3_statuses:
                for _, cond_pb in rev_pb.validated.items():
                    if cond_pb.status == 'True':
                        used_in_l3 = True
            used_in_l7 = False
            for rev_pb in backend_pb.statuses:
                for _, cond_pb in rev_pb.validated.items():
                    if cond_pb.status == 'True':
                        used_in_l7 = True
            if used_in_l7:
                l7_backend_ids.add(backend_id)
            if (not used_in_dns and
                not used_in_l3 and
                backend_pb.spec.selector.type == backend_pb.spec.selector.YP_ENDPOINT_SETS and
                backend_pb.spec.selector.port.policy == backend_pb.spec.selector.port.KEEP):
                for es_pb in backend_pb.spec.selector.yp_endpoint_sets:
                    should_namespace_be_migrated = True
                    affected_backend_ids.add(backend_id)

                for es_pb in backend_pb.spec.selector.yp_endpoint_sets:
                    if es_pb.port.policy != es_pb.port.KEEP:
                        affected_backend_ids.discard(backend_id)
                    if es_pb.weight.policy != es_pb.weight.KEEP:
                        affected_backend_ids.discard(backend_id)
                    req = {
                        'endpoint_set_id': es_pb.endpoint_set_id,
                        'client_name': 'awacszerodiffer',
                        'cluster_name': es_pb.cluster.lower(),
                    }
                    resp = requests.post('http://sd.yandex.net:8080/resolve_endpoints/json', json=req)
                    if resp.json()['resolve_status'] != 2:
                        affected_backend_ids.discard(backend_id)

        # if should_be_migrated:
        #   affected_ns_ids.add(namespace_id)
        # print 'affected', len(affected_ns_ids), 'of', i
        # continue

        if affected_backend_ids != l7_backend_ids:
            print('adjacent_backends')
            print(adjacent_backends)
            print()
            print('affected_backend_ids != l7_backend_ids')
            print('affected_backend_ids')
            print(', '.join(sorted(affected_backend_ids)))
            print()
            print('l7_backend_ids')
            print(', '.join(sorted(l7_backend_ids)))
            print()
            print('affected_backend_ids - l7_backend_ids')
            print(', '.join(sorted(affected_backend_ids - l7_backend_ids)))
            print()
            print('l7_backend_ids - affected_backend_ids')
            print(', '.join(sorted(l7_backend_ids - affected_backend_ids)))
            print()
            print('l7_backend_ids & affected_backend_ids')
            print(', '.join(sorted(l7_backend_ids & affected_backend_ids)))
            print()
            print('all l7 backends can be fixed', l7_backend_ids.issubset(affected_backend_ids))

        backends_to_be_fixed = l7_backend_ids & affected_backend_ids
        for backend_ids in adjacent_backends:
            if (any(backend_id in backends_to_be_fixed for backend_id in backend_ids) and
                not all(backend_id in backends_to_be_fixed for backend_id in backend_ids)):
                print('group {} is mixed, wontfix'.format(backend_ids))
                backends_to_be_fixed -= backend_ids

        print('backends_to_be_fixed', ', '.join(sorted(backends_to_be_fixed)))

        if should_namespace_be_migrated:
            print('Migrate {} ({} affected backends, {} RPS)?'.format(namespace_id, len(affected_backend_ids), max_rps))
            if input() == 'y':
                for backend_pb in backend_pbs:
                    if backend_pb.meta.id not in backends_to_be_fixed:
                        continue
                    backend_pb.spec.selector.type = backend_pb.spec.selector.YP_ENDPOINT_SETS_SD
                    d.update_backend(namespace_id=backend_pb.meta.namespace_id, backend_id=backend_pb.meta.id,
                                     version=backend_pb.meta.version, spec_pb=backend_pb.spec,
                                     comment='SWATOPS-270: use SD')
                    print(backend_pb.meta.id)
                print('Migrated https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/show/'.format(namespace_id))

    print('Total', len(affected_ns_ids))
    print('Affected ns ids')
    print('\n'.join(sorted(affected_ns_ids)))


def enable_matcher_map_fix(d, namespace_id, balancer_id):
    balancer_pb = d.get_balancer(namespace_id, balancer_id)
    meta_pb = balancer_pb.meta

    spec_pb = balancer_pb.spec
    if spec_pb.yandex_balancer.config.HasField('l7_macro'):
        return False, 'is l7_macro'

    cfg = spec_pb.yandex_balancer.yaml

    if 'enable_matcher_map_fix: true' in cfg:
        return False, 'enable_matcher_map_fix: true'

    lines = cfg.split('\n')
    i = lines.index('  workers: !f get_workers()')
    lines.insert(i + 1, '  enable_matcher_map_fix: true')
    new_cfg = '\n'.join(lines)
    spec_pb.yandex_balancer.yaml = new_cfg
    d.update_balancer(
        namespace_id, balancer_id, meta_pb.version, spec_pb,
        comment='SWAT-5801: enable_matcher_map_fix: true')
    return True, 'ok'


@cli.command('enable_matcher_map_fix_activation')
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--balancer-id')
def enable_matcher_map_fix_activation(app, namespace_id, balancer_id=None):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    balancer_ids = d.list_balancer_ids(namespace_id) if balancer_id is None else [balancer_id]
    for balancer_id in balancer_ids:
        try:
            ok, msg = enable_matcher_map_fix(d, namespace_id, balancer_id)
            if ok:
                click.echo(
                    'enable_matcher_map_fix activated {}'.format(
                        create_awacs_balancer_href(namespace_id, balancer_id, d.nanny_url)))
                time.sleep(1)
                raise Exception
            else:
                click.echo(click.style(msg, fg='red'))
        except Exception:
            click.echo('failed to enable_matcher_map_fix {}:{}'.format(namespace_id, balancer_id))
            raise


@cli.command('set_namespace_annotations')
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--comment', type=str)
@click.option('--set', '-s', 'set_annotations', multiple=True, type=(str, str))
@click.option('--unset', '-u', 'unset_annotations', multiple=True, type=str)
@click.option('--annotations-file', type=click.Path(exists=True, file_okay=True, readable=True))
def set_namespace_annotations(app, comment, namespace_id, set_annotations, unset_annotations, annotations_file):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    try:
        namespace_pb = d.get_namespace(namespace_id)

        if annotations_file:
            with open(annotations_file) as f:
                annotations_data = yaml.load(f.read(), Loader=yaml.SafeLoader)
        else:
            annotations_data = {}

    except Exception as e:
        click.echo('SKIPPED: {}: {}'.format(namespace_id, e))
        return

    meta_pb = namespace_pb.meta
    spec_pb = namespace_pb.spec
    if spec_pb.deleted or spec_pb.incomplete:
        click.echo('SKIPPED: {}: not applicable'.format(namespace_id))
        return

    for k in annotations_data.get('unset', []):
        meta_pb.annotations.pop(k, None)

    for k, v in annotations_data.get('set', {}).items():
        meta_pb.annotations[k] = v

    for k in unset_annotations:
        meta_pb.annotations.pop(k, None)

    for k, v in set_annotations:
        meta_pb.annotations[k] = v

    if not len(meta_pb.annotations):
        click.echo("SKIPPED: {}: resulting annotations list is empty and request would fail".format(namespace_id))
        return

    click.echo("UPDATING: {}".format(namespace_id))

    try:
        d.update_namespace(
            namespace_id=namespace_id,
            auth_pb=meta_pb.auth,
            meta_version=meta_pb.version,
            annotations=meta_pb.annotations,
            comment=comment or 'Update annotations',
        )
    except Exception as e:
        click.echo(
            'FAIL: {} : {}'.format(
                create_awacs_namespace_href(namespace_id, d.nanny_url),
                e,
            )
        )
        raise


"""
@cli.command('fix_auth')
@click.pass_obj
def fix_auth(d):
    for namespace_pb in d.iter_all_namespaces():
        if u'moridin' in namespace_pb.meta.auth.staff.owners.logins or \
                u'keepclean' in namespace_pb.meta.auth.staff.owners.logins:
            prev = list(namespace_pb.meta.auth.staff.owners.logins)
            if u'moridin' in namespace_pb.meta.auth.staff.owners.logins:
                namespace_pb.meta.auth.staff.owners.logins.remove('moridin')
            if u'keepclean' in namespace_pb.meta.auth.staff.owners.logins:
                namespace_pb.meta.auth.staff.owners.logins.remove('keepclean')
            if not namespace_pb.meta.auth.staff.owners.logins:
                namespace_pb.meta.auth.staff.owners.logins.append('nanny-robot')
            upd = list(namespace_pb.meta.auth.staff.owners.logins) or ['nanny-robot']
            print('UPDATED NS', namespace_pb.meta.id, prev, '->', upd)
            try:
                d.update_namespace(namespace_id=namespace_pb.meta.id, auth_pb=namespace_pb.meta.auth)
            except Exception as e:
                print('EXCEPTION NS', e)

    for upstream_pb in d.iter_all_upstreams():
        if u'moridin' in upstream_pb.meta.auth.staff.owners.logins or \
                u'keepclean' in upstream_pb.meta.auth.staff.owners.logins:
            prev = list(upstream_pb.meta.auth.staff.owners.logins)
            if u'moridin' in upstream_pb.meta.auth.staff.owners.logins:
                upstream_pb.meta.auth.staff.owners.logins.remove('moridin')
            if u'keepclean' in upstream_pb.meta.auth.staff.owners.logins:
                upstream_pb.meta.auth.staff.owners.logins.remove('keepclean')
            if not upstream_pb.meta.auth.staff.owners.logins:
                upstream_pb.meta.auth.staff.owners.logins.append('nanny-robot')
            upd = list(upstream_pb.meta.auth.staff.owners.logins) or ['nanny-robot']
            print('UPDATED UP', upstream_pb.meta.namespace_id, upstream_pb.meta.id, prev, '->', upd)
            try:
                d.update_upstream(namespace_id=upstream_pb.meta.namespace_id,
                                  upstream_id=upstream_pb.meta.id,
                                  version='0', auth_pb=upstream_pb.meta.auth)
            except Exception as e:
                print('EXCEPTION UP', e)

    for balancer_pb in d.iter_all_balancers():
        if u'moridin' in balancer_pb.meta.auth.staff.owners.logins or \
                u'keepclean' in balancer_pb.meta.auth.staff.owners.logins:
            prev = list(balancer_pb.meta.auth.staff.owners.logins)
            if u'moridin' in balancer_pb.meta.auth.staff.owners.logins:
                balancer_pb.meta.auth.staff.owners.logins.remove('moridin')
            if u'keepclean' in balancer_pb.meta.auth.staff.owners.logins:
                balancer_pb.meta.auth.staff.owners.logins.remove('keepclean')
            if not balancer_pb.meta.auth.staff.owners.logins:
                balancer_pb.meta.auth.staff.owners.logins.append('nanny-robot')
            upd = list(balancer_pb.meta.auth.staff.owners.logins) or ['nanny-robot']
            print('UPDATED', balancer_pb.meta.namespace_id, balancer_pb.meta.id,
                  prev, '->', upd)
            try:
                d.update_balancer(namespace_id=balancer_pb.meta.namespace_id, balancer_id=balancer_pb.meta.id,
                                  version='0', auth_pb=balancer_pb.meta.auth)
            except Exception as e:
                print('EXCEPTION', e)

    for backend_pb in d.iter_all_backends():
        if u'moridin' in backend_pb.meta.auth.staff.owners.logins or \
                u'keepclean' in backend_pb.meta.auth.staff.owners.logins:
            prev = list(backend_pb.meta.auth.staff.owners.logins)
            if u'moridin' in backend_pb.meta.auth.staff.owners.logins:
                backend_pb.meta.auth.staff.owners.logins.remove('moridin')
            if u'keepclean' in backend_pb.meta.auth.staff.owners.logins:
                backend_pb.meta.auth.staff.owners.logins.remove('keepclean')
            if not backend_pb.meta.auth.staff.owners.logins:
                backend_pb.meta.auth.staff.owners.logins.append('nanny-robot')
            upd = list(backend_pb.meta.auth.staff.owners.logins) or ['nanny-robot']
            print('UPDATED', backend_pb.meta.namespace_id, backend_pb.meta.id,
                  prev, '->', upd)
            try:
                d.update_backend(namespace_id=backend_pb.meta.namespace_id, backend_id=backend_pb.meta.id,
                                 version='0', auth_pb=backend_pb.meta.auth)
            except Exception as e:
                print('EXCEPTION', e)
            time.sleep(.2)

    for cert_pb in d.iter_all_certs():
        if u'moridin' in cert_pb.meta.auth.staff.owners.logins or \
                u'keepclean' in cert_pb.meta.auth.staff.owners.logins:
            prev = list(cert_pb.meta.auth.staff.owners.logins)
            if u'moridin' in cert_pb.meta.auth.staff.owners.logins:
                cert_pb.meta.auth.staff.owners.logins.remove('moridin')
            if u'keepclean' in cert_pb.meta.auth.staff.owners.logins:
                cert_pb.meta.auth.staff.owners.logins.remove('keepclean')
            if not cert_pb.meta.auth.staff.owners.logins:
                cert_pb.meta.auth.staff.owners.logins.append('nanny-robot')
            upd = list(cert_pb.meta.auth.staff.owners.logins) or ['nanny-robot']
            print('UPDATED CERT', cert_pb.meta.namespace_id, cert_pb.meta.id,
                  prev, '->', upd)
            try:
                d.update_cert(namespace_id=cert_pb.meta.namespace_id, cert_id=cert_pb.meta.id,
                              version=cert_pb.meta.version, auth_pb=cert_pb.meta.auth)
            except Exception as e:
                print('EXCEPTION', e)
            time.sleep(.2)
"""


@cli.command('balancersupport4053')
@click.pass_obj
def balancersupport4053(app):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    d = app.awacs_client
    ns_ids = '''https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-constructor-int.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-jamsarm.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-jsapi.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-mobmaps-proxy-api.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-nmaps.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-redirect.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-specprojects-ext.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-sputnica.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-telegram-bot.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-tiles-api.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-upload-api.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-viber-bot.slb.maps.yandex.net
https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/front-maps.tst.slb.maps.yandex.net'''.splitlines()
    ns_ids = [ns_id.rsplit('/')[-1] for ns_id in ns_ids]
    for namespace_id in ns_ids:
        try:
            namespace_pb = d.get_namespace(namespace_id)
        except Exception as e:
            click.echo('SKIPPED: {}: {}'.format(namespace_id, e))
            continue

        meta_pb = namespace_pb.meta
        spec_pb = namespace_pb.spec
        if 'MAPS_FRONT' in spec_pb.rps_limiter_allowed_installations.installations:
            click.echo('ALREADY PROCESSED: {}'.format(namespace_id))
            continue
        spec_pb.rps_limiter_allowed_installations.installations.append('MAPS_FRONT')
        try:
            d.update_namespace(
                namespace_id=namespace_id,
                auth_pb=meta_pb.auth,
                meta_version=meta_pb.version,
                spec_pb=spec_pb,
                comment='Allow MAPS_FRONT',
            )
        except Exception as e:
            click.echo('FAIL: {} : {}'.format(namespace_id, e))
            raise
        click.echo('DONE: {}'.format(namespace_id))


if __name__ == '__main__':
    cli()
