import os
import re
import time
import warnings

import nanny_rpc_client
import requests
import yaml
from infra.awacs.proto import api_stub, api_pb2


AWACS_KEY = os.getenv("AWACS_TOKEN", '')
STAFF_KEY = 'OAuth ' + os.getenv('STAFF_OAUTH', '')
NS_MATCHER = re.compile('list/([^/]+)/')


def get_logins(session, slug):
    url = 'https://staff-api.yandex-team.ru/v3/groupmembership?group.url={}&person.official.is_dismissed=false&person.official.is_robot=false&_fields=person.login'
    response = session.get(url.format(slug[1:]), headers={'Authorization': STAFF_KEY}, verify=False)
    response.raise_for_status()
    for person in response.json()['result']:
        login = (person.get('person') or {}).get('login')
        if login:
            yield login


def get_telegram_bindings(session):
    response = session.get('http://juggler-api.search.yandex.net/api/notifications/list_telegram_bindings?do=1')
    response.raise_for_status()
    return response.json()['bindings']


def patch_namespace(awacs_client, idx, total, ns_id, dry_run=True):
    try:
        req = api_pb2.GetNamespaceRequest(id=ns_id)
        resp = awacs_client.get_namespace(req)

        req = api_pb2.UpdateNamespaceRequest()
        req.meta.id = ns_id
        req.meta.CopyFrom(resp.namespace.meta)
        req.spec.CopyFrom(resp.namespace.spec)
        need_patch = False
        for rule in req.spec.alerting.juggler_raw_notify_rules.balancer[:2]:
            template_kwargs = yaml.load(rule.template_kwargs, Loader=yaml.SafeLoader)
            if 'sms' in template_kwargs['method'] and 'telegram' not in template_kwargs['method']:
                template_kwargs['method'].remove('sms')
                template_kwargs['method'].append('telegram')
                rule.template_kwargs = yaml.safe_dump(template_kwargs)
                need_patch = True
        if need_patch:
            print(f"[{idx:4d} / {total:4d}] will patch namespace {ns_id!r}")
            if not dry_run:
                awacs_client.update_namespace(req)
            print(f"[{idx:4d} / {total:4d}] namespace updated.")
        else:
            print(f"[{idx:4d} / {total:4d}] will skip namespace {ns_id!r}")
    except Exception as e:
        print(f"[{idx:4d} / {total:4d}] namespace {ns_id!r} patch failed: {e}")


def get_namespaces_to_patch():
    page = 1
    page_size = 1000
    group_cache = {}
    url = 'https://juggler-api.search.yandex.net/api/notify_rules/get_notify_rules?do=1&format=json'

    namespaces_to_patch = set()

    with requests.Session() as s:
        tg_chats = get_telegram_bindings(s)

        while True:
            response = s.post(url, json={"page": page, "page_size": page_size}, verify=False)
            response.raise_for_status()
            rules = response.json()['rules']

            print(f"Got {len(rules)} rules on page {page}")

            if not rules:
                break

            for rule in rules:
                logins = set()

                if 'Generated by awacs' not in str(rule.get('description')):
                    continue

                if not isinstance(rule.get('template_kwargs'), dict):
                    continue

                if not rule.get('namespace') or not rule['namespace'].startswith('awacs.'):
                    continue

                method = rule['template_kwargs'].get('method')
                if isinstance(method, str) and method != 'sms':
                    continue
                if not isinstance(method, list) or 'sms' not in method:
                    continue

                logins_and_groups = rule['template_kwargs'].get('login')
                if not isinstance(logins_and_groups, list):
                    continue

                for login in logins_and_groups:
                    if not login.startswith('@'):
                        logins.add(login)
                    elif login in group_cache:
                        logins.update(group_cache[login])
                    else:
                        group_cache[login] = list(get_logins(s, login))
                        logins.update(group_cache[login])

                tg_available = len(list(filter(lambda login: login in tg_chats, logins)))
                if len(logins) > 3 and tg_available > 1 and float(tg_available) / len(logins) >= 0.75:
                    awacs_ns = re.search(NS_MATCHER, rule['description']).group(1)
                    namespaces_to_patch.add(awacs_ns)
                    # writer.writerow([rule.get("namespace"), rule.get("rule_id"), awacs_ns, len(logins), tg_available])
                    # print(f'Namespace {awacs_ns!r} has {len(logins)} sms receivers in rule {rule.get("rule_id")!r}')

            page += 1

    return namespaces_to_patch


def main():
    dry_run = os.getenv('DRY_RUN', '1') != '0'
    namespaces = get_namespaces_to_patch()
    total = len(namespaces)
    print(f"Will patch {total} namespaces")

    rpc = nanny_rpc_client.RetryingRpcClient(rpc_url='https://awacs.yandex-team.ru/api/', oauth_token=AWACS_KEY)
    client = api_stub.NamespaceServiceStub(rpc)

    for idx, namespace in enumerate(sorted(namespaces), start=1):
        patch_namespace(client, idx, total, namespace, dry_run=dry_run)
        time.sleep(3)


if __name__ == '__main__':
    warnings.simplefilter('ignore')
    main()
