import click

from awacs.model.l3_balancer import errors
from awacs.model.l3_balancer import l3_balancer
from awacs.web.helpers_service import _enable_dynamic_weights  # noqa
from infra.awacs.proto import model_pb2
from infra.awacs.tools.awacstoolslib.util import cli


DEFAULT_L3MGR_API_URL = 'https://l3-api.tt.yandex-team.ru/'


def create_l3mgr_svc_href(svc_id):
    return 'https://l3.tt.yandex-team.ru/service/{0}'.format(svc_id)


def create_l3mgr_cfg_href(svc_id, cfg_id):
    return 'https://l3.tt.yandex-team.ru/service/{0}/config/{1}'.format(svc_id, cfg_id)


@cli.command()
@click.option('--svc-id', required=True)
@click.option('--cfg-1-id', required=True)
@click.option('--cfg-2-id', required=True)
@click.option('--full-rs-diff', default=False)
def diff(svc_id, cfg_1_id, cfg_2_id, full_rs_diff=False):
    cfg_1 = l3_balancer.ServiceConfig.from_api(svc_id, cfg_1_id)
    cfg_2 = l3_balancer.ServiceConfig.from_api(svc_id, cfg_2_id)
    rs_group_1 = rs_group_2 = vs_cfg_1 = vs_cfg_2 = None
    try:
        rs_group_1 = l3_balancer.RSGroup.from_vs(cfg_1.virtual_services)
    except errors.RSGroupsConflict as e:
        print('{}: RS groups are inconsistent across VS: {}'.format(cfg_1_id, e))
    try:
        rs_group_2 = l3_balancer.RSGroup.from_vs(cfg_2.virtual_services)
    except errors.RSGroupsConflict as e:
        print('{}: RS groups are inconsistent across VS: {}'.format(cfg_2_id, e))

    try:
        vs_cfg_1 = l3_balancer.VirtualServiceConfig.from_virtual_services(cfg_1.virtual_services)
    except errors.VSConfigsConflict as e:
        print('{}: VS configs are inconsistent across VS: {}'.format(cfg_1_id, e))
    try:
        vs_cfg_2 = l3_balancer.VirtualServiceConfig.from_virtual_services(cfg_2.virtual_services)
    except errors.VSConfigsConflict as e:
        print('{}: VS configs are inconsistent across VS: {}'.format(cfg_2_id, e))

    if vs_cfg_1 and vs_cfg_2:
        print('cfg_1 config == cfg_2 config:', cfg_1.config == cfg_2.config)
    if rs_group_1 and rs_group_2:
        print('cfg_1 real servers == cfg_2 real servers:', rs_group_1 == rs_group_2)

    if full_rs_diff:
        if not (rs_group_1 and rs_group_2):
            print('Sorry, full rs diff is too difficult to show')
        else:
            rs_common = rs_group_1.real_servers & rs_group_2.real_servers
            print('cfg_1.rs & cfg_2.rs ({}):'.format(len(rs_common)))
            for rs in sorted(rs_common):
                print(' *', rs)
            rs1_not_rs2 = rs_group_1.real_servers - rs_group_2.real_servers
            print('cfg_1.rs - cfg_2.rs ({}):'.format(len(rs1_not_rs2)))
            for rs in sorted(rs1_not_rs2):
                print(' *', rs)
            rs2_not_rs1 = rs_group_2.real_servers - rs_group_1.real_servers
            print('cfg_2.rs - cfg_1.rs ({}):'.format(len(rs2_not_rs1)))
            for rs in sorted(rs2_not_rs1):
                print(' *', rs)


def _add_vs(l3mgr, svc_id, port, is_external=None, ip_family=None, use_existing_ip=None):
    print('Getting current config...')
    s = l3_balancer.Service.from_api(svc_id)

    rs_group = l3_balancer.RSGroup.from_vs(s.virtual_services)
    assert s.abc_slug == 'rclb'
    vs_cfg = l3_balancer.VirtualServiceConfig.from_virtual_services(s.virtual_services)
    curr_rs_ids = s.virtual_services[0][u'rs']
    curr_vs_ids = [vs[u'id'] for vs in s.virtual_services]

    if use_existing_ip is not None:
        ip_addr = use_existing_ip
    else:
        assert is_external is not None
        assert ip_family is not None
        print('Allocating {} IP{} address...'.format('external' if is_external else 'internal', ip_family))
        is_v4 = ip_family == 'v4'
        resp = l3mgr.get_ip(abc_code='rclb', v4=is_v4, external=is_external)
        ip_addr = resp['object']

    print('Creating new virtual server...')
    resp = l3mgr.create_virtual_server(
        svc_id=svc_id,
        ip=ip_addr,
        port=port,
        protocol='TCP',
        config=vs_cfg.config,
        rs=curr_rs_ids,
        groups=rs_group.to_string(),
    )
    vs_id = resp['object']['id']

    new_vs_ids = curr_vs_ids + [vs_id]
    resp = l3mgr.save_config(svc_id=svc_id, vs_ids=new_vs_ids, comment='Add VS for port {} on IP {}'.format(port,
                                                                                                            ip_addr))

    new_cfg_id = resp['object']['id']

    print('Processing new config...')
    l3mgr.process_service_cfg(svc_id=svc_id, cfg_id=new_cfg_id)

    print('Success: {}'.format(create_l3mgr_cfg_href(svc_id, new_cfg_id)))


@cli.command('add_vs')
@click.pass_obj
@click.option('--svc-id', required=True)
@click.option('--ip-family', type=click.Choice(['v4', 'v6']), required=True)
@click.option('--port', type=int, required=True)
@click.option('--is-external', type=bool)
def add_vs(app, svc_id, ip_family, port, is_external):
    l3mgr = app.l3mgr_client
    return _add_vs(l3mgr, svc_id, ip_family, port, is_external)


@cli.command('add_vs_on_existing_ip')
@click.pass_obj
@click.option('--svc-id', type=str, required=True)
@click.option('--ip', type=str, required=True)
@click.option('--port', type=int, required=True)
def add_vs_on_existing_ip(app, svc_id, ip, port):
    l3mgr = app.l3mgr_client
    return _add_vs(l3mgr, svc_id, port, is_external=None, ip_family=None, use_existing_ip=ip)


@cli.command()
@click.pass_obj
@click.option('--abc-service', type=str, required=True)
def ls(app, abc_service):
    l3mgr = app.l3mgr_client
    resp = l3mgr.get('/api/v1/service', params={'abc__exact': abc_service, 'archive': False, '_limit': 5000})
    print(resp['total'], 'total')
    assert resp['total'] == len(resp['objects'])
    active_count = 0
    for obj in resp['objects']:
        if obj['state'] == 'DISABLED':
            continue
        active_count += 1
        svc_id = obj['id']
        svc = l3_balancer.Service.from_api(svc_id)
        ips = set()
        for vs in svc.virtual_services:
            ips.add(vs['ip'])
        print(svc.fqdn, 'has', len(ips), 'IPs:', sorted(ips))
    print('ACTIVE count', active_count)


@cli.command('awacs_324')
@click.pass_obj
@click.option('--svc-id', type=str, required=True)
def awacs_324(app, svc_id):
    l3mgr = app.l3mgr_client
    new_cfg_id = _enable_dynamic_weights(l3mgr, svc_id, u'Dynamic weights enabled with l3mgrctl')
    print('Success:', create_l3mgr_cfg_href(svc_id, new_cfg_id))


@cli.command('awacs_1338')
@click.pass_obj
@click.option('--namespace-id', type=str, required=True)
@click.option('--id', type=str, required=True)
@click.option('--ip', type=str, required=True)
@click.option('--dry-run', type=bool, default=True)
def awacs_1338(app, namespace_id, id, ip, dry_run):
    # ./l3mgrctl awacs_1338 \
    # --namespace-id kp-nginx-stable-balancer.kp.yandex.net \
    # --id kp-nginx-stable-balancer.kp.yandex.net \
    # --ip 5.45.202.180
    a = app.awacs_client
    pb = a.get_l3_balancer(namespace_id, id)
    spec_pb = pb.spec
    print('=== CURRENT')
    print(spec_pb)
    print()

    for port in (80, 443):
        hc_pb = model_pb2.L3BalancerSpec.VirtualServer.HealthCheckSettings(
            url='/ping',
            check_type=(model_pb2.L3BalancerSpec.VirtualServer.HealthCheckSettings.CT_SSL_GET if port == 443 else
                        model_pb2.L3BalancerSpec.VirtualServer.HealthCheckSettings.CT_HTTP_GET)
        )
        spec_pb.virtual_servers.add(
            ip=ip,
            port=port,
            health_check_settings=hc_pb,
            traffic_type=model_pb2.L3BalancerSpec.VirtualServer.TT_EXTERNAL
        )
    print('=== UPDATED')
    print(spec_pb)
    print()
    if not dry_run:
        a.update_l3_balancer(namespace_id, id, pb.meta.version, pb.spec)
        print('Done!')


if __name__ == '__main__':
    cli()
