# -*- coding: utf-8 -*-
from collections import OrderedDict

import click
import requests
from infra.swatconf import YASM_API_URL, Signal, Graphic, Text, Alert, Panel, setup, FlapOptions, NotificationOptions


COUNT_SUFFIX = '-count_summ'
TIMER_SUFFIX = '-timer_hgram'
default_flaps = FlapOptions(critical=300, stable=60)

GREEN = '#0a4'
ORANGE = '#e90'
LIGHT_BLUE = '#5ab'
BLUE = '#009'
RED = '#f00'
DARK_RED = '#cc0000'
colors = {
    'Create': GREEN,
    'Update': ORANGE,
    'Get': LIGHT_BLUE,
    'List': BLUE,
    'Remove': RED,
}

default_repeat_nots = NotificationOptions('on_status_change', {
    "status": "CRIT",
    'method': ["telegram"],
    'login': ['awacs-alerts'],
    'repeat': 3600
})

nodata_ok_aggregator_kwargs = {
    'nodata_mode': 'force_ok',
    'unreach_mode': 'force_ok',
    'unreach_service': [{'check': 'yasm_alert:virtual-meta'}],
}


def create_awacs_panel(ctype):
    name = ('' if ctype == 'prod' else ctype + '-') + 'awacs'
    panel = Panel(name, charts=[
        Graphic(signals=[
            Signal('resolver-backend-ctl-updated-endpoint-sets_summ', title='Updated endpoint sets')
        ]),
        Graphic(title='L3 ctl', width=1, signals=[
            Signal('worker-l3-ctl-l3mgr-config-created_summ', color=GREEN),
            Signal('worker-l3-ctl-l3mgr-config-skipped_summ', color=ORANGE),
            Signal('worker-l3-ctl-l3mgr-error_summ', color=RED),
            Signal('worker-l3-ctl-l3mgr-error-vs-not-same_summ', color=RED),
            Signal('worker-l3-ctl-l3mgr-rs-groups-conflict_summ', color=RED),
            Signal('worker-l3-ctl-l3mgr-unsupported-parameter_summ', color=RED),
        ]),
        Graphic(title='L7 ctl', width=1, signals=[
            Signal('sum(worker-l7-ctl-force-processed_summ,'
                   'worker_2-l7-ctl-force-processed_summ)', color=BLUE),
            Signal('sum(worker-l7-ctl-processed_summ,'
                   'worker_2-l7-ctl-processed_summ)', color=GREEN),
            Signal('sum(worker-l7-ctl-transport-nanny-error_summ,'
                   'worker_2-l7-ctl-transport-nanny-error_summ)', color=RED),
            Signal('sum(worker-l7-ctl-transport-unexpected-error_summ,'
                   'worker_2-l7-ctl-transport-unexpected-error_summ)', color=RED),
            Signal('sum(worker-l7-ctl-validator-infinite-loop_summ,'
                   'worker_2-l7-ctl-validator-infinite-loop_summ)', color=RED),
        ]),
        Graphic(title='Resolver backend ctl', signals=[
            Signal('resolver-backend-ctl-resolve-error_summ'),
            Signal('resolver-backend-ctl-resolve-success_summ'),
            Signal('resolver-backend-ctl-yp-endpoint-set-is-empty-error_summ'),
            Signal('resolver-backend-ctl-yp-endpoint-set-does-not-exist-error_summ'),
            Signal('resolver-backend-ctl-instances-list-is-empty-error_summ'),
        ]),
        Graphic(title='YP.SD vs YP.Master', width=1, signals=[
            Signal('resolver-backend-ctl-yp-endpoint-set-master-resolve_summ'),
            Signal('resolver-backend-ctl-yp-endpoint-set-sd-resolve_summ'),
        ]),
        Graphic(title='YP.SD resolver errors', width=1, signals=[
            Signal('sum(resolver-clients-yp-sd-resolver-total-error_summ,'
                   'resolver-clients-yp-sd-resolver-total-exception_summ)', color=DARK_RED),
            Signal('resolver-backend-ctl-yp-sd-obsolete-response-error_summ', color=RED),
            Signal('resolver-backend-ctl-yp-endpoint-set-does-not-exist-error_summ', color=ORANGE),
        ]),
        Graphic(title='Alerting namespace ctl', width=1, signals=[
            Signal('worker-ns-ctl-alerting-sync-succeeded_summ'),
            Signal('worker-ns-ctl-alerting-sync-failed_summ'),
            Signal('worker-ns-ctl-alerting-juggler-ns-created_summ'),
            Signal('worker-ns-ctl-alerting-yasm-alerts-created_summ'),
            Signal('worker-ns-ctl-alerting-orly-brake-awacs-namespace-alerting-sync-forbidden_summ'),
        ]),
        Graphic(title='In-progress configurations', width=1, signals=[
            Signal('api-awacs-cache-in-progress-balancers-counter_axxx',
                   title='In-progress L7', color=GREEN),
            Signal('api-awacs-cache-in-progress-l3-balancers-counter_axxx',
                   title='In-progress L3', color=BLUE),
        ]),
        Graphic(title='L7 config validation time', signals=[
            Signal('max(worker-l7-ctl-config-validation-timer_hgram)'),
        ]),
    ])

    orly_alerts = []

    l3_orly_signals = OrderedDict()
    for status in ('allowed', 'forbidden', 'unavailable', 'forbidden-by-transaction-error'):
        signal = Signal('worker-l3-ctl-orly-brake-awacs-push-l3-config-{status}_summ'.format(status=status),
                        title=status.capitalize())
        l3_orly_signals[status] = signal
    panel.append(Graphic(l3_orly_signals.values(), title='Orly L3'))
    orly_alerts.append(Alert('awacs-orly-l3-forbidden', l3_orly_signals['forbidden'],
                             title='Orly L3 forbidden', warn=[5, 8], crit=[8, None],
                             value_modify={'type': 'summ', 'window': 30}).with_juggler_check().update())
    orly_alerts.append(Alert('awacs-orly-l3-unavailable', l3_orly_signals['unavailable'],
                             title='Orly L3 unavailable', warn=[10, 15], crit=[15, None],
                             value_modify={'type': 'summ', 'window': 60}).with_juggler_check().update())

    l7_orly_signals = OrderedDict()
    for status in ('allowed', 'forbidden', 'unavailable', 'forbidden-by-transaction-error'):
        signal = Signal('sum('
                        'worker-l7-ctl-orly-brake-awacs-push-l7-config-{status}_summ,'
                        'worker_2-l7-ctl-orly-brake-awacs-push-l7-config-{status}_summ'
                        ')'.format(status=status),
                        title=status.capitalize())
        l7_orly_signals[status] = signal
    panel.append(Graphic(l7_orly_signals.values(), title='Orly L7'))
    orly_alerts.append(Alert('awacs-orly-l7-forbidden', l7_orly_signals['forbidden'],
                             title='Orly L7 forbidden', warn=[5, 8], crit=[8, None],
                             value_modify={'type': 'summ', 'window': 30}).with_juggler_check().update())
    orly_alerts.append(Alert('awacs-orly-l7-unavailable', l7_orly_signals['unavailable'],
                             title='Orly L7 unavailable', warn=[10, 15], crit=[15, None],
                             value_modify={'type': 'summ', 'window': 60}).with_juggler_check().update())

    panel.append(orly_alerts)
    panel.append([
        Alert(
            'awacs-l7-ctl-infinite-loops',
            Signal('sum(worker-l7-ctl-validator-infinite-loop_summ,'
                   'worker_2-l7-ctl-validator-infinite-loop_summ)'),
            warn=[1, 3],
            crit=[3, None],
            value_modify={'type': 'summ', 'window': 600},
            title='L7 ctl validator infinite loops'
        ).with_juggler_check().update(),
        Alert(
            'awacs-l7-ctl-transport-unexpected-errors',
            Signal('sum(worker-l7-ctl-transport-unexpected-error_summ,'
                   'worker_2-l7-ctl-transport-unexpected-error_summ)'),
            warn=[3, 6],
            crit=[6, None],
            value_modify={'type': 'summ', 'window': 20 * 60},
            title='L7 ctl transport unexpected errors'
        ).with_juggler_check().update(),
        Alert('awacsworkerd-common-created-l3mgr-configs'),
        Alert('awacsworkerd-common-l3-ctl-errors'),
        Alert(
            'awacs-updated-endpoint-sets',
            Signal('resolver-backend-ctl-updated-endpoint-sets_summ'),
            warn=[30, 40],
            crit=[40, None],
            value_modify={'type': 'summ', 'window': 60},
            title='Updated endpoint sets'
        ).with_juggler_check().update(),
        Alert(
            'awacs-backend-resolution-success-total',
            Signal('resolver-backend-ctl-resolve-success_summ'),
            warn=[22000, 32000] if ctype == 'prod' else [None, None],
            crit=[None, 22000] if ctype == 'prod' else [0, 0],
            value_modify={'type': 'summ', 'window': 60 * 5},
            title='Number of succeeded backend resolutions in 5 minutes'
        ).with_juggler_check(flaps_config=default_flaps).update(),
        Alert(
            'awacs-backend-resolution-error-total',
            Signal('resolver-backend-ctl-resolve-error_summ'),
            warn=[1400, 1600],
            crit=[1600, None],
            value_modify={'type': 'summ', 'window': 60 * 5},
            title='Number of failed backend resolutions in 5 minutes'
        ).with_juggler_check(flaps_config=default_flaps).update(),
        Alert(
            'awacs-yp-sd-obsolete-responses',
            Signal('resolver-backend-ctl-yp-sd-obsolete-response-error_summ'),
            warn=[30, 50],
            crit=[50, None],
            value_modify={'type': 'summ', 'window': 60 * 5},
            title='YP.SD obsolete responses'
        ).with_juggler_check(flaps_config=default_flaps).update(),
    ])
    panel.append([
        Alert('awacs-ns-ctl-alerting-sync-succeeded-total',
              Signal('worker-ns-ctl-alerting-sync-succeeded_summ'),
              warn=[10, 20], crit=[None, 10], title='Namespace Alerting Successful Iterations',
              value_modify={'type': 'summ', 'window': 60 * 60},
              ).with_juggler_check(flaps_config=default_flaps).update(),
        Alert('awacs-ns-ctl-alerting-sync-failed-total',
              Signal('worker-ns-ctl-alerting-sync-failed_summ'),
              warn=[60, 300], crit=[300, None], title='Namespace Alerting Failed Iterations',
              value_modify={'type': 'summ', 'window': 60 * 60},
              ).with_juggler_check(flaps_config=default_flaps).update(),
    ])
    panel.append([
        Alert('awacs-stuck-balancers-total',
              Signal('api-awacs-cache-stuck-balancers-counter_axxx'),
              warn=[None, None], crit=[15, None], title='Stuck In Progress L7 Configs'
              ).with_juggler_check(flaps_config=default_flaps, notifications=[default_repeat_nots]).update(),
        Alert('awacs-stuck-l3-balancers-total',
              Signal('api-awacs-cache-stuck-l3-balancers-counter_axxx'),
              warn=[None, None], crit=[3, None], title='Stuck In Progress L3 Configs'
              ).with_juggler_check(flaps_config=default_flaps).update(),
        Alert('awacs-stuck-namespaces-total',
              Signal('api-awacs-cache-stuck-namespaces-counter_axxx'),
              warn=[5, 10], crit=[10, None], title='Stuck Being Created Namespaces'
              ).with_juggler_check(flaps_config=default_flaps).update(),
        Alert('awacs-stuck-being-created-l3-balancers-total',
              Signal('api-awacs-cache-stuck-being-created-l3-balancers-counter_axxx'),
              warn=[3, 6], crit=[6, None], title='Stuck Being Created L3 Balancers'
              ).with_juggler_check(flaps_config=default_flaps).update(),
    ])
    panel.append([
        Alert(
            'awacs-in-flight-requests-total',
            Signal('api-capacity-limiter-in-flight-requests_ammx'),
            warn=[20, 40],
            crit=[40, None],
            title='In-flight requests'
        ).with_juggler_check().update(),
        Alert(
            'awacs-too-many-requests-total',
            Signal('api-capacity-limiter-too-many-requests_summ'),
            warn=[1, 1] if ctype == 'prod' else [None, None],
            crit=[1, None] if ctype == 'prod' else [500, None],
            title='Too many requests'
        ).with_juggler_check().update(),
        Alert(
            'awacs-l7-config-validation-too-long',
            Signal('max(worker-l7-ctl-config-validation-timer_hgram)'),
            warn=[180, 240] if ctype == 'prod' else [None, None],
            crit=[240, None] if ctype == 'prod' else [240, None],
            title='Too long L7 config validation'
        ).with_juggler_check(aggregator_kwargs=nodata_ok_aggregator_kwargs).update(),

    ])
    panel.append(
        Alert('awacs-yasm-quota', Signal('modules-unistat-limit_usage_perc_xxxx'), warn=[75, 90], crit=[90, None],
              title='Yasm quota %').with_juggler_check().update())
    panel.append(
        Alert('awacs-expires-soon-certs',
              Signal('api-awacs-cache-expires-soon-certs-counter_axxx'), warn=[None, None],
              crit=[1, None], title='Expiring soon certs count'
              ).with_juggler_check(notifications=[default_repeat_nots]).update()
    )
    panel.append(
        Alert('awacs-indiscoverable-too-long-certs',
              Signal('api-awacs-cache-indiscoverable-too-long-certs-counter_axxx'), warn=[None, None],
              crit=[1, None], title='Indiscoverable too long certs count').with_juggler_check(notifications=[default_repeat_nots]).update()
    )
    panel.append(
        Alert('awacs-namespaces-with-enabled-its-counter',
              Signal('api-awacs-cache-namespaces-with-enabled-its-counter_axxx'), warn=[900, 950],
              crit=[950, None], title='Namespaces with ITS').with_juggler_check(notifications=[default_repeat_nots]).update()
    )
    panel.upsert()


def create_awacs_aux_panel(panel_name, alerts_prefix, sources, kazoo_conn_loss_signals):
    def create_graphic(title, prefixes, suffix, methods, **kwargs):
        signals = []
        for method in methods:
            signal_name = 'sum({})'.format(', '.join([prefix + method + suffix for prefix in prefixes]))
            signals.append(Signal(signal_name, title=method))
        kwargs.setdefault('width', 2)
        return Graphic(signals, title=title, **kwargs)

    controls_updater_failures_signal = Signal(
        'sum({})'.format(','.join(p + '-controls-updater-failure_summ' for p in sources)),
        title='Failure')
    crashes_and_ooms_signal = Signal('sum('
                                     'hsum(portoinst-cores_total_hgram),'
                                     'hsum(portoinst-cores_dumped_hgram),'
                                     'hsum(portoinst-ooms_slot_hgram),'
                                     'portoinst-ooms_summ'
                                     ')')
    panel = Panel(panel_name, charts=[
        Graphic(title='Gevent monitor thread triggered (# per second)', width=2, normalize=True, signals=[
            Signal('*-gevent-monitor-triggered_summ'),
            Signal('const(5)'),
        ]),
        Graphic(title='Gevent monitor thread blocked (# per second)', width=2, normalize=True, signals=[
            Signal('*-gevent-event-loop-blocked_summ'),
        ]),
        Graphic(title='Kazoo # of requests waiting for the response', width=1, normalize=True, signals=[
            Signal('*-*-zookeeper-client-pending-queue-size_ammx'),
        ]),
        Graphic(title='Kazoo # of requests waiting to be sent', width=1, normalize=True, signals=[
            Signal('*-*-zookeeper-client-queue-size_ammx'),
        ]),
        Graphic(title='Kazoo bytes sent (# per second)', width=1, normalize=True, signals=[
            Signal('*-*-zookeeper-connection-bytes-sent_summ'),
        ]),
        Graphic(title='Kazoo bytes received (# per second)', width=1, normalize=True, signals=[
            Signal('*-*-zookeeper-connection-bytes-received_summ'),
        ]),
        Graphic(title='Received cache events (# per second)', width=1, normalize=True, signals=[
            Signal('*-cache-run-callbacks-calls_summ'),
        ]),
        Graphic(title='Called cache callbacks (# per second)', width=1, normalize=True, signals=[
            Signal('*-cache-callback-calls_summ'),
        ]),
        Graphic(title='Kazoo connection losses', width=1, signals=[
            Signal('*-*-zookeeper-kazoo-state-lost_summ'),
        ]),
        Graphic(title='Exclusive service lock release timeouts (SWAT-7046)', width=1, signals=[
            Signal('*-exclusive-services-lock-release-soft-timeouts_summ'),
            Signal('*-exclusive-services-lock-release-hard-timeouts_summ'),
        ]),
        Graphic(title='Controls updater', normalize=True, width=1, signals=[
            Signal('sum({})'.format(','.join(p + '-controls-updater-success_summ' for p in sources)),
                   title='Success'),
            controls_updater_failures_signal,
        ]),
        Graphic(title='CPU usage', normalize=True, signals=[
            Signal('havg(portoinst-cpu_usage_cores_hgram)'),
            Signal('quant(portoinst-cpu_usage_cores_hgram, max)'),
        ]),
        Graphic(title='Memory usage', signals=[
            Signal('portoinst-anon_usage_gb_tmmv'),
        ]),
        Graphic(title='Cores, crashes, OOMs', signals=[
            crashes_and_ooms_signal,
        ])
    ])

    panel.append(
        Alert(alerts_prefix + 'cpu', Signal('havg(portoinst-cpu_usage_cores_hgram)'),
              warn=[2.5, 3], crit=[3, None],
              title='CPU').with_juggler_check(flaps_config=default_flaps).update())
    panel.append(
        Alert(alerts_prefix + 'crashes', crashes_and_ooms_signal,
              warn=[1, 1],
              crit=[1, None],
              title='Cores, crashes, OOMs', value_modify={'type': 'summ', 'window': 360}).with_juggler_check().update())
    panel.append(
        Alert(alerts_prefix + 'page-faults', Signal('portoinst-major_page_faults_summ'),
              warn=[50, 100],
              crit=[100, None],
              title='Page faults', value_modify={'type': 'summ', 'window': 360}).with_juggler_check().update())
    panel.append(
        Alert(alerts_prefix + 'kazoo-conn-loss',
              Signal('sum({})'.format(', '.join(kazoo_conn_loss_signals))),
              warn=[2, 4],
              crit=[4, None],
              title='Number of kazoo connection losses in 1800 seconds',
              value_modify={'type': 'summ', 'window': 1800}).with_juggler_check().update()
    )

    Alert(
        alerts_prefix + 'controls-updater-failures',
        controls_updater_failures_signal,
        warn=[1, 3],
        crit=[3, None],
        title='Number of controls updater failures in 60 seconds',
        value_modify={'type': 'summ', 'window': 60}
    ).with_juggler_check().update()

    prefixes = [s + '-clients-' for s in sources]
    clients = ['yp-sd-resolver', 'l3mgr', 'abc', 'its', 'nanny', 'staff', 'passport',
               'ya-vault', 'httpgridfs', 'gencfg', 'certificator']
    panel.append(create_graphic('Client errors', prefixes, '-total-error_summ', clients, width=1))
    panel.append(create_graphic('Client exceptions', prefixes, '-total-exception_summ', clients, width=1))

    warn_limits = {
        'yp-sd-resolver': [60, 100],
    }
    for client in clients:
        warn = warn_limits.get(client, [6, 9])
        crit = [warn[-1], None]

        client_signals = []
        for prefix in prefixes:
            for metric in ('total-error_summ', 'total-exception_summ'):
                client_signals.append(prefix + client + '-' + metric)
        signal_name = 'sum(' + ','.join(client_signals) + ')'
        Alert(
            alerts_prefix + '{}-client-errors'.format(client),
            Signal(signal_name),
            warn=warn,
            crit=crit,
            title='Number of {} client errors in 10 minutes'.format(client),
            value_modify={'type': 'summ', 'window': 600}
        ).with_juggler_check().update()
    panel.upsert()


def create_awacs_api_panel(ctype='prod'):
    def create_graphic(title, prefix, suffix, methods, **kwargs):
        signals = []
        c = {}
        for req_type in colors:
            m = None
            for method in methods:
                if method.startswith(req_type):
                    if m is None or len(method) < len(m):
                        m = method
            if m:
                c[m] = colors[req_type]
        for method in methods:
            signal_name = prefix + method + suffix
            signals.append(Signal(signal_name, title=method.split('-')[-1], color=c.get(method)))
        kwargs.setdefault('width', 2)
        return Graphic(signals, title=title, **kwargs)

    name = ('' if ctype == 'prod' else ctype + '-') + 'awacs-api'
    panel = Panel(name)

    methods = ('CreateBackend', 'GetBackend', 'ListBackendRevisions', 'ListBackends', 'RemoveBackend', 'UpdateBackend')
    panel.append(create_graphic('Backend service', 'api-services-http-rpc-backend-service-', COUNT_SUFFIX, methods))

    methods = ('CreateBalancer', 'GetBalancer', 'GetBalancerAspectsSet', 'GetBalancerRevision',
               'GetQuickStartBalancersUnion', 'ListBalancerAspectsSets', 'ListBalancerRevisions', 'ListBalancers',
               'RemoveBalancer', 'UpdateBalancer', 'UpdateQuickStartBalancersUnion')
    panel.append(create_graphic('Balancer service', 'api-services-http-rpc-balancer-service-', COUNT_SUFFIX, methods))

    methods = ('CreateCertificate', 'GetCertificate', 'GetCertificateRenewal', 'ListCertificateRenewals',
               'ListCertificateRevisions', 'ListCertificates', 'OrderCertificate', 'RemoveCertificate')
    panel.append(
        create_graphic('Certificate service', 'api-services-http-rpc-certificate-service-', COUNT_SUFFIX, methods))

    methods = ('DraftComponent', 'PublishComponent', 'RetireComponent', 'GetComponent',
               'ListComponents', 'RemoveComponent')
    panel.append(create_graphic('Component service', 'api-services-http-rpc-component-service-', COUNT_SUFFIX, methods))

    methods = ('CreateDomain', 'CreateDomainOperation', 'ListDomains', 'GetDomain', 'GetDomain', 'ListDomainOperations',
               'GetDomainOperation', 'RemoveDomain', 'CancelDomainOrder', 'CancelDomainOperation', 'GetDomainRevision',
               'ListDomainRevisions')
    panel.append(create_graphic('Domain service', 'api-services-http-rpc-domain-service-', COUNT_SUFFIX, methods))

    methods = ('GetDnsRecord', 'GetNameServerConfig', 'ListDnsRecordRevisions', 'ListDnsRecords', 'ListNameServers',
               'RemoveDnsRecord')
    panel.append(
        create_graphic('DNS record service', 'api-services-http-rpc-dns-record-service-', COUNT_SUFFIX, methods))

    methods = ('CreateEndpointSet', 'GetEndpointSet', 'GetEndpointSetRevision', 'ListEndpointSetRevisions')
    panel.append(
        create_graphic('Endpoint service', 'api-services-http-rpc-endpoint-set-service-', COUNT_SUFFIX, methods))

    methods = ('CreateL3Balancer', 'GetL3Balancer', 'ListL3BalancerRevisions', 'ListL3Balancers', 'UpdateL3Balancer')
    panel.append(
        create_graphic('L3 balancer service', 'api-services-http-rpc-l3-balancer-service-', COUNT_SUFFIX, methods))

    methods = ('CreateNamespace', 'GetNamespace', 'GetNamespaceAspectsSet', 'ListNamespaces', 'RemoveNamespace',
               'UpdateNamespace', 'UpdateNamespaceOrderContext')
    panel.append(create_graphic('Namespace service', 'api-services-http-rpc-namespace-service-', COUNT_SUFFIX, methods))

    methods = ('CreateUpstream', 'GetUpstream', 'GetUpstreamRevision', 'LintUpstream', 'ListUpstreamRevisions',
               'ListUpstreams', 'RemoveUpstream', 'UpdateUpstream')
    panel.append(create_graphic('Upstream service', 'api-services-http-rpc-upstream-service-', COUNT_SUFFIX, methods))

    methods = ('backend-service-CreateBackend', 'balancer-service-CreateBalancer',
               'balancer-service-GetQuickStartBalancersUnion', 'balancer-service-UpdateBalancer',
               'certificate-service-CreateCertificate', 'certificate-service-OrderCertificate',
               'l3-balancer-service-CreateL3Balancer', 'l3-balancer-service-UpdateL3Balancer',
               'namespace-service-CreateNamespace', 'namespace-service-RemoveNamespace',
               'upstream-service-CreateUpstream', 'upstream-service-LintUpstream', 'upstream-service-UpdateUpstream')
    panel.append(create_graphic('4xx', 'api-services-http-statuses-4xx-rpc-', COUNT_SUFFIX, methods, width=1))

    methods = ('backend-service-CetBackend', 'balancer-service-GetQuickStartBalancersUnion',
               'certificate-service-GetCertificateRenewal', 'dns-record-service-GetDnsRecord',
               'namespace-service-CetNamespace', 'namespace-service-GetNamespaceAspectsSet',
               'upstream-service-GetUpstream')
    panel.append(create_graphic('404', 'api-services-http-statuses-404-rpc-', COUNT_SUFFIX, methods, width=1))

    methods = (
        'backend-service-ListBackends', 'balancer-service-ListBalancers', 'dns-record-service-GetNameServerConfig',
        'namespace-service-ListNamespaces', 'upstream-service-ListUpstreams')
    panel.append(create_graphic('429', 'api-services-http-statuses-429-rpc-', COUNT_SUFFIX, methods, width=1))

    methods = ('5xx', '4xx', '404', '429')
    panel.append(create_graphic('Total 5xx, 4xx, 404, 429', 'api-services-http-statuses-', '-total-count_summ', methods,
                                width=1))

    signals = []
    for quant in ('20', '40', '60', '80', '99'):
        signals.append(Signal('quant(api-services-http-total-timer_hgram, {})'.format(quant), title=quant))
    panel.append(Graphic(signals, title='API timer total'))

    methods = ('upstream-service-RemoveUpstream', 'namespace-service-RemoveNamespace', 'backend-service-RemoveBackend',
               'balancer-service-RemoveBalancer', 'certificate-service-RemoveCertificate',
               'domain-service-RemoveDomain',
               'dns-record-service-RemoveDnsRecord', 'l3-balancer-service-RemoveL3Balancer', 'knob-service-RemoveKnob',
               'component-service-RemoveComponent', 'component-service-RetireComponent', 'knob-service-RemoveKnobValue')
    panel.append(create_graphic('Desctructive calls', 'api-services-http-rpc-', COUNT_SUFFIX, methods, stacked=True))

    panel.upsert()


def create_awacs_api_methods_panel(ctype='prod'):
    name = ('' if ctype == 'prod' else ctype + '-') + 'awacs-api-methods'
    panel = Panel(name)

    def create_graphic(title, prefix, suffix, methods, **kwargs):
        kwargs.setdefault('width', 2)
        signals = []
        for method in methods:
            signal_name = prefix + method + suffix
            title = method.split('-')[-1]
            for quant in ('20', '40', '60', '80', '99'):
                signals.append(Signal('quant({}, {})'.format(signal_name, quant), title=quant, continuous=True))
            panel.append(Graphic(signals, title=title, **kwargs))
            signals = []
        if len(methods) % 2:
            panel.append(Text(width=2))

    panel.append(Text('Backends'))
    methods = ('CreateBackend', 'GetBackend', 'ListBackendRevisions', 'ListBackends', 'RemoveBackend', 'UpdateBackend')
    create_graphic('Backend service', 'api-services-http-rpc-backend-service-', TIMER_SUFFIX, methods)

    panel.append(Text('Balancers'))
    methods = ('CreateBalancer', 'GetBalancer', 'GetBalancerAspectsSet', 'GetBalancerRevision',
               'GetQuickStartBalancersUnion', 'ListBalancerAspectsSets', 'ListBalancerRevisions', 'ListBalancers',
               'RemoveBalancer', 'UpdateBalancer', 'UpdateQuickStartBalancersUnion')
    create_graphic('Balancer service', 'api-services-http-rpc-balancer-service-', TIMER_SUFFIX, methods)

    panel.append(Text('Certificates'))
    methods = ('CreateCertificate', 'GetCertificate', 'GetCertificateRenewal', 'ListCertificateRenewals',
               'ListCertificateRevisions', 'ListCertificates', 'OrderCertificate', 'RemoveCertificate')
    create_graphic('Certificate service', 'api-services-http-rpc-certificate-service-', TIMER_SUFFIX, methods)

    panel.append(Text('DNS Records'))
    methods = ('GetDnsRecord', 'GetNameServerConfig', 'ListDnsRecordRevisions', 'ListDnsRecords', 'ListNameServers',
               'RemoveDnsRecord')
    create_graphic('DNS record service', 'api-services-http-rpc-dns-record-service-', TIMER_SUFFIX, methods)

    panel.append(Text('Endpoint Sets'))
    methods = ('CreateEndpointSet', 'GetEndpointSet', 'GetEndpointSetRevision', 'ListEndpointSetRevisions')
    create_graphic('Endpoint service', 'api-services-http-rpc-endpoint-set-service-', TIMER_SUFFIX, methods)

    panel.append(Text('L3 Balancers'))
    methods = ('CreateL3Balancer', 'GetL3Balancer', 'ListL3BalancerRevisions', 'ListL3Balancers', 'UpdateL3Balancer')
    create_graphic('L3 balancer service', 'api-services-http-rpc-l3-balancer-service-', TIMER_SUFFIX, methods)

    panel.append(Text('Namespaces'))
    methods = ('CreateNamespace', 'GetNamespace', 'GetNamespaceAspectsSet', 'ListNamespaces', 'RemoveNamespace',
               'UpdateNamespace', 'UpdateNamespaceOrderContext')
    create_graphic('Namespace service', 'api-services-http-rpc-namespace-service-', TIMER_SUFFIX, methods)

    panel.append(Text('Upstreams'))
    methods = ('CreateUpstream', 'GetUpstream', 'GetUpstreamRevision', 'LintUpstream', 'ListUpstreamRevisions',
               'ListUpstreams', 'RemoveUpstream', 'UpdateUpstream')
    create_graphic('Upstream service', 'api-services-http-rpc-upstream-service-', TIMER_SUFFIX, methods)

    panel.upsert()


def create_awacsstatusd_panel(ctype='prod'):
    name = ('' if ctype == 'prod' else ctype + '-') + 'awacsstatusd'
    panel = Panel(name, charts=[
        Graphic(title='Gevent monitor thread triggered (# per second)', width=2, normalize=True, signals=[
            Signal('unistat-gevent-monitor-triggered_summ'),
            Signal('const(20)'),
        ]),
        Graphic(title='Gevent monitor thread blocked (# per second)', width=2, normalize=True, signals=[
            Signal('unistat-gevent-event-loop-blocked_summ'),
        ]),
        Graphic(title='Kazoo # of requests waiting for the response', width=1, normalize=True, signals=[
            Signal('unistat-awacsstatusd-zookeeper-client-pending-queue-size_ammx'),
        ]),
        Graphic(title='Kazoo # of requests waiting to be sent', width=1, normalize=True, signals=[
            Signal('unistat-awacsstatusd-zookeeper-client-queue-size_ammx'),
        ]),
        Graphic(title='Kazoo bytes sent (# per second)', width=1, normalize=True, signals=[
            Signal('unistat-awacsstatusd-zookeeper-connection-bytes-sent_summ'),
        ]),
        Graphic(title='Kazoo bytes received (# per second)', width=1, normalize=True, signals=[
            Signal('unistat-awacsstatusd-zookeeper-connection-bytes-received_summ'),
        ]),
        Graphic(title='Kazoo connection losses', width=1, signals=[
            Signal('unistat-awacsstatusd-zookeeper-kazoo-state-lost_summ'),
        ]),
        Graphic(title='Controls updater', normalize=True, width=1, signals=[
            Signal('unistat-controls-updater-success_summ', title='Success'),
            Signal('unistat-controls-updater-failure_summ', title='Failure'),
        ]),
        Graphic(title='CPU usage', normalize=True, signals=[
            Signal('havg(portoinst-cpu_usage_cores_hgram)'),
            Signal('quant(portoinst-cpu_usage_cores_hgram, max)'),
        ]),
        Graphic(title='Memory usage', signals=[
            Signal('portoinst-anon_usage_gb_tmmv'),
            Signal('portoinst-ooms_summ'),
        ])
    ])
    panel.upsert()

    Alert(
        'awacsstatusd-controls-updater-failures',
        Signal('unistat-controls-updater-failure_summ', title='Failure'),
        warn=[1, 3],
        crit=[3, None],
        title='Number of controls updater failures in 60 seconds',
        value_modify={'type': 'summ', 'window': 60}
    ).with_juggler_check().update()


def create_awacscrond_panel(ctype='prod'):
    name = ('' if ctype == 'prod' else ctype + '-') + 'awacscrond'
    panel = Panel(name, charts=[
        Graphic(title='IDM roles', width=2, signals=[
            Signal('unistat-cron-webauth-syncer-add-role_summ', color=GREEN),
            Signal('unistat-cron-webauth-syncer-remove-role_summ', color=RED),
            Signal('unistat-cron-webauth-syncer-update-role_summ', color=ORANGE),
        ]),
        Graphic(title='IDM syncer iterations', width=2, normalize=True, signals=[
            Signal('unistat-cron-webauth-syncer-success-iterations-count_summ', color=GREEN),
            Signal('unistat-cron-webauth-syncer-iterations-count_summ', color=BLUE),
        ]),
        Graphic(title='Cron iteration duration', width=2, normalize=True, signals=[
            Signal('div(unistat-cron-iteration-timer_axxx, 60)', color=BLUE),
        ]),
        Graphic(title='CPU usage', normalize=True, signals=[
            Signal('havg(portoinst-cpu_usage_cores_hgram)'),
            Signal('quant(portoinst-cpu_usage_cores_hgram, max)'),
        ]),
        Graphic(title='Memory usage', signals=[
            Signal('portoinst-anon_usage_gb_tmmv'),
        ]),
    ])
    panel.append([
        Alert('unistat-cron-iteration-timer_axxx'.format(name),
              Signal('div(unistat-cron-iteration-timer_axxx, 60)'),
              warn=[None, None], crit=[3*60, None], title='Cron iteration duration in minutes',
              ).with_juggler_check(flaps_config=default_flaps).update(),
    ])
    panel.append([
        Alert('unistat-cron-dismissed-cleaner-clean-nses-count_summ'.format(name),
              Signal('unistat-cron-dismissed-cleaner-clean-nses-count_summ'),
              warn=[None, None], crit=[15, None], title='Dismissed workers to clean',
              value_modify={'type': 'summ', 'window': 600},
              ).with_juggler_check(flaps_config=default_flaps).update(),
    ])
    panel.upsert()


def create_awacsworkerd_panel(panel_name):
    def create_graphic(title, prefixes, suffix, methods, **kwargs):
        signals = []
        for method in methods:
            signal_name = 'sum({})'.format(', '.join([prefix + method + suffix for prefix in prefixes]))
            signals.append(Signal(signal_name, title=method))
        kwargs.setdefault('width', 2)
        return Graphic(signals, title=title, **kwargs)

    panel = Panel(panel_name, charts=[
        Graphic(title='L3 ctl', width=1, signals=[
            Signal('worker-l3-ctl-l3mgr-config-created_summ', color=GREEN),
            Signal('worker-l3-ctl-l3mgr-config-skipped_summ', color=ORANGE),
            Signal('worker-l3-ctl-l3mgr-error_summ', color=RED),
            Signal('worker-l3-ctl-l3mgr-error-vs-not-same_summ', color=RED),
            Signal('worker-l3-ctl-l3mgr-rs-groups-conflict_summ', color=RED),
            Signal('worker-l3-ctl-l3mgr-unsupported-parameter_summ', color=RED),
        ]),
        Graphic(title='L7 ctl', width=1, signals=[
            Signal('worker-l7-ctl-force-processed_summ', color=BLUE),
            Signal('worker-l7-ctl-processed_summ', color=GREEN),
            Signal('worker-l7-ctl-transport-nanny-error_summ', color=RED),
            Signal('worker-l7-ctl-transport-unexpected-error_summ', color=RED),
            Signal('worker-l7-ctl-validator-infinite-loop_summ', color=RED),
        ]),
        Graphic(title='Alerting namespace ctl', width=1, signals=[
            Signal('worker-ns-ctl-alerting-sync-succeeded_summ'),
            Signal('worker-ns-ctl-alerting-sync-failed_summ'),
            Signal('worker-ns-ctl-alerting-juggler-ns-created_summ'),
            Signal('worker-ns-ctl-alerting-yasm-alerts-created_summ'),
            Signal('worker-ns-ctl-alerting-orly-brake-awacs-namespace-alerting-sync-forbidden_summ'),
        ]),
        Graphic(title='ITS namespace ctl', width=1, signals=[
            Signal('worker-ns-ctl-its-update-section_summ', color=GREEN),
            Signal('worker-ns-ctl-its-delete-section_summ', color=ORANGE),
            Signal('worker-ns-ctl-its-sync-succeeded_summ', color=LIGHT_BLUE),
            Signal('worker-ns-ctl-its-sync-failed_summ', color=DARK_RED),
            Signal('worker-ns-ctl-its-orly-brake-awacs-namespace-its-sync-forbidden_summ', color=RED),
        ]),
    ])

    for btype in ('l3', 'l7'):
        signals = OrderedDict()
        for status in ('allowed', 'forbidden', 'unavailable', 'forbidden-by-transaction-error'):
            signal = Signal('worker-{btype}-ctl-orly-brake-awacs-push-{btype}-config-{status}_summ'.format(
                btype=btype, status=status),
                title=status.capitalize())
            signals[status] = signal
        panel.append(Graphic(signals.values(), title='Orly {}'.format(btype), width=1))

    prefixes = ['worker-clients-']
    methods = ['yp-sd-resolver', 'l3mgr', 'abc', 'its', 'nanny', 'staff', 'passport',
               'ya-vault', 'httpgridfs', 'gencfg', 'certificator']
    panel.append(create_graphic('Client errors', prefixes, '-total-error_summ', methods, width=1))
    panel.append(create_graphic('Client exceptions', prefixes, '-total-exception_summ', methods, width=1))

    panel.append([
        Alert(
            '{}-l7-ctl-infinite-loops'.format(panel_name),
            Signal('worker-l7-ctl-validator-infinite-loop_summ'),
            warn=[1, 3],
            crit=[3, None],
            value_modify={'type': 'summ', 'window': 600},
            title='L7 ctl validator infinite loops'
        ).with_juggler_check().update(),
        Alert(
            '{}-created-l3mgr-configs'.format(panel_name),
            Signal('worker-l3-ctl-l3mgr-config-created_summ'),
            warn=[25, 35],
            crit=[35, None],
            value_modify={'type': 'summ', 'window': 180},
            title='Created l3mgr configs'
        ).with_juggler_check().update(),
        Alert(
            '{}-l3-ctl-errors'.format(panel_name),
            Signal('sum('
                   'worker-l3-ctl-l3mgr-error_summ,'
                   'worker-l3-ctl-l3mgr-error-vs-not-same_summ,'
                   'worker-l3-ctl-l3mgr-rs-groups-conflict_summ,'
                   'worker-l3-ctl-l3mgr-unsupported-parameter_summ'
                   ')'),
            warn=[3, 5],
            crit=[5, None],
            value_modify={'type': 'summ', 'window': 600},
            title='L3 ctl errors in 10 minutes'
        ).with_juggler_check().update(),
    ])
    panel.append(Alert('{}-cpu'.format(panel_name),
                       Signal('havg(portoinst-cpu_usage_cores_hgram)'),
                       warn=[1.5, 2], crit=[2, None],
                       title='CPU').with_juggler_check(flaps_config=default_flaps).update())
    panel.append([
        Alert('{}-ns-ctl-its-sync-succeeded-total'.format(panel_name),
              Signal('worker-ns-ctl-its-sync-succeeded_summ'),
              warn=[10, 20], crit=[None, 10], title='Namespace ITS Successful Iterations',
              value_modify={'type': 'summ', 'window': 60 * 60},
              ).with_juggler_check(flaps_config=default_flaps).update(),
        Alert('{}-ns-ctl-its-sync-failed-total'.format(panel_name),
              Signal('worker-ns-ctl-its-sync-failed_summ'),
              warn=[60, 300], crit=[300, None], title='Namespace ITS Failed Iterations',
              value_modify={'type': 'summ', 'window': 60 * 60},
              ).with_juggler_check(flaps_config=default_flaps).update(),
        Alert('{}-ns-ctl-its-update-section-total'.format(panel_name),
              Signal('worker-ns-ctl-its-update-section_summ'),
              warn=[30, 100], crit=[100, None], title='Namespace ITS Updated Sections',
              value_modify={'type': 'summ', 'window': 60 * 60},
              ).with_juggler_check(flaps_config=default_flaps).update(),
        Alert('{}-ns-ctl-its-delete-section-total'.format(panel_name),
              Signal('worker-ns-ctl-its-delete-section_summ'),
              warn=[30, 100], crit=[100, None], title='Namespace ITS Deleted Sections',
              value_modify={'type': 'summ', 'window': 60 * 60},
              ).with_juggler_check(flaps_config=default_flaps).update(),
    ])
    panel.append([
        Alert('{}-ns-ctl-alerting-sync-succeeded-total'.format(panel_name),
              Signal('worker-ns-ctl-alerting-sync-succeeded_summ'),
              warn=[10, 20], crit=[None, 10], title='Namespace Alerting Successful Iterations',
              value_modify={'type': 'summ', 'window': 60 * 60},
              ).with_juggler_check(flaps_config=default_flaps).update(),
        Alert('{}-ns-ctl-alerting-sync-failed-total'.format(panel_name),
              Signal('worker-ns-ctl-alerting-sync-failed_summ'),
              warn=[60, 300], crit=[300, None], title='Namespace Alerting Failed Iterations',
              value_modify={'type': 'summ', 'window': 60 * 60},
              ).with_juggler_check(flaps_config=default_flaps).update(),
    ])
    panel.upsert()


PANELS = [
    'awacs', 'awacs-aux', 'awacs-api', 'awacs-api-methods', 'awacsstatusd', 'awacscrond',
    'dev-awacs', 'dev-awacs-aux', 'dev-awacs-api', 'dev-awacs-api-methods', 'dev-awacsstatusd', 'dev-awacscrond',
    'awacsworkerd-common', 'awacsworkerd-common-aux', 'awacs-resources'
]


def list_alerts(itype):
    url = YASM_API_URL + 'alerts/list'
    limit = 10  # maximum allowed by yasm
    offset = 0
    while 1:
        params = {
            'itype': itype,
            'offset': offset,
            'limit': limit,
            'with_checks': 1,
        }
        r = requests.get(url, params=params)
        r.raise_for_status()
        data = r.json()
        response = data['response']
        if not response['result']:
            break
        for item in response['result']:
            yield item
        offset += 10


def delete_alert(name):
    url = YASM_API_URL + 'alerts/delete'
    params = {
        'name': name
    }
    r = requests.post(url, json=params)
    if r.status_code == 404:
        return False
    r.raise_for_status()
    return True


def delete_juggler_check(juggler_token, host, service):
    url = 'https://juggler.yandex-team.ru/api/checks/remove_check?do=1&host_name={}&service_name={}'.format(host, service)
    r = requests.get(url, headers={
        'Authorization': 'OAuth {}'.format(juggler_token),
    })
    if r.status_code == 404:
        return False
    r.raise_for_status()
    return True


@click.command()
@click.option('--no-alerts', '-n', is_flag=True)
@click.option('--panel', '-p', multiple=True, type=click.Choice(PANELS))
@click.option('--clean', '-c', is_flag=True, default=False)
@click.option('--juggler-token', '-jt', default=None,
              help='https://oauth.yandex-team.ru/authorize?response_type=token&client_id=cd178dcdc31a4ed79f42467f2d89b0d0')
def main(no_alerts, panel, clean, juggler_token):
    CURRENT_ALERT_DESCRIPTION = '110522'

    if clean:
        alerts_to_be_removed = set()
        juggler_checks_to_be_removed = set()
        for a in list_alerts(itype='awacs'):
            if a.get('description') != CURRENT_ALERT_DESCRIPTION:
                alerts_to_be_removed.add(a['name'])
                jc = a['juggler_check']
                juggler_checks_to_be_removed.add((jc['host'], jc['service']))
        print 'Juggler checks to be removed:'
        for host, service in sorted(juggler_checks_to_be_removed):
            print '*', host, ':', service
        print 'Alerts to be removed:'
        for alert in sorted(alerts_to_be_removed):
            print '*', alert
        if click.confirm('Please confirm'):
            print
            for host, service in sorted(juggler_checks_to_be_removed):
                if delete_juggler_check(juggler_token, host, service):
                    print 'Juggler check {}:{} deleted'.format(host, service)
                else:
                    print 'Juggler check {}:{} not found'.format(host, service)
            for alert in sorted(alerts_to_be_removed):
                if delete_alert(alert):
                    print 'YASM alert {} deleted'.format(alert)
                else:
                    print 'YASM alert {} not found'.format(alert)
        return

    def update_panel(panel_name):
        assert panel_name in PANELS
        if not panel:
            return True
        return panel_name in panel

    # Common setup
    setup(
        juggler_namespace='awacs',
        disable_alerts_updating=no_alerts,
        user='nanny-robot',
        editors=['alonger', 'disafonov', 'ferenets', 'pirogov', 'romanovich'],
        alert_description=CURRENT_ALERT_DESCRIPTION
    )

    setup(juggler_hostname='awacs_production', default_tag='itype=awacs;ctype=prod')
    if update_panel('awacs'):
        create_awacs_panel(ctype='prod')
    if update_panel('awacs-aux'):
        create_awacs_aux_panel(panel_name='awacs-aux',
                               alerts_prefix='awacs-',
                               sources=('api', 'worker', 'worker_2', 'resolver'),
                               kazoo_conn_loss_signals=[
                                   'api-awacsd-zookeeper-kazoo-state-lost_summ',
                                   'resolver-awacsresolverd-zookeeper-kazoo-state-lost_summ',
                                   'worker-awacsworkerd-zookeeper-kazoo-state-lost_summ',
                                   'worker_2-awacsworkerd-zookeeper-kazoo-state-lost_summ',
                               ])
    if update_panel('awacs-api'):
        create_awacs_api_panel()
    if update_panel('awacs-api-methods'):
        create_awacs_api_methods_panel()
    if update_panel('awacs-resources'):
        create_awacs_resources_panel()
    setup(default_tag='itype=awacscrond;ctype=prod;prj=awacs')
    if update_panel('awacscrond'):
        create_awacscrond_panel()

    setup(juggler_hostname='awacsstatusd_production', default_tag='itype=awacsstatusd;ctype=prod;prj=awacs')
    if update_panel('awacsstatusd'):
        create_awacsstatusd_panel()

    setup(juggler_hostname='awacs_production',
          default_tag='itype=awacs;ctype=prod')
    if update_panel('awacsworkerd-common'):
        create_awacsworkerd_panel('awacsworkerd-common')

    # Prestable
    setup(alert_suffix='-prestable')

    setup(juggler_hostname='awacsstatusd_prestable', default_tag='itype=awacsstatusd;ctype=prestable;prj=awacs')
    if update_panel('dev-awacsstatusd'):
        create_awacsstatusd_panel(ctype='dev')

    setup(juggler_hostname='awacscrond_prestable', default_tag='itype=awacscrond;ctype=prestable;prj=awacs')
    if update_panel('dev-awacscrond'):
        create_awacscrond_panel(ctype='dev')

    setup(juggler_hostname='awacs_prestable', default_tag='itype=awacs;ctype=prestable')
    if update_panel('dev-awacs'):
        create_awacs_panel(ctype='dev')
    if update_panel('dev-awacs-aux'):
        create_awacs_aux_panel(panel_name='dev-awacs-aux',
                               alerts_prefix='awacs-',  # we rely on `setup(alert_suffix='-prestable')` :shrug:
                               sources=('api', 'worker', 'worker_2', 'resolver'),
                               kazoo_conn_loss_signals=[
                                   'api-awacsd-zookeeper-kazoo-state-lost_summ',
                                   'resolver-awacsresolverd-zookeeper-kazoo-state-lost_summ',
                                   'worker-awacsworkerd-zookeeper-kazoo-state-lost_summ',
                                   'worker_2-awacsworkerd-zookeeper-kazoo-state-lost_summ',
                               ])
    if update_panel('dev-awacs-api'):
        create_awacs_api_panel(ctype='dev')
    if update_panel('dev-awacs-api-methods'):
        create_awacs_api_methods_panel(ctype='dev')


def create_awacs_resources_panel():
    panel_name = 'awacs-resources'
    itype_awacs_s_ids = (
        'production_awacs_api',
        'production_awacs_resolver',
        'production_awacs_worker_restricted_to_man',
        'production_awacs_worker_restricted_to_msk',
        'production_awacs_worker_restricted_to_sas',
        'production_awacs_worker_restricted_to_vla',
        'production_awacs_worker_l3',
        'production_awacs_worker_common',
    )
    tag_by_s_id = {
        s_id: 'itype=awacs;ctype=prod;prj=awacs;nanny={}'.format(s_id) for s_id in itype_awacs_s_ids
    }
    tag_by_s_id['production_awacsstatusd_sas'] = 'itype=awacsstatusd;ctype=prod;prj=awacs;nanny=production_awacsstatusd_sas'
    tag_by_s_id['production_awacsstatusd_vla'] = 'itype=awacsstatusd;ctype=prod;prj=awacs;nanny=production_awacsstatusd_vla'
    tag_by_s_id['production_awacs_cron'] = 'itype=awacscrond;ctype=prod;prj=awacs'

    s_ids = sorted(tag_by_s_id.keys())
    multi_python_process_s_ids = (
        'production_awacs_worker_restricted_to_man',
        'production_awacs_worker_restricted_to_msk',
        'production_awacs_worker_restricted_to_sas',
        'production_awacs_worker_restricted_to_vla',
    )
    single_python_process_s_ids = [s_id for s_id in s_ids if s_id not in multi_python_process_s_ids]

    const_100 = Signal('const(100)', title='100%', color='black')
    const_1 = Signal('const(1)', title='1 CPU', color='black')

    def get_memory_usage_max_perc_signal(s_id):
        return Signal('quant(portoinst-anon_limit_usage_perc_hgram,max)', title=s_id, tag=tag_by_s_id[s_id])

    def get_logs_usage_max_perc_signal(s_id):
        return Signal('portoinst-volume_/logs_usage_perc_txxx', title=s_id, tag=tag_by_s_id[s_id])

    panel = Panel(panel_name, charts=[
        Graphic(title='Memory usage max (% limit)', width=1, signals=[
            get_memory_usage_max_perc_signal(s_id) for s_id in s_ids
        ] + [const_100]),
        Graphic(title='Major page faults', width=1, normalize=True, signals=[
            Signal('portoinst-major_page_faults_summ', title=s_id, tag=tag_by_s_id[s_id])
            for s_id in s_ids
        ]),
        Graphic(title='CPU usage max (% limit)', width=1, signals=[
            Signal('quant(portoinst-cpu_limit_usage_perc_hgram,max)', title=s_id,
                   tag=tag_by_s_id[s_id])
            for s_id in s_ids
        ] + [const_100]),
        Graphic(name='cpu-usage-single', title='CPU usage max among single-process services (cores)', width=1, signals=[
            Signal(
                'portoinst-cpu_usage_cores_txxx',
                title=s_id,
                tag=tag_by_s_id[s_id])
            for s_id in
            single_python_process_s_ids
        ] + [const_1]),
        Graphic(name='cpu-usage-multi', title='CPU usage max among multi-process services (cores)', width=1, signals=[
            Signal('portoinst-cpu_usage_cores_txxx', title=s_id, tag=tag_by_s_id[s_id])
            for s_id in multi_python_process_s_ids
        ]),
        Graphic(title='CPU wait max', width=1, signals=[
            Signal('quant(portoinst-cpu_wait_slot_hgram,max)', title=s_id, tag=tag_by_s_id[s_id])
            for s_id in s_ids
        ]),
        Graphic(title='/logs usage (% limit)', width=1, signals=[
            get_logs_usage_max_perc_signal(s_id) for s_id in s_ids
        ] + [const_100]),
        Graphic(title='cwd usage (% limit)', width=1, signals=[
            Signal('portoinst-volume_cwd_usage_perc_txxx', title=s_id, tag=tag_by_s_id[s_id])
            for s_id in s_ids
        ] + [const_100]),
        Graphic(title='Cores, crashes, OOMsx', width=1, signals=[
            Signal('sum(hsum(portoinst-cores_total_hgram),'
                   'hsum(portoinst-cores_dumped_hgram),'
                   'hsum(portoinst-ooms_slot_hgram),'
                   'portoinst-ooms_summ)',
                   title=s_id, tag=tag_by_s_id[s_id])
            for s_id in s_ids
        ]),
    ])
    panel.append(Text(text='Memory', width=1))
    panel.append([
        Alert(
            '{}-{}-memory-usage-max'.format(panel_name, s_id),
            title='{}\nmemory usage max (% limit)'.format(s_id),
            signal=get_memory_usage_max_perc_signal(s_id),
            warn=[80, 90],
            crit=[90, None],
        ).with_juggler_check(notifications=[default_repeat_nots]).update()
        for s_id in s_ids
    ])
    panel.append(Text(text='/logs', width=1))
    panel.append([
        Alert(
            '{}-{}-logs-usage-max'.format(panel_name, s_id),
            title='{}\nlogs usage max (% limit)'.format(s_id),
            signal=get_logs_usage_max_perc_signal(s_id),
            warn=[80, 90],
            crit=[90, None],
        ).with_juggler_check(notifications=[default_repeat_nots]).update()
        for s_id in s_ids
    ])
    panel.upsert()


if __name__ == '__main__':
    main()
