# coding: utf-8
import os
from collections import Counter

import click
import jinja2
import jsonschema
import startrek_client
import yaml
import six


TASKS_DIR = './tasks/'

DEFAULT_COMMENT = (
    u'Пожалуйста, обратите внимание на этот тикет.\n\n'
    u'По возникшим вопросам и проблемам можно обращаться в очередь '
    u'((https://st.yandex-team.ru/createTicket?queue=BALANCERSUPPORT BALANCERSUPPORT))'
)


def create_issue(st, summary, description, queue, author, followers,
                 assignee=None, invitee=None, tags=(), parent_issue_id=None, comment=None):
    kwargs = dict(
        queue=queue,
        author=author,
        followers=followers,
        tags=tags,
        summary=summary,
        description=description,
    )
    comment = comment or DEFAULT_COMMENT
    if assignee:
        kwargs['assignee'] = assignee
    issue = st.issues.create(**kwargs)
    if parent_issue_id is not None:
        issue.links.create(relationship='is subtask for', issue=parent_issue_id)
    if invitee is not None:
        issue.comments.create(
            text=comment,
            summonees=sorted(invitee))
    return issue


INPUT_SCHEMA = {
    'type': 'object',
    'properties': {
        'meta': {
            'type': 'object',
            'properties': {
                'queue': {'type': 'string'},
                'parent_issue_id': {'type': 'string'},
                'author': {'type': 'string'},
                'tags': {'type': 'array', 'items': {'type': 'string'}},
                'followers': {'type': 'array', 'items': {'type': 'string'}},
                'summary_template': {'type': 'string'},
                'description_template': {'type': 'string'}
            },
            'required': ['queue', 'parent_issue_id', 'author', 'tags', 'followers', 'summary_template',
                         'description_template'],
        },
        'issues': {
            'type': 'object',
            'patternProperties': {
                '.*': {
                    'properties': {
                        'assignee': {'type': 'string'},
                        'context': {'type': 'object'},
                        'tags': {'type': 'array', 'items': {'type': 'string'}},
                        'invitee': {'type': 'array', 'items': {'type': 'string'}},
                    },
                    'required': ['assignee', 'context'],
                },
            },
        },
    },
    'required': ['meta', 'issues'],
}

MAX_ISSUES = 1000


def get_tasks_resolution_status(startrek_client, task):
    task_dir = os.path.join(TASKS_DIR, task)
    progress_filepath = os.path.join(task_dir, 'progress.yaml')
    if not os.path.exists(progress_filepath):
        raise click.BadParameter('no progress for task "{}" (may be tickets are not created yet?)'.format(task))

    with open(progress_filepath) as f:
        progress_data = yaml.load(f.read(), Loader=yaml.FullLoader)

    stats = Counter()
    created_issues = progress_data.get('created_issues', {})
    for issue_data in six.itervalues(created_issues):
        ticket = startrek_client.issues[issue_data['issue_key']]
        stats[ticket.status.display] += 1

    for status, cnt in six.iteritems(stats):
        click.echo("%15s: %5d" % (status, cnt))


@click.command()
@click.argument('task')
@click.argument('startrek-token', envvar='STARTREK_TOKEN')
@click.option('--dry-run', is_flag=True)
@click.option('--create-many', is_flag=True)
@click.option('--create-as-many-as', default=MAX_ISSUES)
@click.option('--get-resolution-status', is_flag=True)
def main(task, startrek_token, dry_run=False, create_many=False, create_as_many_as=MAX_ISSUES,
         get_resolution_status=False):
    st = startrek_client.Startrek(useragent='python',
                                  base_url='https://st-api.yandex-team.ru',
                                  token=startrek_token)

    if get_resolution_status:
        return get_tasks_resolution_status(st, task)

    task_dir = os.path.join(TASKS_DIR, task)
    input_filepath = os.path.join(task_dir, 'input.yaml')
    if not os.path.exists(input_filepath):
        raise click.BadParameter('Task "{}" is not found.'.format(task))

    with open(input_filepath) as f:
        try:
            input_data = yaml.load(f.read(), Loader=yaml.FullLoader)
            jsonschema.validate(instance=input_data, schema=INPUT_SCHEMA)
        except (yaml.YAMLError, jsonschema.ValidationError) as e:
            raise click.BadParameter('input.yaml for task "{}" appears to be invalid: {}'.format(task, e))

    meta_data = input_data['meta']
    issues_data = input_data['issues']

    progress_filepath = os.path.join(task_dir, 'progress.yaml')
    try:
        with open(progress_filepath) as f:
            progress_data = yaml.load(f.read(), Loader=yaml.FullLoader)
    except (IOError, ValueError):
        progress_data = {'created_issues': {}}

    created_issues = progress_data.get('created_issues', {})

    c = 0
    print('len(issues_data) == {}'.format(len(issues_data)))
    print('len(created_issues) == {}'.format(len(created_issues)))
    for issue_id, issue_data in sorted(six.iteritems(issues_data)):
        if issue_id in created_issues:
            print('Skipping "{}" issue as already created...'.format(issue_id))
            continue

        summary = jinja2.Template(meta_data['summary_template']).render(**issue_data['context'])
        description = jinja2.Template(meta_data['description_template']).render(**issue_data['context'])
        comment = None
        if 'comment_template' in meta_data:
            comment = jinja2.Template(meta_data['comment_template']).render(**issue_data['context'])
        kwargs = {
            'summary': summary,
            'description': description,
            'assignee': issue_data.get('assignee', 'NONE'),
            'queue': meta_data['queue'],
            'followers': meta_data['followers'] + issue_data.get('followers', []),
            'author': meta_data['author'],
            'tags': sorted(set(meta_data.get('tags', [])) | set(issue_data.get('tags', []))),
            'parent_issue_id': meta_data['parent_issue_id'],
            'invitee': sorted(set(meta_data.get('invitee', [])) | set(issue_data.get('invitee', []))),
            'comment': comment
        }
        if dry_run:
            summary = kwargs.pop('summary')
            description = kwargs.pop('description')
            print('---')
            print(yaml.dump(kwargs))
            print('Summary', summary)
            print('Description')
            print(description)
            print('---')
            if comment:
                print('Comment')
                print(comment)
            issue_url = '-'
        else:
            issue = create_issue(st, **kwargs)

            created_issues[issue_id] = {'issue_key': str(issue.key)}

            with open(progress_filepath, 'w') as f:
                f.write(yaml.dump(progress_data))

            issue_url = 'https://st.yandex-team.ru/{}'.format(issue.key)
            print(issue_id, issue_url)

        if not create_many:
            print('Processed just one issue "{}" ({}), '
                  'please specify --create-many to process all of them'.format(issue_id, issue_url))
            break

        c += 1
        if c >= create_as_many_as:
            print('Created {} issues (as many as {}), stopping'.format(c, create_as_many_as))
            break


# @click.command('say_sorry_and_close_example')
# @click.argument('task')
# @click.argument('startrek-token', envvar='STARTREK_TOKEN')
# @click.option('--dry-run', is_flag=True)
def say_sorry_and_close_example(task, startrek_token, dry_run=False):
    issue_ids_to_close = []

    task_dir = os.path.join(TASKS_DIR, 'swat-5786')
    progress_filepath = os.path.join(task_dir, 'progress.yaml')
    with open(progress_filepath) as f:
        progress_data = yaml.load(f.read(), Loader=yaml.FullLoader)

    created_issues = progress_data['created_issues']
    issue_keys_to_close = [created_issues[id]['issue_key'] for id in issue_ids_to_close]
    if dry_run:
        print('Issues to be closed:')
        print(issue_keys_to_close)
        print(len(created_issues) - len(issue_keys_to_close), 'will be left')
        return
    else:
        st = startrek_client.Startrek(useragent='python',
                                      base_url='https://st-api.yandex-team.ru',
                                      token=startrek_token)

        for issue_key in issue_keys_to_close:
            issue = st.issues[issue_key]
            transition = issue.transitions['close']
            text = (u'Этот тикет был создан по ошибке, прошу прощения за беспокойство. '
                    u'C твоей стороны не потребуется никаких действий в связи с анонсируемыми изменениями.')
            transition.execute(comment=text, resolution='won\'tFix')
            print('Closed', issue_key)


if __name__ == '__main__':
    main()
