#!/usr/bin/env python3

import os
import sys
import textwrap
import warnings

import yaml
import requests


STAFF_KEY = 'OAuth ' + os.getenv('STAFF_OAUTH')

TEMPLATE = {
    'meta': {
        'author': 'torkve',
        'summary_template': 'Обновить базовый образ сервисов "{{abc}}"',
        'description_template': textwrap.dedent(
            """\
            В Яндексе уже ((https://clubs.at.yandex-team.ru/infra-cloud/709 два года)) официально не поддерживается Ubuntu Precise.
            Несмотря на это мы долгое время поддерживали обратную совместимость precise, но больше этого делать ((https://clubs.at.yandex-team.ru/skynet/227 не можем)).

            В настоящий момент сервисы с контейнерами на основе Ubuntu Precise теряют возможность использования %%sky run%% для попадания в данные контейнеры, возможны и другие проблемы.
            В частности могут возникнуть проблемы с использованием вашего сервиса в setrace.

            В вашем ведении имеются следующие сервисы:
            {%- for service in services %}
              * ((https://nanny.yandex-team.ru/ui/#/services/catalog/{{service}}/instance_spec {{service}}))
            {%- endfor %}

            Постарайтесь ASAP обновить данные сервисы на поддерживаемый образ Ubuntu: для этого необходимо:
            1. Перейти по ссылке на сервис
            2. В разделе "**List of layers to assemble instance FS**" удалить базовый образ, выбрать из селектора "**Choose bottom Layer**" актуальный образ и нажать "**Add**"
            3. Сохранить и выкатить снапшот.

            Спасибо!
            """
        ).strip(),
        'comment_template': textwrap.dedent(
            """\
            Пожалуйста, обрати внимание на этот тикет. Он непосредственно влияет на работоспособность сервисов.
            """
        ).strip(),
        'followers': [],
        'parent_issue_id': 'SWAT-8139',
        'queue': 'SWATOPS',
        'tags': ['precise-base-layer-outdated'],
    },
    'issues': {},
}


def get_abc_name(session, abc_id):
    url = f'https://staff-api.yandex-team.ru/v3/groups?_one=1&service.id={abc_id}&_fields=url'
    response = session.get(url, headers={'Authorization': STAFF_KEY}, verify=False)
    response.raise_for_status()
    slug = response.json()['url']

    assert slug.startswith('svc_')
    return slug[4:]


def main():
    groups = {}
    resolved_abcs = {"0": "unknown abc service"}
    for line in open('services.txt'):
        try:
            service_name, status, layer_type, abc_id, users = line.split()
        except ValueError:
            service_name = line.split()[0]
            print(f"Service {service_name!r} has no administrators", file=sys.stderr)
            continue

        if status == 'OFFLINE':
            continue

        users = ','.join(filter(lambda s: not s.startswith('robot-'), users.split(',')))

        if not users:
            print(f"Service {service_name!r} has no non-robot administrators", file=sys.stderr)
        else:
            groups.setdefault(abc_id, []).append([service_name, layer_type, users])

    idx = 1
    with requests.Session() as s:
        for abc, services in groups.items():
            all_users = set()
            records = []
            for service, layer_type, users in services:
                all_users.update(users.split(','))
                if abc not in resolved_abcs:
                    resolved_abcs[abc] = get_abc_name(s, abc)

                records.append(service)

            all_users = list(sorted(all_users))
            TEMPLATE['issues'][str(idx)] = {
                'assignee': all_users[0],
                'invitee': all_users,
                'context': {
                    'abc': resolved_abcs[abc],
                    'services': records,
                }
            }

            idx += 1

    with open('input.yaml', 'w') as f:
        yaml.dump(TEMPLATE, f, allow_unicode=True, width=160)


if __name__ == '__main__':
    warnings.simplefilter('ignore')
    main()
