#!/usr/bin/env python3
import os
import re
import textwrap
import warnings
from collections import defaultdict

import requests
import yaml


STAFF_KEY = 'OAuth ' + os.getenv('STAFF_OAUTH')
NS_MATCHER = re.compile('list/([^/]+)/')

TEMPLATE = {
    'meta': {
        'author': 'torkve',
        'summary_template': 'Авторизовать {{username}} у JugglerSearchBot в телеграме',
        'description_template': textwrap.dedent(
            """\
            Мы в awacs очень хотим перестать рассылать пользователям SMS-сообщения с алертами: ((https://clubs.at.yandex-team.ru/infra-cloud/1353 анонс)).
            Для этого необходимо, чтобы в каждом awacs namespace с настроенным алертингом было достаточно людей, которые будут получать уведомления в случае проблем с балансерами — иначе всегда есть риск, что сработавший алерт потеряется, а при неудачном стечении обстоятельства ваша команда может быть даже пойдёт на LSR. Никому бы этого не хотелось.

            Ты не зарегистрирован у ((https://t.me/JugglerSearchBot бота Juggler)), но являешься получателем уведомлений для следующих неймспейсов:
            {%- for ns in awacs_namespaces %}
              * ((https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{{ns}}/show/ {{ns}}))
            {%- endfor %}

            Чтобы исправить эту проблему, достаточно сделать всего три простых действия:
            1. Открыть в телеграме диалог с ботом ((https://t.me/JugglerSearchBot @JugglerSearchBot)).
            2. Сказать этому боту %%/start%%
            3. После знакомства с ботом закрыть этот тикет.

            Спасибо!
            """
        ).strip(),
        'comment_template': textwrap.dedent(
            """\
            Пожалуйста, обрати внимание на этот тикет. Его решение на займёт у тебя больше минуты, но избавит тебя от SMS-спама, а Яндекс — от лишних трат.
            """
        ).strip(),
        'followers': [],
        'parent_issue_id': 'AWACS-529',
        'queue': 'AWACSCONFIGS',
        'tags': ['default-telegram-alerting'],
    },
    'issues': {},
}


def get_logins(session, slug):
    url = 'https://staff-api.yandex-team.ru/v3/groupmembership?group.url={}&person.official.is_dismissed=false&person.official.is_robot=false&_fields=person.login'
    response = session.get(url.format(slug[1:]), headers={'Authorization': STAFF_KEY}, verify=False)
    response.raise_for_status()
    for person in response.json()['result']:
        login = (person.get('person') or {}).get('login')
        if login:
            yield login


def get_telegram_bindings(session):
    response = session.get('http://juggler-api.search.yandex.net/api/notifications/list_telegram_bindings?do=1')
    response.raise_for_status()
    return response.json()['bindings']


def main():
    page = 1
    page_size = 1000
    group_cache = {}
    url = 'https://juggler-api.search.yandex.net/api/notify_rules/get_notify_rules?do=1&format=json'
    logins_to_namespaces = defaultdict(set)

    with requests.Session() as s:
        tg_chats = get_telegram_bindings(s)

        while True:
            response = s.post(url, json={"page": page, "page_size": page_size}, verify=False)
            response.raise_for_status()
            rules = response.json()['rules']

            print(f"Got {len(rules)} rules on page {page}")

            if not rules:
                break

            for rule in rules:
                logins = set()

                if 'Generated by awacs' not in str(rule.get('description')):
                    continue

                if not isinstance(rule.get('template_kwargs'), dict):
                    continue

                if not rule.get('namespace') or not rule['namespace'].startswith('awacs.'):
                    continue

                method = rule['template_kwargs'].get('method')
                if isinstance(method, str) and method != 'sms':
                    continue
                if not isinstance(method, list) or 'sms' not in method:
                    continue

                logins_and_groups = rule['template_kwargs'].get('login')
                if not isinstance(logins_and_groups, list):
                    continue

                for login in logins_and_groups:
                    if not login.startswith('@'):
                        logins.add(login)
                    elif login in group_cache:
                        logins.update(group_cache[login])
                    else:
                        group_cache[login] = list(get_logins(s, login))
                        logins.update(group_cache[login])

                awacs_ns = re.search(NS_MATCHER, rule['description']).group(1)
                for unavailable in filter(lambda login: login not in tg_chats, logins):
                    logins_to_namespaces[unavailable].add(awacs_ns)

            page += 1

    for login, namespaces in logins_to_namespaces.items():
        TEMPLATE['issues'][login] = {
            'assignee': login,
            'invitee': [login],
            'context': {
                'username': login,
                'awacs_namespaces': list(sorted(namespaces)),
            },
        }

    with open('input.yaml', 'w') as f:
        yaml.dump(TEMPLATE, f, allow_unicode=True, width=160)


if __name__ == '__main__':
    warnings.simplefilter('ignore')
    main()
