# coding: utf-8
from libcpp cimport bool
from libcpp.vector cimport vector
from cython.parallel import prange
from cython import boundscheck, wraparound


cdef extern from "infra/awacs/vendor/awacs/pire/pire.cpp":
    TFsm ParseRegexp(const char* begin, bool surround, bool caseInsensitive) except +
    bool DoIntersect(TFsm& fsm1, TFsm& fsm2) nogil
    bool ContainsBeginMark(TFsm& fsm)
    bool ContainsEndMark(TFsm& fsm)
    char* Dump(TFsm& scanner)


cdef extern from "library/cpp/regex/pire/pire.h" namespace "NPire":
    cpdef cppclass TFsm:
        TFsm()
        void Canonize()


cpdef parse_regexp(line, surround=False, case_insensitive=False):
    c_fsm = ParseRegexp(line, surround, case_insensitive)
    ret = Fsm()
    ret.c_fsm = c_fsm
    return ret


@boundscheck(False)
@wraparound(False)
cpdef get_first_intersection_idx(pairs, fsms):
    cdef vector[TFsm] xs
    cdef vector[TFsm] ys
    cdef Fsm fsm_1
    cdef Fsm fsm_2
    for fsm_id_1, fsm_id_2 in pairs:
        fsm_1 = fsms[fsm_id_1]
        xs.push_back(fsm_1.c_fsm)
        fsm_2 = fsms[fsm_id_2]
        ys.push_back(fsm_2.c_fsm)
    cdef int j
    n = len(pairs)
    for j in prange(n, nogil=True):
        if DoIntersect(xs[j], ys[j]):
            with gil:
                return j
    return -1


cdef class Fsm:
    cdef TFsm c_fsm

    cpdef canonize(self):
        self.c_fsm.Canonize()

    cpdef contains_begin_mark(self):
        return ContainsBeginMark(self.c_fsm)
    
    cpdef contains_end_mark(self):
        return ContainsEndMark(self.c_fsm)
    
    cpdef dump(self):
        return Dump(self.c_fsm)
