# coding: utf-8
import flask
import gevent.threadpool
from sepelib.core import config

from awacs import version
from awacs.model import storage_modern
from awacs.model.backend import BackendCtlManager
from infra.swatlib.logutil import rndstr
from . import webserver
from .base import ApplicationBase


class Application(ApplicationBase):
    name = 'awacsresolverd'

    def __init__(self, instance_id):
        super(Application, self).__init__(instance_id, cache_structure=storage_modern.construct_awacsresolverd_zk_structure())
        self.manager = BackendCtlManager(self.coord, member_id=instance_id, party_suffix='', cache=self.cache)
        self.web = None

    def _run_webserver(self):
        # 1.
        # `app` and `self.web` have to be created and run in the same native thread
        # due to their use of hub-local events.
        # 2.
        # We run webserver in a separate greenlet to make sure there are at *least two*
        # of them in this native thread. This is to avoid random "this operation would block forever"
        # errors, see https://github.com/gevent/gevent/issues/890 for more details.
        # 3.
        # We have to make sure that LocalWebServer does not use logging module, as it will lead it to
        # an attempt to acquire logging's lock from the main gevent hub and failure.
        app = flask.Flask(self.name)
        self.web = webserver.LocalWebServer(cfg=config.get(), app=app, version=version.VERSION)
        gevent.spawn(self.web.run).get()

    def run(self, op_id=''):
        super(Application, self).run(op_id=op_id)
        self.manager.start()

        pool = gevent.threadpool.ThreadPool(1)
        pool.spawn(self._run_webserver).wait()

    def term(self, *args, **kwargs):
        self.web.stop()

    def stop(self, op_id=''):
        op_id = op_id or rndstr()
        self.log.info('%s: stopping web...', op_id)
        self.web.stop()
        self.log.info('%s: web stopped, stopping manager...', op_id)
        self.manager.stop(timeout=30)
        self.log.info('%s: manager stopped', op_id)
        super(Application, self).stop(op_id=op_id)
