# coding: utf-8
import socket

from sepelib.core import config
from sepelib.flask.h.error_handlers import register_error_handlers
from sepelib.yandex import oauth

from awacs import version
from awacs.lib import rpc, zookeeper_client
from awacs.model import zk, cache, apicache, storage_modern, controls_updater
from awacs.web.app import create_status_app
from awacs.web.server import AwacsWebServer
from infra.swatlib.auth import passport
from .base import ApplicationBase


class Application(ApplicationBase):
    name = 'awacsstatusd'

    def __init__(self, instance_id):
        super(Application, self).__init__(instance_id, cache_structure=storage_modern.construct_awacsstatusd_zk_structure())
        webapp = self.init_web_application(
            name=self.name,
            debug=config.get_value('run.debug', False),
            production=config.get_value('run.production', False),
            sentry_client=self.sentry_client
        )
        register_error_handlers(webapp)
        self.web = AwacsWebServer(config.get(), webapp,
                                  version=version.VERSION,
                                  instance_id=instance_id)

    @staticmethod
    def init_web_application(name, debug=False, production=False, sentry_client=None):
        return create_status_app(name,
                                 socket.gethostname(),
                                 version=version.VERSION,
                                 version_timestamp=version.TIMESTAMP,
                                 debug=debug,
                                 production=production,
                                 sentry_client=sentry_client)

    # do not init and shutdown mongo connection:
    def init_mongodb_connection(self):
        pass

    def shutdown_mongodb_connection(self):
        pass

    # /

    # configure only what we need:
    def configure_injector(self, binder):
        """
        :type binder: inject.Binder
        """
        binder.bind(rpc.authentication.IRpcAuthenticator,
                    rpc.authentication.CachingPassportAuthenticator.from_inject(tvm_client=None))
        binder.bind(zookeeper_client.IZookeeperClient, self.coord)
        binder.bind(passport.IPassportClient,
                    passport.PassportClient.from_config(config.get_value('passport')))
        binder.bind(oauth.IOAuth,
                    oauth.OAuth.from_config(config.get_value('oauth')))
        zk_storage = zk.ZkStorage(self.coord)
        binder.bind(zk.IZkStorage, zk_storage)
        binder.bind(cache.IAwacsCache, self.cache)
        binder.bind(apicache.IAwacsApiCache, self.apicache)

        binder.bind(controls_updater.IControlsUpdater, self.controls_updater)

    # /

    def run(self):
        super(Application, self).run()
        self.web.run()

    def term(self, *args, **kwargs):
        self.web.stop()

    def stop(self):
        self.web.stop()
        super(Application, self).stop()
