# coding: utf-8
import flask
import gevent.threadpool
from sepelib.core import config

from awacs import version
from awacs.model import storage_modern
from awacs.model.balancer.manager import (
    BalancerCtlManager,
    BalancerOrderCtlManager,
    BalancerOperationCtlManager,
    BalancerRemovalCtlManager, get_allowed_balancer_matcher,
)
from awacs.model.certs.manager import CertCtlManager, CertOrderCtlManager
from awacs.model.certs.renewal_manager import CertRenewalCtlManager, CertRenewalOrderCtlManager
from awacs.model.dao import IDao
from awacs.model.dns_records.manager import (
    DnsRecordCtlManager,
    DnsRecordOrderCtlManager,
    DnsRecordRemovalCtlManager,
    DnsRecordOperationCtlManager,
)
from awacs.model.domain.managers import DomainCtlManager, DomainOrderCtlManager, DomainOperationCtlManager
from awacs.model.l3_balancer.manager import L3BalancerCtlManager, L3BalancerOrderCtlManager
from awacs.model.namespace.manager import NamespaceOrderCtlManager, NamespaceCtlManager, NamespaceOperationCtlManager
from awacs.model.l7heavy_config.manager import L7HeavyConfigCtlManager, L7HeavyConfigOrderCtlManager
from . import webserver
from .base import ApplicationBase


class Application(ApplicationBase):
    name = 'awacsworkerd'

    MANAGER_NAMES = [
        'namespace_order_ctl_manager',
        'namespace_ctl_manager',
        'balancer_ctl_manager',
        'balancer_order_ctl_manager',
        'balancer_op_ctl_manager',
        'balancer_removal_ctl_manager',
        'dns_record_ctl_manager',
        'dns_record_order_ctl_manager',
        'dns_record_removal_ctl_manager',
        'dns_record_op_ctl_manager',
        'l3_balancer_ctl_manager',
        'l3_balancer_order_ctl_manager',
        'cert_ctl_manager',
        'cert_order_ctl_manager',
        'cert_renewal_ctl_manager',
        'cert_renewal_order_ctl_manager',
        'domain_ctl_manager',
        'domain_order_ctl_manager',
        'domain_operation_ctl_manager',
        'namespace_operation_ctl_manager',
        'l7heavy_config_ctl_manager',
        'l7heavy_config_order_ctl_manager',
    ]

    def __init__(self, instance_id):
        super(Application, self).__init__(instance_id, cache_structure=storage_modern.construct_awacsworkerd_zk_structure())

        self._workload_restricted = config.get_value('run.workload.restricted', False)
        self._workload_managers_to_run = set(config.get_value('run.workload.managers', []))
        self._workload_balancer_locations_to_process = set(config.get_value('run.workload.balancer_locations', []))

        role = config.get_value('run.role', default='default')
        large_namespace_ids = set(config.get_value('run.large_namespace_ids', []))
        if role == 'default':
            allowed_namespace_id_matcher = lambda namespace_id: True
        elif role == 'common':
            allowed_namespace_id_matcher = lambda namespace_id: namespace_id not in large_namespace_ids
        elif role == 'common-leq-k':
            allowed_namespace_id_matcher = lambda namespace_id: (namespace_id not in large_namespace_ids and
                                                                 namespace_id[0] <= 'k')
        elif role == 'common-gtr-k':
            allowed_namespace_id_matcher = lambda namespace_id: (namespace_id not in large_namespace_ids and
                                                                 namespace_id[0] > 'k')
        elif role == 'common-from-a-to-f':
            allowed_namespace_id_matcher = lambda namespace_id: (namespace_id not in large_namespace_ids and
                                                                 namespace_id[0] <= 'f')
        elif role == 'common-from-g-to-r':
            allowed_namespace_id_matcher = lambda namespace_id: (namespace_id not in large_namespace_ids and
                                                                 'f' < namespace_id[0] <= 'r')
        elif role == 'common-from-s-to-z':
            allowed_namespace_id_matcher = lambda namespace_id: (namespace_id not in large_namespace_ids and
                                                                 namespace_id[0] > 'r')
        elif role == 's.yandex-team.ru':
            bound_large_namespace_ids = config.get_value(
                'run.bound_large_namespace_ids', {})  # instance_id -> [namespace_id]
            all_bound_namespace_ids = set()
            for namespace_ids in bound_large_namespace_ids.values():
                all_bound_namespace_ids.update(namespace_ids)

            if instance_id in bound_large_namespace_ids:
                # if we're bound to some namespaces, we run only them
                allowed_namespace_ids = set(bound_large_namespace_ids[instance_id])
            else:
                # if we're not bound, we run all namespaces except bound ones
                allowed_namespace_ids = large_namespace_ids - all_bound_namespace_ids
            allowed_namespace_id_matcher = lambda namespace_id: namespace_id in allowed_namespace_ids
        elif role == 'large':
            allowed_namespace_id_matcher = lambda namespace_id: namespace_id in large_namespace_ids
        else:
            raise AssertionError('unknown awacsworkerd role: {}'.format(role))
        party_suffix = '-{}'.format(role)
        self.namespace_order_ctl_manager = NamespaceOrderCtlManager(
            coord=self.coord,
            member_id=instance_id,
            party_suffix=party_suffix,
            cache=self.cache,
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )
        self.namespace_ctl_manager = NamespaceCtlManager(
            coord=self.coord,
            member_id=instance_id,
            party_suffix=party_suffix,
            cache=self.cache,
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )
        self.namespace_operation_ctl_manager = NamespaceOperationCtlManager(
            zk_client=self.coord,
            cache=self.cache,
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )

        if self._workload_restricted and self._workload_balancer_locations_to_process:
            allowed_balancer_matcher = get_allowed_balancer_matcher(self._workload_balancer_locations_to_process)
        else:
            allowed_balancer_matcher = lambda balancer_pb: True

        self.balancer_ctl_manager = BalancerCtlManager(self.coord, self.cache,
                                                       allowed_namespace_id_matcher=allowed_namespace_id_matcher,
                                                       allowed_balancer_matcher=allowed_balancer_matcher)
        self.balancer_order_ctl_manager = BalancerOrderCtlManager(self.coord, self.cache,
                                                                  allowed_namespace_id_matcher=allowed_namespace_id_matcher,
                                                                  allowed_balancer_matcher=allowed_balancer_matcher)
        self.balancer_op_ctl_manager = BalancerOperationCtlManager(self.coord,
                                                                   instance_id,
                                                                   self.cache,
                                                                   allowed_namespace_id_matcher=allowed_namespace_id_matcher)
        self.balancer_removal_ctl_manager = BalancerRemovalCtlManager(self.coord,
                                                                      self.cache,
                                                                      allowed_namespace_id_matcher=allowed_namespace_id_matcher,
                                                                      allowed_balancer_matcher=allowed_balancer_matcher)

        self.dns_record_ctl_manager = DnsRecordCtlManager(self.coord, self.cache, allowed_namespace_id_matcher)
        self.dns_record_order_ctl_manager = DnsRecordOrderCtlManager(
            self.coord, self.cache, allowed_namespace_id_matcher)
        self.dns_record_removal_ctl_manager = DnsRecordRemovalCtlManager(
            self.coord, self.cache, allowed_namespace_id_matcher)
        self.dns_record_op_ctl_manager = DnsRecordOperationCtlManager(
            self.coord, self.cache, allowed_namespace_id_matcher)

        self.l3_balancer_ctl_manager = L3BalancerCtlManager(self.coord, self.cache, allowed_namespace_id_matcher)
        self.l3_balancer_order_ctl_manager = L3BalancerOrderCtlManager(self.coord, self.cache,
                                                                       allowed_namespace_id_matcher)

        self.l7heavy_config_ctl_manager = L7HeavyConfigCtlManager(
            zk_client=self.coord,
            cache=self.cache,
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )
        self.l7heavy_config_order_ctl_manager = L7HeavyConfigOrderCtlManager(
            zk_client=self.coord,
            cache=self.cache,
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )

        member_id = instance_id
        party_suffix = ''
        self.cert_ctl_manager = CertCtlManager(
            self.coord, member_id, party_suffix, self.cache, allowed_namespace_id_matcher)
        self.cert_order_ctl_manager = CertOrderCtlManager(
            self.coord, member_id, party_suffix, self.cache, allowed_namespace_id_matcher)
        self.cert_renewal_ctl_manager = CertRenewalCtlManager(
            self.coord, member_id, party_suffix, self.cache, allowed_namespace_id_matcher)
        self.cert_renewal_order_ctl_manager = CertRenewalOrderCtlManager(
            self.coord, member_id, party_suffix, self.cache, allowed_namespace_id_matcher)

        self.domain_ctl_manager = DomainCtlManager(self.coord, instance_id, self.cache, allowed_namespace_id_matcher)
        self.domain_order_ctl_manager = DomainOrderCtlManager(self.coord,
                                                              instance_id,
                                                              self.cache,
                                                              allowed_namespace_id_matcher)
        self.domain_operation_ctl_manager = DomainOperationCtlManager(self.coord,
                                                                      instance_id,
                                                                      self.cache,
                                                                      allowed_namespace_id_matcher)
        self.web = None

    def _run_webserver(self):
        # 1.
        # `app` and `self.web` have to be created and run in the same native thread
        # due to their use of hub-local events.
        # 2.
        # We run webserver in a separate greenlet to make sure there are at *least two*
        # of them in this native thread. This is to avoid random "this operation would block forever"
        # errors, see https://github.com/gevent/gevent/issues/890 for more details.
        # 3.
        # We have to make sure that LocalWebServer does not use logging module, as it will lead it to
        # an attempt to acquire logging's lock from the main gevent hub and failure.
        app = flask.Flask(self.name)
        self.web = webserver.LocalWebServer(cfg=config.get(), app=app, version=version.VERSION)
        gevent.spawn(self.web.run).get()

    def _assigned(self, manager_name):
        return not self._workload_restricted or manager_name in self._workload_managers_to_run

    def run(self, op_id=''):
        super(Application, self).run(op_id=op_id)
        for attr in dir(self):
            if attr.endswith('_manager'):
                assert attr in self.MANAGER_NAMES, 'Manager {} is configured but not listed'.format(attr)

        for manager_name in self.MANAGER_NAMES:
            assert hasattr(self, manager_name), 'Listed manager {} is not configured'.format(manager_name)
            if self._assigned(manager_name):
                getattr(self, manager_name).start()
        self._create_default_name_servers()

        pool = gevent.threadpool.ThreadPool(1)
        pool.spawn(self._run_webserver).wait()

    def term(self, *args, **kwargs):
        if self.web is not None:
            self.web.stop()

    def stop(self, op_id=''):
        if self.web is not None:
            self.web.stop()
        for manager_name in reversed(self.MANAGER_NAMES):
            if self._assigned(manager_name):
                getattr(self, manager_name).stop()
        super(Application, self).stop(op_id=op_id)

    @staticmethod
    def _create_default_name_servers():
        IDao.instance().create_default_name_servers()
