# coding: utf-8
import signal
import logging
import logging.config
import warnings

import inject
import gevent
from gevent import config as GEVENT_CONFIG
import gevent.signal
import pymongo
import raven.conf
import raven.breadcrumbs
import raven.handlers.logging
from raven.transport.gevent import GeventedHTTPTransport
from sepelib.core import config
from sepelib.yandex import oauth

from infra.swatlib.logutil import rndstr
from infra.swatlib import httpgridfsclient
from infra.swatlib import metrics
from infra.swatlib import sandbox
from infra.swatlib import orly_client
from infra.swatlib.auth import abc, passport
from awacs import version, resolver
from awacs.lib import (staffclient, mongo, l3mgrclient, nannyclient, ypclient, l7heavy_client,
                       nannyrpcclient, itsclient, itsrpcclient, ypliterpcclient,
                       startrekclient, ya_vault, certificator, nannystaffcache,
                       yasm_client, juggler_client, yp_service_discovery, dns_resolver, idmclient,
                       dns_manager_client, yql_client, racktables)
from awacs.model import db, zk, cache, apicache, staffcache, dao, controls_updater
from awacs.lib import rpc, zookeeper_client


class OpIdFilter(logging.Filter):
    def filter(self, record):
        record.op_id = getattr(record, 'op_id', '-')
        return True


class ApplicationBase(object):
    """
    God object, managing service lifetime.
    """
    ENABLE_CACHE_EXTENDED_SIGNALS = False

    NANNY_MONGODB_CONNECTION_ALIAS = 'nanny'

    LOG_FORMAT = "%(hostname)s %(process)s %(asctime)s %(levelname)s [%(name)s] [%(op_id)s] %(message)s"
    LOG_FILTER = OpIdFilter()

    log = logging.getLogger('awacs')

    @staticmethod
    def setup_sentry_logging(sentry_client):
        handler = raven.handlers.logging.SentryHandler(sentry_client, level=logging.WARN)
        raven.conf.setup_logging(handler)
        logging.getLogger('raven').setLevel(logging.ERROR)
        raven.breadcrumbs.ignore_logger('kazoo.client', allow_level=logging.WARN)
        raven.breadcrumbs.ignore_logger('kazoo.handlers', allow_level=logging.WARN)

    @staticmethod
    def create_sentry_client(sentry_dsn):
        """
        :type sentry_dsn: str
        :rtype: raven.Client
        """
        return raven.Client(
            dsn=sentry_dsn,
            transport=GeventedHTTPTransport,
            release=version.VERSION,
            processors=(
                # for details please see
                # https://github.com/getsentry/raven-python/blob/68487496abc11a2c62effebc7e9dec1a16ef0544/raven/processors.py#L76
                'raven.processors.SanitizeKeysProcessor',
            ),
            sanitize_keys=frozenset((
                'authorization',
                'session_id',
                'sessionid2',
                'x-ya-rsa-signature',
                'private_key',
                'cert_secret',
                'cert_tarball',
                'cert_value',
            ))
        )

    def init_mongodb_connection(self):
        logger = logging.getLogger('main')
        try:
            # the convention is to store UTC-only datetimes in database
            # we use tz_aware=True just because it is more consistent with
            # bson.json_util.object_hook, which always returns tz-aware datetimes
            mongo.register_database(config_name='mongo',
                                    read_preference=pymongo.ReadPreference.PRIMARY,
                                    tz_aware=True,
                                    connect=True)
        except Exception as e:
            logger.error('failed to establish mongo connection: %s', e)
            raise
        else:
            logger.debug('successfully established mongo connection')
        try:
            mongo.register_database(config_name='nanny_mongo',
                                    alias=self.NANNY_MONGODB_CONNECTION_ALIAS,
                                    read_preference=pymongo.ReadPreference.PRIMARY,
                                    tz_aware=True,
                                    connect=True)
        except Exception as e:
            logging.getLogger('main').error('failed to establish nanny mongo connection: %s', e)
            raise
        else:
            logger.debug('successfully established nanny mongo connection')

        inject.instance(db.IMongoStorage).ensure_indexes()
        inject.instance(resolver.IGencfgGroupInstancesCache).ensure_indexes()
        inject.instance(resolver.INannyInstancesCache).ensure_indexes()

    def shutdown_mongodb_connection(self):
        mongo.disconnect()

    def __init__(self, instance_id, cache_structure):
        self._gevent_monitor_triggered_cnt = None
        self._gevent_event_loop_blocked_cnt = None

        sentry_dsn = config.get_value('sentry.dsn', default=None)
        if sentry_dsn:
            self.sentry_client = self.create_sentry_client(sentry_dsn)
            self.setup_sentry_logging(sentry_client=self.sentry_client)
        else:
            self.sentry_client = None
        # common error handlers
        # Init coordination service client
        # Several instances can run on one host, use port for distinction.
        self.coord = zookeeper_client.ZookeeperClient(config.get_value('coord'), identifier=instance_id,
                                                      metrics=metrics.ROOT_REGISTRY.path(self.name))
        self.cache = cache.AwacsCache(self.coord, structure=cache_structure,
                                      enable_extended_signals=self.ENABLE_CACHE_EXTENDED_SIGNALS)
        self.apicache = apicache.AwacsApiCache()
        self.controls_updater = controls_updater.ControlsUpdater(controls_dir=u'./controls')
        self.controls_updater.process()
        self.dns_resolver = dns_resolver.DnsResolver()
        inject.clear_and_configure(self.configure_injector)

    def configure_injector(self, binder):
        """
        :type binder: inject.Binder
        """
        binder.bind(rpc.authentication.IRpcAuthenticator,
                    rpc.authentication.CachingPassportAuthenticator.from_inject(tvm_client=None))
        binder.bind(zookeeper_client.IZookeeperClient, self.coord)
        binder.bind(passport.IPassportClient,
                    passport.PassportClient.from_config(config.get_value('passport')))
        binder.bind(oauth.IOAuth,
                    oauth.OAuth.from_config(config.get_value('oauth')))
        nanny_config = dict(config.get_value('nanny'))
        nanny_config['url'] = nanny_config['url'].rstrip('/') + '/v2'
        binder.bind(nannyclient.INannyClient,
                    nannyclient.NannyClient.from_config(nanny_config))
        binder.bind(nannyrpcclient.INannyRpcClient,
                    nannyrpcclient.NannyRpcClient.from_config(nanny_config))
        yp_lite_config = dict(config.get_value('yp_lite'))
        binder.bind(ypliterpcclient.IYpLiteRpcClient,
                    ypliterpcclient.YpLiteRpcClient.from_config(yp_lite_config))
        binder.bind(httpgridfsclient.IHttpGridfsClient,
                    httpgridfsclient.HttpGridfsClient.from_config(config.get_value('gridfs')))
        binder.bind(l3mgrclient.IL3MgrClient,
                    l3mgrclient.L3MgrClient.from_config(config.get_value('l3mgr')))
        binder.bind(l7heavy_client.IL7HeavyClient,
                    l7heavy_client.L7HeavyClient.from_config(config.get_value('l7_heavy')))
        binder.bind(itsclient.IItsClient,
                    itsclient.ItsClient.from_config(config.get_value('its')))
        binder.bind(itsrpcclient.IItsRpcClient,
                    itsrpcclient.ItsRpcClient.from_config(config.get_value('its')))
        binder.bind(startrekclient.IStartrekClient,
                    startrekclient.StartrekClient.from_config(config.get_value('startrek', {})))
        binder.bind(orly_client.IOrlyClient,
                    orly_client.OrlyClient.from_config(config.get_value('orly')))
        binder.bind(yp_service_discovery.IResolver,
                    yp_service_discovery.Resolver.from_config(config.get_value('sd')))
        binder.bind(yql_client.IYqlClient,
                    yql_client.YqlClient.from_config(config.get_value('yql')))
        mongo_storage = db.MongoStorage()
        binder.bind(db.IMongoStorage, mongo_storage)
        zk_storage = zk.ZkStorage(self.coord)
        binder.bind(zk.IZkStorage, zk_storage)
        binder.bind(cache.IAwacsCache, self.cache)
        binder.bind(apicache.IAwacsApiCache, self.apicache)
        binder.bind(dao.IDao, dao.Dao(zk_storage, mongo_storage, self.cache))
        binder.bind(dns_resolver.IDnsResolver, self.dns_resolver)
        binder.bind(dns_manager_client.IDnsManagerClient,
                    dns_manager_client.DnsManagerClient.from_config(config.get_value('dns_manager', {})))

        staff_cache = staffcache.StaffCache('/cache/staff/')
        staff_client = staffclient.StaffClient.from_config(config.get_value('staff'), cache=staff_cache)
        binder.bind(staffclient.IStaffClient, staff_client)

        abc_client = abc.AbcClient.from_config(config.get_value('abc'))
        binder.bind(abc.IAbcClient, abc_client)

        yp_client_factory = ypclient.YpObjectServiceClientFactory.from_config(config.get_value('yp'))
        binder.bind(ypclient.IYpObjectServiceClientFactory, yp_client_factory)

        gencfg_cache = resolver.GencfgGroupInstancesCache(
            mem_maxsize=config.get_value('run.gencfg_memcache_maxsize', default=100))
        gencfg_client = resolver.GencfgClient.from_config(config.get_value('gencfg'), cache=gencfg_cache)
        binder.bind(resolver.IGencfgGroupInstancesCache, gencfg_cache)
        binder.bind(resolver.IGencfgClient, gencfg_client)

        nanny_cache = resolver.NannyInstancesCache(
            mem_maxsize=config.get_value('run.nanny_memcache_maxsize', default=100))
        nanny_client = resolver.NannyClient.from_config(
            nanny_config, gencfg_client, yp_client_factory, cache=nanny_cache)
        binder.bind(resolver.INannyInstancesCache, nanny_cache)
        binder.bind(resolver.INannyClient, nanny_client)

        binder.bind(certificator.ICertificatorClient, certificator.CertificatorClient.from_config(config))
        binder.bind(ya_vault.IYaVaultClient, ya_vault.YaVaultClient.from_config(config))

        nanny_staff_cache = config.get_value('nanny_staff_cache')
        binder.bind(nannystaffcache.INannyStaffCache,
                    nannystaffcache.NannyStaffCache(connection_alias=self.NANNY_MONGODB_CONNECTION_ALIAS,
                                                    collection_name=nanny_staff_cache['collection']))

        binder.bind(yasm_client.IYasmClient,
                    yasm_client.YasmClient.from_config(config.get_value('yasm_client')))
        binder.bind(idmclient.IIDMClient,
                    idmclient.IDMClient.from_config(config.get_value('idm_client')))
        binder.bind(juggler_client.IJugglerClient,
                    juggler_client.JugglerClient.from_config(config.get_value('juggler_client')))

        binder.bind(controls_updater.IControlsUpdater, self.controls_updater)
        binder.bind(sandbox.ISandboxClient, sandbox.SandboxClient())
        binder.bind(racktables.IRacktablesClient, racktables.RacktablesClient())

    def _gevent_monitor_cb(self, hub):
        self._gevent_monitor_triggered_cnt.inc(1)

    def _gevent_monitor_blocking_cb(self, hub):
        assert hub.periodic_monitoring_thread

        did_block = hub.periodic_monitoring_thread._greenlet_tracer.did_block_hub(hub)
        if not did_block:
            return

        self._gevent_event_loop_blocked_cnt.inc(1)

    # Public methods
    def setup_environment(self):
        # Patch urllib3 connection pool to use gevent queue
        import urllib3
        from urllib3.connectionpool import ConnectionPool, log as connectionpool_log
        from gevent.queue import LifoQueue
        ConnectionPool.QueueCls = LifoQueue
        connectionpool_log.setLevel(logging.CRITICAL)

        gevent_registry = metrics.ROOT_REGISTRY.path('gevent')
        self._gevent_monitor_triggered_cnt = gevent_registry.get_counter('monitor-triggered')
        self._gevent_event_loop_blocked_cnt = gevent_registry.get_counter('event-loop-blocked')
        GEVENT_CONFIG.max_blocking_time = 0.1
        GEVENT_CONFIG.max_memory_usage = None

        # Temporarily set monitor_thread to True to let start_periodic_monitoring_thread do its job.
        GEVENT_CONFIG.monitor_thread = True
        mt = gevent.get_hub().start_periodic_monitoring_thread()
        # It's *essential* to set it back to False, so that no other gevent hub would start
        # its own standard expensive monitor thread.
        GEVENT_CONFIG.monitor_thread = False

        # We don't use standard monitor_blocking function due to its ineffectiveness --
        # when finding a blocking greenlet, it iterates over ALL of them to print
        # their stacks.
        # Let's use our much simpler version instead:

        # XXX by romanovich@
        # _gevent_monitor_blocking_cb has to go first, because gevent will
        # forcibly set the period of the first monitoring function to GEVENT_CONFIG.max_blocking_time:
        # https://github.com/gevent/gevent/blob/29e795eeb934305ef955a5a005dde41e5832a1b6/src/gevent/_monitor.py#L127
        # /XXX

        mt.add_monitoring_function(function=self._gevent_monitor_blocking_cb,
                                   period=GEVENT_CONFIG.max_blocking_time)
        # Let's disable an expensive default blocking monitor:
        mt.add_monitoring_function(function=mt.monitor_blocking, period=None)
        # Let's disable an expensive standard memory usage monitor:
        mt.add_monitoring_function(function=mt.monitor_memory_usage, period=None)
        # Let's introduce a sentinel function:
        mt.add_monitoring_function(function=self._gevent_monitor_cb, period=1)

        # Make sure there are no surprises like https://st.yandex-team.ru/AWACS-639
        if len(mt.monitoring_functions()) != 2:
            self.log.exception('len(mt.monitoring_functions()) != 2, '
                               'something might go wrong with gevent monitoring thread!')

        # Disable warnings about not validating SSL certificates
        warnings.simplefilter('ignore', urllib3.exceptions.SecurityWarning)
        self.init_mongodb_connection()

    def teardown_environment(self):
        self.shutdown_mongodb_connection()

    def run(self, op_id=''):
        """
        Start application.
        Blocks until stop was called.
        """
        self.log.info('=' * 30)
        self.log.info('starting service...')
        self.controls_updater.start()
        self.controls_updater.process()
        gevent.signal.signal(signal.SIGTERM, self.term)
        self.setup_environment()
        self.coord.start().wait()
        self.cache.start(stacksample=config.get_value('run.stacksample_cache', default=False))

    def term(self, *args, **kwargs):
        pass

    def stop(self, op_id=''):
        """
        Gracefully stop application.
        Can block for a long time or throw exception, be ready.
        """
        if not op_id:
            op_id = rndstr()
        self.log.info('%s: stopping cache...', op_id)
        self.cache.stop()
        self.log.info('%s: cache stopped, stopping coord', op_id)
        self.coord.stop()
        self.log.info('%s: coord stopped, shutting down controls_updater', op_id)
        self.controls_updater.stop()
        self.log.info('%s: controls updater stopped, shutting down mongo connection', op_id)
        self.shutdown_mongodb_connection()
        self.log.info('%s: mongo connection shut down, tearing down env', op_id)
        self.teardown_environment()
        self.log.info('%s: env tore down, finished', op_id)
