"""
A small and simple web server that intends to be run in a separate system thread in
gevent-patched environment. Can be useful to provide a responsive HTTP server
(for serving /yasm_stats, /ping, etc) in a CPU-heavy process.

It uses gevent-safe pywsgi.WSGIServer and does not use logging module to avoid
errors that occur when different system threads with their own gevent hubs
try to "use" each other's hub-local gevent objects like locks.

Please keep it that way.
"""
import flask
from gevent import pywsgi, socket
from sepelib.flask.h import prep_response
from infra.swatlib.util.fs import set_close_exec
from infra.swatlib.webserver import WSGIHandler, setup_access_log_stream
from infra.swatlib.metrics import MetricsExt


class LocalWebServer(object):
    DEFAULT_BACKLOG = 128

    @classmethod
    def _create_server_socket(cls, host, port):
        """
        Try to create ipv6 socket, falling back to ipv4.
        Start listening on it.

        :returns: listening socket
        """
        if host == '0.0.0.0':
            host = ''
        try:
            listener = socket.socket(socket.AF_INET6)
        except EnvironmentError as e:
            listener = socket.socket()
        listener.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        # mark the socket fd as non-inheritable
        set_close_exec(listener.fileno())
        listener.bind((host, port))
        listener.listen(cls.DEFAULT_BACKLOG)
        return listener

    def __init__(self, cfg, app, version):
        """
        :type cfg: dict
        :type app: flask.Flask
        """
        self.version = version
        self.app = app

        webcfg = cfg['web']
        host = webcfg['http']['host']
        port = webcfg['http']['port']
        logstream = None
        if webcfg.get('access_log'):
            logstream = setup_access_log_stream(webcfg['access_log'])
        listener = self._create_server_socket(host, port)
        self.wsgi = pywsgi.WSGIServer(
            listener=listener,
            application=self.app,
            handler_class=WSGIHandler,
            log=logstream)
        self._register_urls()
        # add export metrics to yasm
        MetricsExt(flask_app=self.app, cfg=webcfg.get('metrics'))

    def run(self):
        self.wsgi.serve_forever()

    def stop(self):
        self.wsgi.stop(timeout=1)

    def start(self):
        self.wsgi.start()

    def _register_urls(self):
        # functions part
        self.app.add_url_rule('/ping', view_func=self.ping)
        self.app.add_url_rule('/version', view_func=self.render_version)
        self.app.add_url_rule('/yoe5choh', view_func=self.show_kazoo_queues)

    @classmethod
    def ping(cls):
        return flask.Response('', status=200)

    def render_version(self):
        return prep_response({'version': self.version}, fmt='txt')

    def show_kazoo_queues(self):
        import inject
        from awacs.lib import zookeeper_client
        coord = inject.instance(zookeeper_client.IZookeeperClient)

        queue = []
        for req, _ in coord.client._queue:
            queue.append(repr(req))

        pending = []
        for req, _, _ in coord.client._pending:
            pending.append(repr(req))

        return prep_response({
            'queue': queue,
            'pending': pending,
        }, fmt='txt')
