import gevent
import inject
import itertools
import logging
import requests
import six
from requests.adapters import HTTPAdapter
from sepelib.http.request import json_request

from infra.swatlib.metrics import InstrumentedSession


log = logging.getLogger(__name__)


class CertificatorApiRequestException(Exception):
    def __init__(self, message, response=None):
        super(CertificatorApiRequestException, self).__init__(message)
        self.response = response

    def __repr__(self):
        if six.PY3:
            message = Exception.__str__(self)
        else:
            message = self.message
        if self.response:
            return '{}. Reason: {}'.format(message, self.response.text[:500])
        return message

    __str__ = __repr__


class ICertificatorClient(object):
    """
    Interface to be used in dependency injection.
    """
    @classmethod
    def instance(cls):
        """
        :rtype: CertificatorClient
        """
        return inject.instance(cls)


class CertificatorClient(ICertificatorClient):
    _ENVIRONMENTS = {
        'production': 'https://crt-api.yandex-team.ru',
        'testing': 'https://crt-api.test.yandex-team.ru',
    }
    _DEFAULT_REQ_TIMEOUT = 30
    _DEFAULT_ATTEMPTS = 2

    def __init__(self, url=None, req_timeout=None, attempts=None, token=None, environment=None):
        super(CertificatorClient, self).__init__()
        assert not environment or environment in self._ENVIRONMENTS
        self.environment = environment or 'production'
        self._base_url = (url or self._ENVIRONMENTS.get(environment)).rstrip('/')
        self._req_timeout = req_timeout or self._DEFAULT_REQ_TIMEOUT
        self._attempts = attempts or self._DEFAULT_ATTEMPTS
        self._token = token
        self._session = InstrumentedSession('certificator',
                                            http_codes_to_exclude_from_total_error={429})
        self._session.mount('https://', HTTPAdapter(max_retries=self._attempts))
        self._session.mount('http://', HTTPAdapter(max_retries=self._attempts))

    def send_create_request(self, ca_name, common_name, subject_alternative_names, abc_service_id,
                            is_ecc=False,  desired_ttl_days=None, notify_on_expiration=False):
        """
        :type ca_name: six.text_type
        :type common_name: six.text_type
        :type subject_alternative_names: list[six.text_type]
        :type abc_service_id: int
        :type desired_ttl_days: Optional[int]
        :type is_ecc: bool
        :type notify_on_expiration: Optional[bool]
        :rtype: dict
        """
        url = '{base_url}/api/certificate/'.format(base_url=self._base_url)
        headers = {}
        if self._token:
            headers = {'Authorization': 'OAuth {}'.format(self._token)}
        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'post',
                url,
                ok_statuses=[requests.codes.ok, requests.codes.created],
                exception=CertificatorApiRequestException,
                session=self._session,
                headers=headers,
                json={
                    'ca_name': ca_name,
                    'type': 'host',
                    'common_name': common_name,
                    'hosts': ','.join(itertools.chain([common_name, ], subject_alternative_names)),
                    'abc_service': abc_service_id,
                    'is_ecc': is_ecc,
                    'desired_ttl_days': desired_ttl_days,
                    'notify_on_expiration': notify_on_expiration,
                }
            )
        response_notify_on_expiration = response.get('notify_on_expiration')
        if response_notify_on_expiration != notify_on_expiration:
            log.error(
                'Certificator responded with a wrong `notify_on_expiration` value, '
                'expected: %s, received: %s',
                notify_on_expiration, response_notify_on_expiration
            )
        return response

    def get_cert(self, cert_order_id):
        """
        :type cert_order_id: six.text_type
        :rtype: dict
        """
        url = '{base_url}/api/certificate/{order_id}/'.format(base_url=self._base_url,
                                                              order_id=cert_order_id)
        headers = {}
        if self._token:
            headers = {'Authorization': 'OAuth {}'.format(self._token)}
        return json_request(
            'get', url, ok_statuses=[requests.codes.ok],
            exception=CertificatorApiRequestException, session=self._session, headers=headers,
        )

    def revoke_certificate(self, cert_order_id):
        """
        :type cert_order_id: six.text_type
        """
        url = '{base_url}/api/certificate/{order_id}/'.format(base_url=self._base_url,
                                                              order_id=cert_order_id)
        headers = {}
        if self._token:
            headers = {'Authorization': 'OAuth {}'.format(self._token)}
        json_request(
            'delete', url, ok_statuses=[requests.codes.ok],
            exception=CertificatorApiRequestException, session=self._session, headers=headers,
        )

    def list_auto_managed_hosts(self):
        url = '{base_url}/api/hosts/auto-managed/'.format(base_url=self._base_url)
        headers = {}
        if self._token:
            headers = {'Authorization': 'OAuth {}'.format(self._token)}
        response = json_request(
            'get', url, ok_statuses=[requests.codes.ok],
            exception=CertificatorApiRequestException, session=self._session, headers=headers,
        )
        return [item['host'] for item in response]

    @classmethod
    def from_config(cls, config):
        return cls(url=config.get_value('certificator.url'),
                   token=config.get_value('certificator.token'),
                   environment=config.get_value('certificator.environment'),
                   )
