# coding: utf-8
"""
Initial implementation of gevent-powered go-style contexts,
is to be hugely revised and documented.
"""
import logging
import weakref

from gevent import getcurrent, Timeout
from gevent.event import AsyncResult

from infra.swatlib.logutil import rndstr


class ICtx(object):
    __slots__ = ()

    def done(self):
        """
        :rtype: bool
        """
        raise NotImplementedError

    def error(self):
        """
        :rtype: six.text_type
        """
        raise NotImplementedError

    def link(self, cb):
        """
        :type cb: callable
        """
        raise NotImplementedError

    def unlink(self, cb):
        """
        :type cb: callable
        """
        raise NotImplementedError

    def with_cancel(self):
        """
        :rtype: (CancellableCtx, callable)
        """
        raise NotImplementedError

    def with_forced_timeout(self, seconds):
        """
        :rtype: ForcedTimeoutCtx
        """
        raise NotImplementedError


class CtxBase(object):
    __slots__ = ()

    def with_cancel(self):
        ctx = CancellableCtx(self)

        def cancel_func(err):
            ctx.set_done(err)

        return ctx, cancel_func

    def with_forced_timeout(self, seconds):
        return ForcedTimeoutCtx(seconds, parent_ctx=self)

    def with_op(self, op_id, log):
        """
        :type op_id: six.text_type
        :type log: logging.Logger
        :rtype: OpCtx
        """
        return OpCtx(op_id=op_id, log=log, parent_ctx=self)


class BackgroundCtx(CtxBase, ICtx):
    __slots__ = ()

    def done(self):
        return False

    def error(self):
        return None

    def link(self, cb):
        pass

    def unlink(self, cb):
        pass


class CancellableCtx(CtxBase, ICtx):
    def __init__(self, parent_ctx=None):
        CtxBase.__init__(self)

        self._done = AsyncResult()

        self._parent_ctx = parent_ctx
        self._cb = None
        if parent_ctx:
            self_ref = weakref.ref(self)

            def cb(parent_done_ev):
                obj = self_ref()
                if obj is not None:
                    obj.set_done(parent_done_ev.get())

            cb.auto_unlink = True
            self._cb = cb
            parent_ctx.link(cb)

    def __del__(self):
        self._cb = None
        if self._parent_ctx:
            self._parent_ctx.unlink(self._cb)

    def link(self, cb):
        self._done.rawlink(cb)

    def unlink(self, cb):
        self._done.unlink(cb)

    def done(self):
        return self._done.ready()

    def error(self):
        if self._done.ready():
            return self._done.get()

    def set_done(self, err):
        self._done.set(err)


class OpLoggerAdapter(logging.LoggerAdapter):
    def __init__(self, log, op_id):
        """
        :type log: logging.Logger
        :type op_id: six.text_type
        """
        logging.LoggerAdapter.__init__(self, log, {'op_id': op_id})

    def warn(self, msg, *args, **kwargs):
        return self.warning(msg, *args, **kwargs)


class OpCtx(CtxBase, ICtx):
    __slots__ = ('op_id', '_parent_ctx', '_log_adapter', '_log')

    def __init__(self, op_id, log, parent_ctx=None):
        """
        :type op_id: six.text_type
        :type log: logging.Logger
        :type parent_ctx: CtxBase
        """
        CtxBase.__init__(self)
        self.op_id = op_id
        self._parent_ctx = parent_ctx
        self._log_adapter = None
        if log is None:
            raise ValueError('log must be specified')
        self._log = log

    def _get_log_adapter(self):
        return OpLoggerAdapter(self._log, self.op_id)

    @property
    def log(self):
        if self._log_adapter is None:
            self._log_adapter = self._get_log_adapter()
        return self._log_adapter

    def id(self):
        return self.op_id

    def link(self, cb):
        if self._parent_ctx is not None:
            self._parent_ctx.link(cb)

    def unlink(self, cb):
        if self._parent_ctx is not None:
            self._parent_ctx.unlink(cb)

    def done(self):
        if self._parent_ctx is not None:
            return self._parent_ctx.done()
        else:
            return False

    def error(self):
        if self._parent_ctx is not None:
            return self._parent_ctx.error()

    def with_op(self, op_id, log=None):
        if log is not None:
            raise ValueError('cannot set a new log')
        return OpCtx(op_id='{}-{}'.format(self.op_id, op_id), log=self._log, parent_ctx=self)

    def __repr__(self):
        return 'OpCtx(id={})'.format(self.op_id)


class CtxTimeoutCancelled(BaseException):
    pass


class CtxTimeoutExceeded(BaseException):
    pass


class ForcedTimeoutCtx(CancellableCtx, Timeout, ICtx):
    def __init__(self, seconds, parent_ctx=None):
        """
        :param Ctx parent_ctx:
        :param float seconds:
        """
        CancellableCtx.__init__(self, parent_ctx=parent_ctx)
        Timeout.__init__(self, seconds=seconds, exception=CtxTimeoutExceeded('timeout exceeded'))
        self._throw_cb = None

    def start(self):
        curr = getcurrent()
        self_ref = weakref.ref(self)

        def cb(done_ev):
            obj = self_ref()
            if obj is not None:
                obj.cancel()
                curr.throw(CtxTimeoutCancelled(done_ev.get()))

        cb.auto_unlink = True
        self._throw_cb = cb
        self._done.rawlink(self._throw_cb)
        Timeout.start(self)

    def cancel(self):
        Timeout.cancel(self)
        self._done.unlink(self._throw_cb)
        self._throw_cb = None
