import inject
import six

from infra.swatlib.httpclient import HttpClient, HttpClientException


class DnsManagerError(HttpClientException):
    pass


class IDnsManagerClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: DnsManagerClient
        """
        return inject.instance(cls)


class DnsManagerClient(object):
    ACCOUNT = "robot-awacs-dns"

    def __init__(self, api_url, oauth_token):
        self.token = oauth_token
        self._client = HttpClient(client_name='dns_manager',
                                  exc_cls=DnsManagerError,
                                  base_url=api_url,
                                  token=self.token,
                                  verify=True,
                                  error_parser=self._parse_error)

    @classmethod
    def from_config(cls, config):
        return cls(
            api_url=config['api_url'],
            oauth_token=config['oauth_token'],
        )

    def _parse_error(self, e):
        try:
            resp = e.response.json()
        except:
            return six.text_type(e)
        errors = []
        for error in resp['errors']:
            errors.append('Error {}: {}'.format(error['code'], error.get('error', error.get('message', None))))
        return ';'.join(errors)

    def create_request(self, json):
        return self._client.post('api/v3.0/requests', json=json)

    def cancel_request(self, request_id):
        return self._client.delete('api/v3.0/requests/{}'.format(request_id))

    def get_request_status(self, request_id):
        return self._client.get('api/v3.0/requests/{}'.format(request_id))

    def list_requests(self):
        return self._client.get('api/v3.0/requests')

    def add_record(self, name, type, ttl, rdata):
        req = {"primitives": [{
            "operation": "add",
            "name": name,
            "type": type,
            "data": rdata,
            "ttl": ttl}]
        }
        # "Authorization": None => not send "Authorization" header
        return self._client.put('v2.3/{}/primitives'.format(self.ACCOUNT), json=req, headers={"Authorization": None, "X-Auth-Token": self.token})

    def remove_record(self, name, type, rdata):
        req = {"primitives": [{
            "operation": "delete",
            "name": name,
            "type": type,
            "data": rdata}]
        }
        # "Authorization": None => not send "Authorization" header
        return self._client.put('v2.3/{}/primitives'.format(self.ACCOUNT), json=req, headers={"Authorization": None, "X-Auth-Token": self.token})
