import contextlib

import six


class Event(object):
    __slots__ = (u'name', u'log_level', u'associated_exception')

    def __init__(self, name, log_level=None, associated_exception=None):
        """
        :type name: six.text_type
        :type log_level: Optional[logging.{DEBUG|INFO|WARN|EXCEPTION|...}]
        :type associated_exception: Optional[Exception]
        """
        self.name = name
        self.log_level = log_level
        self.associated_exception = associated_exception


class EventsReporter(object):
    def __init__(self, events, metrics_registry):
        """
        :type events: Iterable[Event | six.text_type]
        :type metrics_registry: swatlib.metrics.Registry
        """
        self._event_signals = {}
        self._event_log_levels = {}
        self._exceptions = {}
        for event in events:
            if isinstance(event, Event):
                event_name = event.name
                if event.log_level is not None:
                    self._event_log_levels[event_name] = event.log_level
                if event.associated_exception is not None:
                    exc = event.associated_exception
                    if exc in self._exceptions:
                        raise RuntimeError(u'Cannot associate exception "{}" with event "{}" '
                                           u'because it already has assigned event "{}"'.format(exc, event_name,
                                                                                                self._exceptions[exc]))
                    self._exceptions[exc] = event_name
            elif isinstance(event, six.text_type):
                event_name = event
            else:
                raise ValueError()
            self._event_signals[event_name] = metrics_registry.get_counter(event_name)

    def report(self, event_name, ctx):
        """
        :type event_name: six.text_type
        :type ctx: awacs.lib.context.OpCtx
        """
        if event_name not in self._event_signals:
            ctx.log.exception(u'Unknown event "%s"', event_name)
            return
        self._event_signals[event_name].inc(1)
        if event_name in self._event_log_levels:
            log_level = self._event_log_levels[event_name]
            ctx.log.log(log_level, u'!!! event happened: "%s"', event_name)

    @contextlib.contextmanager
    def report_error(self, ctx, message=None):
        if not self._exceptions:
            yield
        else:
            try:
                yield
            except Exception as e:
                for exc_type, event_name in six.iteritems(self._exceptions):
                    if isinstance(e, exc_type):
                        if message is not None:
                            ctx.log.exception(u'%s: %s', message, e)
                        self.report(event_name, ctx=ctx)
                raise
