import logging
import inject
import requests
from requests.adapters import HTTPAdapter

from infra.swatlib.httpclient import HttpClient, HttpClientException


class IDMError(HttpClientException):
    pass


class IIDMClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: IDMClient
        """
        return inject.instance(cls)


class IDMClient(object):
    DEFAULT_REQ_TIMEOUT = 10
    DEFAULT_VERIFY_SSL = True

    API_PATH = '/api/v1'
    PAGE_SIZE = 250

    @classmethod
    def from_config(cls, d):
        return cls(url=d['url'],
                   token=d['token'],
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'),
                   max_retries=d.get('max_retries'))

    def __init__(self, url, token, req_timeout=None, verify_ssl=None, max_retries=None):
        req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        verify_ssl = self.DEFAULT_VERIFY_SSL if verify_ssl is None else verify_ssl
        self._client = HttpClient(client_name='idm',
                                  exc_cls=IDMError,
                                  base_url=url,
                                  req_timeout=req_timeout,
                                  token=token,
                                  verify=verify_ssl,
                                  max_retries=max_retries)

    def iterate_system_roles(self, system_id):
        # https://wiki.yandex-team.ru/intranet/idm/api/public/#spisoknod
        uri = '{}/rolenodes/?system={}&limit={}'.format(self.API_PATH, system_id, self.PAGE_SIZE)
        while uri is not None:
            body = self._client.get(uri)
            for node in body['objects']:
                yield node
            uri = body['meta']['next']

    def update_role(self, system_id, role_slug_path, responsibilities=None):
        url = '{}/rolenodes/{}{}'.format(self.API_PATH, system_id, role_slug_path)
        data = {'create': True}
        if responsibilities is not None:
            data['responsibilities'] = responsibilities
        return self._client.put(url, json=data)

    def remove_role(self, system_id, role_slug_path):
        url = '{}/rolenodes/{}{}'.format(self.API_PATH, system_id, role_slug_path)
        return self._client.delete(url)

    def batch(self, subrequests):
        url = '{}/batch/'.format(self.API_PATH)
        return self._client.post(url, json=subrequests)


