# coding: utf-8
import inject

from infra.swatlib.httpclient import HttpClient, HttpClientException


class ItsError(HttpClientException):
    pass


class IItsClient(object):
    @classmethod
    def instance(cls):
        """:rtype: ItsClient"""
        return inject.instance(cls)

    def get_config(self):
        raise NotImplementedError


class ItsClient(IItsClient):
    ITS_API_URL = 'https://its.yandex-team.ru/'
    DEFAULT_REQ_TIMEOUT = 10  # seconds
    DEFAULT_VERIFY_SSL = True

    @classmethod
    def from_config(cls, d):
        return cls(url=d.get('url'),
                   token=d.get('token'),
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'),
                   max_retries=d.get('max_retries'))

    def __init__(self, url=None, token=None, req_timeout=None, verify_ssl=None, max_retries=None):
        url = url or self.ITS_API_URL
        req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        verify_ssl = self.DEFAULT_VERIFY_SSL if verify_ssl is None else verify_ssl
        self._client = HttpClient(client_name='its',
                                  exc_cls=ItsError,
                                  base_url=url,
                                  req_timeout=req_timeout,
                                  token=token,
                                  verify=verify_ssl,
                                  max_retries=max_retries)

    def get_config(self):
        return self._client.get('/v1/config/')

    def get_location(self, location_path):
        return self._client.get('/v2/config/current/locations/{}/'.format(location_path.strip('/')))

    def create_or_update_location(self, location_path, content):
        self._client.post('/v2/config/current/locations/{}/'.format(location_path.strip('/')), json=content)

    def remove_location(self, location_path, version):
        self._client.post('/v2/config/current/locations/{}/'.format(location_path.strip('/')), json={
            'delete_location': True,
            'version': version
        })

    def list_ruchka_ids(self):
        return set(r['id'] for r in self._client.get('v2/config/current/ruchkas/')['content'])

    def get_ruchka_value(self, path):
        resp = self._client.get('/v1/values/{}/'.format(path.strip('/')), return_json=False)
        return resp.headers.get('ETag').strip('"'), resp.json()

    def remove_ruchka_value(self, path, version):
        return self._client.post('/v1/values/{}/'.format(path.strip('/')), json={
            'delete_value': True,
            'version': version,
            'value': ''
        }, return_json=False)
