# coding: utf-8
from six.moves.urllib import parse as urlparse

import inject
from its_repo import api_stub, api_pb2, model_pb2
from nanny_rpc_client import RequestsRpcClient


class IItsRpcClient(object):
    @classmethod
    def instance(cls):
        """
        :rtype: ItsRpcClient
        """
        return inject.instance(cls)


class ItsRpcClient(IItsRpcClient):
    DEFAULT_REQUEST_TIMEOUT = 30

    @classmethod
    def from_config(cls, d):
        return cls(url=d['url'], req_timeout=d.get('req_timeout'), token=d.get('token'))

    def __init__(self, url, req_timeout=None, token=None):
        super(ItsRpcClient, self).__init__()
        url_api = urlparse.urljoin(url, '/api/')
        self.client = RequestsRpcClient(url_api,
                                        request_timeout=req_timeout or self.DEFAULT_REQUEST_TIMEOUT,
                                        oauth_token=token)
        self.stub = api_stub.DynamicValueServiceStub(self.client)

    def list_dynamic_values(self, labels=None):
        req_pb = api_pb2.ListDynamicValuesRequest()
        if labels:
            req_pb.filter.labels.update(labels)
        return self.stub.list_dynamic_values(req_pb)

    def get_dynamic_value(self, id):
        req_pb = api_pb2.GetDynamicValueRequest(id=id)
        return self.stub.get_dynamic_value(req_pb)

    def create_dynamic_value(self, id, spec_pb, author, labels=None, comment=''):
        """
        :type id: str
        :type spec_pb: model_pb2.DynamicValueSpec
        :type labels: dict[str, str]
        :type comment: str
        """
        req_pb = api_pb2.CreateDynamicValueRequest(id=id, spec=spec_pb, author=author, comment=comment)
        if labels:
            req_pb.labels.update(labels)
        return self.stub.create_dynamic_value(req_pb)

    def update_dynamic_value_content(self, id, version, content, author, comment=''):
        req_pb = api_pb2.UpdateDynamicValueContentRequest(
            id=id, version=version, content=content, author=author, comment=comment)
        return self.stub.update_dynamic_value_content(req_pb)

    def remove_dynamic_value(self, id, version, author):
        req_pb = api_pb2.RemoveDynamicValueRequest(id=id, version=version, author=author)
        return self.stub.remove_dynamic_value(req_pb)

    def list_dynamic_value_snapshots(self, id):
        req_pb = api_pb2.ListDynamicValueSnapshotsRequest(dynamic_value_id=id)
        return self.stub.list_dynamic_value_snapshots(req_pb)
