import json

import six
import time
from six.moves.urllib import parse as urllib
import inject
import requests
import juggler_sdk
try:
    from functools import cached_property
except ImportError:
    from cached_property import cached_property
from collections import namedtuple


class IJugglerClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: JugglerClient
        """
        return inject.instance(cls)


class JugglerSdk(juggler_sdk.JugglerApi):
    def list_telegram_bindings(self, users=None):
        """
        :type users: list[str] | None
        :rtype: dict[str, dict]
        """
        users = users or []
        reply = juggler_sdk.common.fetch_json(
            context=self._context,
            handle='/api/notifications/list_telegram_bindings',
            login=users,
        )
        # NOTE according to the spirit of JugglerSdk it would be right to create
        # and return some special type, like ListTelegramBindingsResult, but
        # let's leave this to efmv@ :)
        return reply['bindings']


class NotifyRule(object):
    def __init__(self, selector, template_name, template_kwargs, description=None, rule_id=None, **kwargs):
        self.selector = selector
        self.template_name = template_name
        self.template_kwargs = template_kwargs
        self.description = description

        self.rule_id = rule_id
        self.kwargs = kwargs

    @cached_property
    def template_kwargs_json(self):
        return json.dumps(self.template_kwargs, sort_keys=True)

    def __eq__(self, other):
        if not isinstance(other, NotifyRule):
            return False

        if self.rule_id and other.rule_id and self.rule_id == other.rule_id:
            return True

        return (self.selector == other.selector
                and self.template_name == other.template_name
                and self.template_kwargs == other.template_kwargs
                and self.description == other.description)

    def __ne__(self, other):
        return not self == other

    def __str__(self):
        return "\n".join([self.selector, self.template_name, str(self.template_kwargs), str(self.description)])

    def __repr__(self):
        return str(self)

    def to_dict(self, namespace):
        res = {
            "selector": self.selector,
            "template_name": self.template_name,
            "template_kwargs": self.template_kwargs,
            "namespace": namespace,
            "description": self.description
        }
        if self.rule_id:
            res['rule_id'] = self.rule_id
        return res


class SyncNotifyRulesResult(namedtuple('SyncNotifyRulesResult', ['add', 'remove'])):
    pass


class SyncChecksResult(namedtuple('SyncChecksResult', ['changed', 'removed'])):
    pass


class CreateOrUpdateNamespaceResult(namedtuple('CreateOrUpdateNamespaceResult', ['created', 'updated'])):
    pass


class JugglerClient(object):
    """
    Docs:
        - https://juggler-sdk.n.yandex-team.ru/
        - https://juggler.yandex-team.ru/doc/ (juggler API v1)
        - http://juggler-api.search.yandex.net/api/ (juggler API v2)
    """
    DEFAULT_URL = u'http://juggler-api.search.yandex.net'

    GET_NOTIFY_RULES_PATH = u'/api/notify_rules/get_notify_rules'
    ADD_OR_UPDATE_NOTIFY_RULE_PATH = u'/api/notify_rules/add_or_update_notify_rule'
    REMOVE_NOTIFY_RULE_PATH = u'/api/notify_rules/remove_notify_rule'

    SET_NAMESPACE_PATH = u'/v2/namespaces/set_namespace'
    LIST_NAMESPACES_PATH = u'/v2/namespaces/get_namespaces'
    REMOVE_NAMESPACE_PATH = u'/v2/namespaces/remove_namespace'

    @classmethod
    def from_config(cls, d):
        return cls(d['token'], d['namespace_prefix'], url=d.get('url'), dry_run=d.get('dry_run'))

    def __init__(self, oauth_token, namespace_prefix, url=None, dry_run=None):
        self._oauth_token = oauth_token
        self._namespace_prefix = namespace_prefix
        self._juggler_url = url or self.DEFAULT_URL
        self._dry_run = dry_run if dry_run is not None else False

    def validate_namespace_name(self, name):
        if not name.startswith(self._namespace_prefix):
            raise ValueError(u'Namespace name without prefix "{}" not valid, got "{}"'.format(
                self._namespace_prefix, name))

    def url(self, path, **kwargs):
        """

        :type path: six.text_type
        :type kwargs: dict[str, Any]
        :rtype: six.text_type
        """
        result_url = self._juggler_url + path
        if kwargs:
            result_url += "?" + urllib.urlencode(kwargs)
        return result_url

    @property
    def headers(self):
        return {
            'Authorization': 'OAuth ' + self._oauth_token
        }

    def _build_juggler_api_client(self, mark=None):
        """

        :type mark: six.text_type | None
        :rtype: JugglerSdk
        """
        return JugglerSdk(api_url=self._juggler_url,
                          mark=mark,
                          oauth_token=self._oauth_token,
                          dry_run=self._dry_run)

    def sync_checks(self, namespace, checks, mark=None):
        """
        :type namespace: six.text_type
        :type checks: list[juggler_sdk.Check]
        :type mark: six.text_type | None
        :rtype: SyncChecksResult
        """
        self.validate_namespace_name(namespace)
        changed = set()
        api_client = self._build_juggler_api_client(mark or namespace)
        for check in checks:
            check.namespace = namespace
            result = api_client.upsert_check(check)
            if result.changed:
                changed.add((check.host, check.service))
        cleanup_result = api_client.cleanup()
        return SyncChecksResult(changed=list(changed), removed=list(cleanup_result.removed))

    def get_telegram_subscribers(self, logins):
        """
        :param list[str] logins: users to check
        :rtype: list[str]
        :return: subset of users who have juggler telegram bot authorized
        """
        api_client = self._build_juggler_api_client()
        return list(api_client.list_telegram_bindings(logins).keys())

    def cleanup_checks(self, namespace, mark=None):
        """

        :type namespace: six.text_type
        :type mark: six.text_type | None
        :rtype: SyncChecksResult
        """
        self.validate_namespace_name(namespace)
        api_client = self._build_juggler_api_client(mark or namespace)
        cleanup_result = api_client.cleanup()
        return SyncChecksResult(changed=[], removed=list(cleanup_result.removed))

    def set_downtime_to_namespace(self, namespace, seconds, description, source):
        """

        :type namespace: six.text_type
        :type description: six.text_type
        :type source: six.text_type
        :type seconds: int
        :rtype: juggler_sdk.downtimes.SetDowntimesResponse
        """
        self.validate_namespace_name(namespace)
        api_client = self._build_juggler_api_client()
        start_time = int(time.time())
        return api_client.set_downtimes(
            filters=[juggler_sdk.DowntimeSelector(namespace=namespace)],
            start_time=start_time,
            end_time=start_time + seconds,
            description=description,
            source=source
        )

    def list_notify_rules(self, namespace, page_size=100):
        """
        http://juggler-api.search.yandex.net/api/notify_rules/get_notify_rules

        :type namespace: six.text_type
        :type page_size: int
        """
        self.validate_namespace_name(namespace)
        resp = requests.post(self.url(self.GET_NOTIFY_RULES_PATH, do=1), json={
            "filters": [
                {"namespace": namespace}
            ],
            "page_size": page_size
        })
        if resp.status_code == 400:
            raise self.BadRequestError(resp.text)
        resp.raise_for_status()
        return [
            NotifyRule(**rule)
            for rule in resp.json().get('rules', []) or []
            if rule.get('namespace') == namespace
        ]

    def add_or_update_notify_rule(self, namespace, notify_rule):
        """
        http://juggler-api.search.yandex.net/api/notify_rules/add_or_update_notify_rule

        :type namespace: six.text_type
        :type notify_rule: NotifyRule
        :rtype: six.text_type
        """
        self.validate_namespace_name(namespace)
        resp = requests.post(self.url(self.ADD_OR_UPDATE_NOTIFY_RULE_PATH, do=1),
                             headers=self.headers,
                             json=notify_rule.to_dict(namespace)
                             )
        if resp.status_code == 400:
            raise self.BadRequestError(resp.text)
        resp.raise_for_status()
        rule_id = resp.json().get('result_id')
        return rule_id

    def remove_notify_rule(self, rule_id):
        """
        http://juggler-api.search.yandex.net/api/notify_rules/remove_notify_rule

        :type rule_id: six.text_type
        """
        resp = requests.post(self.url(self.REMOVE_NOTIFY_RULE_PATH, do=1),
                             headers=self.headers,
                             data={'rule_id': rule_id}
                             )
        if resp.status_code == 400:
            raise self.BadRequestError(resp.text)
        resp.raise_for_status()

    def sync_notify_rules(self, namespace, notify_rules):
        """
        :type namespace: six.text_type
        :type notify_rules: list[NotifyRule]
        :rtype: SyncNotifyRulesResult
        """
        self.validate_namespace_name(namespace)
        remove_rules = []
        add_rules = []
        exists_rules = self.list_notify_rules(namespace)
        for exist_rule in exists_rules:
            if exist_rule not in notify_rules:
                remove_rules.append(exist_rule)

        for new_rule in notify_rules:
            if new_rule not in exists_rules:
                add_rules.append(new_rule)

        add, remove = 0, 0

        for add_rule in add_rules:
            try:
                self.add_or_update_notify_rule(namespace, add_rule)
            except Exception as e:
                e.notify_rule = add_rule
                raise e
            add += 1

        for remove_rule in remove_rules:
            self.remove_notify_rule(remove_rule.rule_id)
            remove += 1

        return SyncNotifyRulesResult(add=add, remove=remove)

    def create_or_update_namespace(self, name, abc_service_slug, inherit_downtimers=False,
                                   downtimers=None, owners=None):
        """
        https://juggler.yandex-team.ru/doc/#/namespaces

        :type name: six.text_type
        :type abc_service_slug: six.text_type
        :type inherit_downtimers: bool
        :type downtimers: list[six.text_type]
        :type owners: list[six.text_type]
        :rtype: CreateOrUpdateNamespaceResult
        """
        self.validate_namespace_name(name)
        find_namespace = self.list_namespaces(name)
        exists_namespace = None
        if find_namespace:
            exists_namespace = find_namespace[0]
        need_set = True
        if exists_namespace:
            need_set = (
                    exists_namespace['abc_service'] != abc_service_slug
                    or sorted(exists_namespace['downtimers']) != sorted(downtimers or [])
                    or sorted(exists_namespace['owners']) != sorted(owners or [])
                    or exists_namespace['inherit_downtimers'] != inherit_downtimers
            )
        created, updated = False, False
        if need_set:
            namespace_data = {
                "abc_service": abc_service_slug,
                "downtimers": downtimers or [],
                "owners": owners or [],
                "inherit_downtimers": inherit_downtimers,
                "name": name,
            }
            if exists_namespace:
                namespace_data['id'] = exists_namespace['id']
                updated = True
            else:
                created = True
            resp = requests.post(self.url(self.SET_NAMESPACE_PATH), headers=self.headers, json=namespace_data)
            if resp.status_code == 400:
                raise self.BadRequestError(resp.text)
            resp.raise_for_status()
        return CreateOrUpdateNamespaceResult(created=created, updated=updated)

    def remove_namespace_if_exists(self, name):
        """
        :type name: six.text_type
        :rtype: bool
        """
        self.validate_namespace_name(name)
        existing_namespaces = self.list_namespaces(name)
        if not existing_namespaces:
            return False
        resp = requests.post(
            url=self.url(self.REMOVE_NAMESPACE_PATH),
            headers=self.headers,
            json={
                'id': existing_namespaces[0]['id']
            })
        if resp.status_code == 400:
            raise self.BadRequestError(resp.text)
        resp.raise_for_status()
        return True

    def list_namespaces(self, name, name_as_prefix=False):
        """

        :type name: six.text_type
        :type name_as_prefix: bool
        :rtype: list[dict]
        """
        self.validate_namespace_name(name)
        data = {"name": name}
        if name_as_prefix:
            data['name'] += "*"
        resp = requests.post(self.url(self.LIST_NAMESPACES_PATH), headers=self.headers, json=data)
        if resp.status_code == 400:
            raise self.BadRequestError(resp.text)
        resp.raise_for_status()
        return resp.json().get('items', [])

    class JugglerApiError(Exception):
        pass

    class BadRequestError(JugglerApiError):
        pass

    class SyncNotifyRulesError(Exception):
        def __init__(self):
            pass
