# coding: utf-8
import inject
import six

from infra.swatlib.httpclient import HttpClientException, HttpClient


class IL3MgrClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: L3MgrClient
        """
        return inject.instance(cls)


class L3MgrException(HttpClientException):
    pass


class L3MgrClient(HttpClient):
    _PRODUCTION_BASE_URL = u'https://l3-api.tt.yandex-team.ru'
    _DEFAULT_REQ_TIMEOUT = 60

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def __init__(self, url=None, req_timeout=_DEFAULT_REQ_TIMEOUT, token=None, verify=True, max_retries=None,
                 connection_timeout=None):
        super(L3MgrClient, self).__init__(
            client_name=u'l3mgr',
            exc_cls=L3MgrException,
            base_url=url or self._PRODUCTION_BASE_URL,
            req_timeout=req_timeout,
            token=token,
            verify=verify,
            max_retries=max_retries,
            connection_timeout=connection_timeout
        )

    def _parse_message_from_http_error(self, e):
        """
        :type e: requests.HTTPError
        :rtype: six.text_type
        """
        try:
            data = e.response.json()
        except ValueError:
            data = {}
        if u'message' in data:
            message = u'{}: {!r}'.format(e, data[u'message'])
        else:
            message = six.text_type(e)
        return message

    @staticmethod
    def _make_headers(use_etag, latest_cfg_id):
        if not use_etag:
            return None
        if use_etag and latest_cfg_id is None:
            raise RuntimeError(u'latest_cfg_id must be set if use_etag=True')
        return {b'If-Match': 'cfg_id={}'.format(int(latest_cfg_id)).encode('utf-8')}

    def get_service(self, svc_id, request_timeout=None):
        """
        :type svc_id: six.text_type | int
        :type request_timeout: float
        :rtype: dict
        """
        return self.get(u'/api/v1/service/{svc_id}'.format(svc_id=svc_id), request_timeout=request_timeout)

    def get_config(self, svc_id, cfg_id):
        """
        :type svc_id: six.text_type | int
        :type cfg_id: six.text_type | int
        :rtype: dict
        """
        url = u'/api/v1/service/{svc_id}/config/{cfg_id}'.format(svc_id=svc_id, cfg_id=cfg_id)
        return self.get(url)

    def get_latest_config(self, svc_id):
        """
        L3mgr doesn't have a simple way to get the latest available config, see https://st.yandex-team.ru/TRAFFIC-12255

        :type svc_id: six.text_type | int
        :rtype: dict
        """
        url = u'/api/v1/service/{svc_id}/config'.format(svc_id=svc_id)
        latest_config = None
        for config in self.get(url, params={u'_limit': 20})[u'objects']:
            if latest_config is None or latest_config[u'id'] < config[u'id']:
                latest_config = config
        return latest_config

    def get_vs(self, svc_id, vs_id, request_timeout=None):
        """
        :type svc_id: six.text_type | int
        :type vs_id: six.text_type | int
        :type request_timeout: float
        :rtype: dict
        """
        url = u'/api/v1/service/{svc_id}/vs/{vs_id}'.format(svc_id=svc_id, vs_id=vs_id)
        return self.get(url, request_timeout=request_timeout)

    def create_service(self, fqdn, abc_code, data=None):
        """
        :type fqdn: six.text_type
        :type abc_code: six.text_type
        :type data: dict[six.text_type, Any]
        :rtype: dict
        """
        data = data or {}
        data.update({
            u'fqdn': fqdn,
            u'abc': abc_code,
        })
        return self.post(u'/api/v1/service', data=data)

    def list_services_by_fqdn(self, fqdn, full=False, request_timeout=3):
        limit = 20
        page = 1
        while True:
            resp = self.get(u'/api/v1/service', request_timeout=request_timeout,
                            params={u'fqdn': fqdn,
                                    u'_full': six.text_type(full).lower(),
                                    u'_limit': limit,
                                    u'_page': page})
            for service in resp[u'objects']:
                yield service
            if resp[u'total'] <= limit * page:
                break
            page += 1

    def get_new_ip(self, abc_code, v4=False, external=False, fqdn=None):
        """
        See https://st.yandex-team.ru/TRAFFIC-7811#1526479733000 for details.

        :type abc_code: six.text_type
        :type v4: bool
        :type external: bool
        :type fqdn: six.text_type | None

        :returns: {"object": "2a02:6b8:0:3400:0:71d:0:17", "result": "OK"}
        """
        url = u'/api/v1/abc/{abc_code}/getip'.format(abc_code=abc_code)
        data = {
            u'v4': v4,
            u'external': external,
        }
        if fqdn is not None:
            data[u'fqdn'] = fqdn
        return self.post(url, data=data)

    def create_virtual_server(self, svc_id, ip, port, protocol, config=None, rs=None, groups=None):
        """
        :type svc_id: six.text_type | int
        :type ip: six.text_type
        :type port: int
        :type protocol: six.text_type
        :type rs: list[six.text_type] | None
        :type groups: list[six.text_type] | None
        :type config: dict | None
        :rtype: dict
        """
        url = u'/api/v1/service/{svc_id}/vs'.format(svc_id=svc_id)
        data = {
            u'ip': ip,
            u'port': port,
            u'protocol': protocol,
        }
        config = config or {}
        for key, value in six.iteritems(config):
            data[u'config-{}'.format(key)] = value
        if groups:
            data[u'groups'] = groups
        if rs:
            data[u'rs'] = rs
        return self.post(url, data=data)

    def create_config_with_vs(self, svc_id, vs_ids, comment, use_etag, latest_cfg_id=None):
        """
        :type svc_id: six.text_type | int
        :type vs_ids: list[int | six.text_type]
        :type comment: six.text_type
        :type latest_cfg_id: six.text_type | int | None
        :type use_etag: bool
        :rtype: dict
        """
        url = u'/api/v1/service/{svc_id}/config'.format(svc_id=svc_id)
        return self.post(
            url,
            data={
                u'vs': vs_ids,
                u'comment': comment,
            },
            headers=self._make_headers(use_etag, latest_cfg_id)
        )

    def create_config_with_rs(self, svc_id, groups, use_etag, latest_cfg_id=None):
        """
        :type svc_id: six.text_type | int
        :param list[six.text_type] groups: Example: ['myservice.sas.yp.yandex-team.ru=127.0.0.1']
        :type latest_cfg_id: six.text_type | int | None
        :type use_etag: bool
        :rtype: dict
        :returns: Example: {'object': {'id': 1785}, 'result': 'OK'}
        """
        url = u'/api/v1/service/{svc_id}/editrs'.format(svc_id=svc_id)
        return self.post(
            url,
            data={u'groups': groups},
            headers=self._make_headers(use_etag, latest_cfg_id)
        )

    def process_config(self, svc_id, cfg_id, use_etag, latest_cfg_id=None, force=False):
        """
        :type svc_id: six.text_type | int
        :type cfg_id: six.text_type | int
        :type latest_cfg_id: six.text_type | int | None
        :type force: bool
        :type use_etag: bool
        :rtype: dict
        :returns: {'object': {'id': 1785}, 'result': 'OK'}
        """
        url = u'/api/v1/service/{svc_id}/config/{cfg_id}/process'.format(svc_id=svc_id, cfg_id=cfg_id)
        if force:
            data = {u'force': force}
        else:
            data = None
        return self.post(url,
                         headers=self._make_headers(use_etag, latest_cfg_id),
                         data=data)

    def add_role(self, svc_id, subject, permission):
        """
        :type svc_id: six.text_type | int
        :param six.text_type subject: e.g. ABC service slug such as "SR"
        :param six.text_type permission: e.g. "l3mgr.editvs_service"
        :rtype: dict
        :returns: {"result": "OK", "object": {"id": 447}}
        """
        url = u'/api/v1/service/{svc_id}/role'.format(svc_id=svc_id)
        return self.post(url, data={
            u'subject': subject,
            u'permission': permission,
        })

    def list_roles(self, svc_id):
        """
        :type svc_id: six.text_type | int
        :rtype: dict
        :returns: {
            "objects": [{
                "url": "/api/v1/service/735/role/69", "permission": "l3mgr.editrs_service",
                "description": "Edit RS pool", "abc": "SR", "id": 69
            }], "page": 1, "limit": 20, "total": 3, "action": []}
        """
        url = u'/api/v1/service/{svc_id}/role'.format(svc_id=svc_id)
        return self.get(url)

    def set_grants(self, svc_id, subjects):
        """
        :type svc_id: six.text_type | int
        :param list[six.text_type] subjects: e.g. ABC service slugs such as "SR"
        :rtype: dict
        :returns: {"result": "OK", "object": {"id": 447}}
        """
        url = u'/api/v1/service/{svc_id}/set_grants'.format(svc_id=svc_id)
        return self.post(url, data={u'grant': subjects})

    def list_grants(self, svc_id):
        """
        :type svc_id: six.text_type | int
        :rtype: dict
        :returns: {
            "objects": [
                "svc_logbroker_administration", "svc_logbroker_development"
            ], "page": 1, "total": 2, "limit": 2}
        """
        url = u'/api/v1/service/{svc_id}/grants'.format(svc_id=svc_id)
        return self.get(url)

    def update_meta(self, svc_id, data):
        url = u'/api/v1/service/{svc_id}/editmeta'.format(svc_id=svc_id)
        return self.post(url, data=data)

    def get_abc_service_info(self, abc_service_slug):
        url = u'/api/v1/abc/{abc_service_slug}'.format(abc_service_slug=abc_service_slug)
        return self.get(url)
