# coding: utf-8
import inject

from infra.swatlib.httpclient import HttpClient, HttpClientException
from awacs.model.util import AWACS_L7HEAVY_OWNER_NAME


class L7HeavyError(HttpClientException):
    pass


class IL7HeavyClient(object):
    @classmethod
    def instance(cls):
        """:rtype: ItsClient"""
        return inject.instance(cls)


class L7HeavyClient(IL7HeavyClient):
    L7HEAVY_API_URL = 'https://its.yandex-team.ru/v2/l7/heavy/'
    DEFAULT_REQ_TIMEOUT = 10  # seconds
    DEFAULT_VERIFY_SSL = True

    @classmethod
    def from_config(cls, d):
        return cls(url=d.get('url'),
                   token=d.get('token'),
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'),
                   max_retries=d.get('max_retries'))

    def __init__(self, url=None, token=None, req_timeout=None, verify_ssl=None, max_retries=None):
        url = url or self.L7HEAVY_API_URL
        req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        verify_ssl = self.DEFAULT_VERIFY_SSL if verify_ssl is None else verify_ssl
        self._client = HttpClient(client_name='l7heavy',
                                  exc_cls=L7HeavyError,
                                  base_url=url,
                                  req_timeout=req_timeout,
                                  token=token,
                                  verify=verify_ssl,
                                  max_retries=max_retries)

    @staticmethod
    def _make_if_match_headers(version):
        return {'If-Match': '"{}"'.format(version)}

    def create_config(self, config_id, group_id, its_value_path, logins=None, groups=None, labels=None,
                      link=None):
        """
        :type config_id: six.text_type
        :type group_id: six.text_type
        :type its_value_path: six.text_type
        :type logins: list[six.text_type] | None
        :type groups: list[six.text_type] | None
        :type labels: dict[six.text_type, six.text_type] | None
        :type link: six.text_type | None
        """
        data = {
            'id': config_id,
            'group_id': group_id,
            'its_value_path': its_value_path,
            'managers': {
                'logins': logins or [],
                'groups': groups or [],
            }
        }
        if labels is not None:
            data['labels'] = labels
        if link is not None:
            data['metadata'] = {'owner': AWACS_L7HEAVY_OWNER_NAME, 'link': link}
        return self._client.post('', json=data)

    def get_config(self, config_id):
        """
        type config_id: six.text_type
        rtype: six.text_type, dict
        """
        assert config_id
        resp = self._client.get('{}/weights/'.format(config_id), return_json=False)
        return resp.headers.get('ETag').strip('"'), resp.json()['config']

    def update_config(self, config_id, version, config):
        """
        type config_id: six.text_type
        type version: six.text_type
        type config: dict
        rtype: six.text_type, dict
        """
        assert config_id
        config.pop('id', None)
        resp = self._client.put('{}/weights/'.format(config_id), headers=self._make_if_match_headers(version),
                                json=config, return_json=False)
        return resp.headers.get('ETag').strip('"'), resp.json()

    def remove_config(self, config_id):
        """
        type config_id: six.text_type
        rtype: six.text_type, dict
        """
        assert config_id
        self._client.delete('{}/weights/'.format(config_id), return_json=False)

    def get_config_sections(self, config_id):
        """
        type config_id: six.text_type
        rtype: six.text_type, list
        """
        resp = self._client.get('{}/weights/sections/'.format(config_id), return_json=False)
        return resp.headers.get('ETag').strip('"'), resp.json()['items']

    def save_sections(self, config_id, version, sections):
        """
        :type config_id: six.text_type
        :type version: six.text_type
        :type sections: dict
        """
        resp = self._client.post('{}/weights/sections/'.format(config_id), headers=self._make_if_match_headers(version),
                                 json={'sections': sections}, return_json=False)
        return resp.headers.get('ETag').strip('"'), resp.json()

    def save_section_weights(self, config_id, version, section_weights):
        """
        :type config_id: six.text_type
        :type version: six.text_type
        :type section_weights: dict[six.text_type, dict[six.text_type, dict[six.text_type, dict[six.text_type, int]]]]
        # section_id -> {locations, fallback_locations} -> location_id (uppercase) -> {weight, default_weight} -> int
        """
        self._client.post('{}/weights/values/'.format(config_id), headers=self._make_if_match_headers(version),
                          json={'sections': section_weights})

    def get_its_version(self, config_id):
        return self._client.get('{}/weights/its_value/'.format(config_id))['current_version']

    def push_weights_to_its(self, config_id, target_version, current_version=''):
        data = {'target_version': target_version, 'current_version': current_version}
        self._client.post('{}/weights/its_value/'.format(config_id), json=data)
