# -*- coding: utf-8 -*-
"""
Nanny (https://wiki.yandex-team.ru/JandeksPoisk/Sepe/nanny/rest-api) client.
"""
from collections import namedtuple, defaultdict

import gevent
import inject
import requests
from enum import Enum
from object_validator import DictScheme, String, List, Integer, Dict, Bool
from requests.adapters import HTTPAdapter
from sepelib.http.request import json_request
from sepelib.yandex import ApiRequestException

from infra.swatlib.metrics import InstrumentedSession
from .instance import (Instance, ContainerResourceLimits, Iss3HooksTimeLimits,
                       Iss3HooksResourceLimits)
import six


class NannyApiRequestException(ApiRequestException):
    """Common-case exception for Nanny request"""
    pass


class Resources(namedtuple('Resources',
                           ['sandbox_bsc_shard',
                            'static_files',
                            'sandbox_files',
                            'url_files',
                            'template_set_files',
                            'l7_fast_balancer_config_template_files'])):
    @classmethod
    def from_dict(cls, params):
        kwargs = {f: params[f] for f in cls._fields}
        return cls(**kwargs)


class Template(namedtuple('Template', ['name', 'content'])):
    @classmethod
    def from_dict(cls, params):
        kwargs = {f: params[f] for f in cls._fields}
        return cls(**kwargs)


class SandboxResource(namedtuple('SandboxResource', ['task_id', 'task_type', 'resource_type'])):
    @classmethod
    def from_dict(cls, params):
        kwargs = {f: params[f] for f in cls._fields}
        return cls(**kwargs)


class RegisteredShard(object):
    __slots__ = ['shard_id']

    @classmethod
    def from_dict(cls, d):
        return cls(d['shard_id'])

    def __init__(self, shard_id):
        self.shard_id = shard_id

    def __repr__(self):
        return "RegisteredShard(shard_id: {})".format(self.shard_id)

    __str__ = __repr__


class ShardContainersSettings(object):

    def __init__(self, install_limits):
        """
        :type install_limits: alemate.lib.instance_models.ContainerResourceLimits
        """
        self.install_limits = install_limits


class SandboxBsconfigShard(object):
    def __init__(self,
                 chosen_type='SANDBOX_SHARD',
                 task_id=None,
                 task_type=None,
                 resource_type=None,
                 registered_shard=None,
                 sandbox_shardmap=None,
                 local_path=None,
                 download_queue=None,
                 download_speed=None,
                 storage=None,
                 containers_settings=None,
                 check_period=None,
                 **_):
        """
        :type chosen_type: str | six.text_type
        :type task_id: str | six.text_type
        :type task_type: str | six.text_type
        :type resource_type: str | six.text_type
        :type registered_shard: RegisteredShard
        :type sandbox_shardmap: SandboxResource
        :type local_path: str | six.text_type
        :type download_queue: str | six.text_type
        :type download_speed: int
        :type storage: six.text_type | types.NoneType
        :type containers_settings: types.NoneType
        :type check_period: six.text_type | types.NoneType
        """
        self.chosen_type = chosen_type
        self.task_id = task_id
        self.task_type = task_type
        self.resource_type = resource_type
        self.registered_shard = registered_shard
        self.sandbox_shardmap = sandbox_shardmap
        self.local_path = local_path
        self.download_queue = download_queue
        self.download_speed = download_speed
        self.storage = storage
        self.containers_settings = containers_settings
        self.check_period = check_period

    @classmethod
    def from_dict(cls, params):
        params = params.copy()
        if 'registered_shard' in params:
            params['registered_shard'] = RegisteredShard.from_dict(params['registered_shard'])
        if 'sandbox_shardmap' in params:
            params['sandbox_shardmap'] = SandboxResource.from_dict(params['sandbox_shardmap'])
        limits_dict = params.get('containers_settings', {}).get('install_limits', {})
        limits = ContainerResourceLimits.from_nanny_response(limits_dict)
        params['containers_settings'] = ShardContainersSettings(install_limits=limits)
        return cls(**params)


class ServiceResources(Resources):
    """Service resources description"""

    class StaticFile(namedtuple('StaticFile', ['local_path',
                                               'is_dynamic',
                                               'content',
                                               'download_queue',
                                               'download_speed',
                                               'storage',
                                               'extract_path',
                                               'check_period',
                                               ])):
        @classmethod
        def from_dict(cls, d):
            return cls(
                local_path=d['local_path'],
                is_dynamic=d.get('is_dynamic', False),
                download_queue=d.get('download_queue', None),
                download_speed=d.get('download_speed', None),
                storage=d.get('storage', None),
                content=d['content'],
                extract_path=d.get('extract_path', None),
                check_period=d.get('check_period', None),
            )

    class SandboxFile(namedtuple('SandboxFile', ['local_path', 'is_dynamic', 'task_id', 'task_type', 'resource_type',
                                                 'download_queue', 'download_speed', 'storage', 'resource_id',
                                                 'extract_path', 'check_period'])):
        @classmethod
        def from_dict(cls, d):
            return cls(
                local_path=d['local_path'],
                is_dynamic=d.get('is_dynamic', False),
                download_queue=d.get('download_queue', None),
                download_speed=d.get('download_speed', None),
                storage=d.get('storage', None),
                task_id=d['task_id'],
                task_type=d['task_type'],
                resource_type=d['resource_type'],
                resource_id=d.get('resource_id'),
                extract_path=d.get('extract_path', None),
                check_period=d.get('check_period', None),
            )

    class UrlFile(namedtuple('UrlFile', ['local_path', 'is_dynamic', 'url', 'chksum',
                                         'download_queue', 'download_speed', 'storage', 'extract_path',
                                         'check_period'])):
        @classmethod
        def from_dict(cls, d):
            return cls(
                local_path=d['local_path'],
                is_dynamic=d.get('is_dynamic', False),
                download_queue=d.get('download_queue', None),
                download_speed=d.get('download_speed', None),
                storage=d.get('storage', None),
                url=d['url'],
                chksum=d.get('chksum'),
                extract_path=d.get('extract_path', None),
                check_period=d.get('check_period'),
            )

    class TemplateSetFile(
        namedtuple('TemplateSetFile', ['local_path', 'is_dynamic', 'layout', 'templates',
                                       'download_queue', 'download_speed', 'storage', 'extract_path', 'check_period'])):
        @classmethod
        def from_dict(cls, d):
            return cls(
                local_path=d['local_path'],
                is_dynamic=d.get('is_dynamic', False),
                download_queue=d.get('download_queue', None),
                download_speed=d.get('download_speed', None),
                storage=d.get('storage', None),
                layout=d['layout'],
                templates=[Template.from_dict(i) for i in d['templates']],
                extract_path=d.get('extract_path', None),
                check_period=d.get('check_period'),
            )

    class BalancingOptionsMixin(object):
        BALANCING_TYPES = ['weighted2', 'rr', 'hashing']

        @classmethod
        def _convert_balancing_type_mode(cls, mode, balancing_type, required=True):
            if mode == 'hashing':
                if not required and 'mode' not in balancing_type:
                    return {}
                mode = balancing_type['mode']
                description = balancing_type.get(mode)
                if description is None:
                    if required:
                        raise NannyApiRequestException('No description for selected "{}" hashing type '
                                                       'found'.format(mode))
                    else:
                        return {"mode": {mode: {}}}
                balancing_type = {"mode": {mode: description}}
            return balancing_type

        @classmethod
        def _convert_balancing_type(cls, balancing_type, required=True):
            if not required and 'mode' not in balancing_type:
                return {}
            mode = balancing_type['mode']
            result = balancing_type.get(mode)

            if result is None:
                if required and mode != 'rr':
                    raise NannyApiRequestException('No description for selected "{}" balancing type '
                                                   'found'.format(mode))
                else:
                    return {mode: {}}

            return {mode: cls._convert_balancing_type_mode(mode, result, required=required)}

        @classmethod
        def _convert_balancing_options(cls, balancing_options, required=True):
            balancing_options = balancing_options.copy()
            if required or 'balancing_type' in balancing_options:
                balancing_options['balancing_type'] = cls._convert_balancing_type(balancing_options['balancing_type'],
                                                                                  required=required)
            return balancing_options

    class L7FastBalancerConfigTemplateFile(BalancingOptionsMixin,
                                           namedtuple('L7FastBalancerConfigTemplateFile',
                                                      ['local_path',
                                                       'is_dynamic',
                                                       'content',
                                                       'download_queue',
                                                       'download_speed',
                                                       'storage',
                                                       'check_period',
                                                       ])):
        Content = namedtuple('Content', ['default_port', 'default_admin_port', 'main_domain', 'secaudited_section_ids',
                                         'sections'])
        Section = namedtuple('Section', ['id', 'prefix', 'https_required', 'balancing_options', 'top_level_domains'])
        TopLevelDomain = namedtuple('TopLevelDomain', ['domains', 'groups'])
        Group = namedtuple('Group', ['name', 'service_id', 'snapshot_id', 'balancing_options'])

        @classmethod
        def _group_from_dict(cls, params):
            if 'balancing_options' in params:
                balancing_options = cls._convert_balancing_options(params['balancing_options'], required=False)
            else:
                balancing_options = None

            return cls.Group(
                name=params['name'],
                service_id=params['service_id'],
                snapshot_id=params['snapshot_id'],
                balancing_options=balancing_options
            )

        @classmethod
        def _top_level_domain_from_dict(cls, params):
            return cls.TopLevelDomain(
                domains=params['domains'],
                groups=[cls._group_from_dict(i) for i in params['groups']]
            )

        @classmethod
        def _section_from_dict(cls, params):
            return cls.Section(
                id=params['id'],
                prefix=params['prefix'],
                https_required=params['https_required'],
                balancing_options=cls._convert_balancing_options(params['balancing_options']),
                top_level_domains=[cls._top_level_domain_from_dict(i) for i in params['top_level_domains']]
            )

        @classmethod
        def _content_from_dict(cls, params):
            return cls.Content(
                default_port=params.get('default_port'),
                default_admin_port=params.get('default_admin_port'),
                main_domain=params.get('main_domain'),
                secaudited_section_ids=params.get('secaudited_section_ids', []),
                sections=[cls._section_from_dict(i) for i in params['sections']]
            )

        @classmethod
        def from_dict(cls, params):
            return cls(
                local_path=params['local_path'],
                is_dynamic=params.get('is_dynamic', False),
                download_queue=params.get('download_queue', None),
                download_speed=params.get('download_speed', None),
                storage=params.get('storage', None),
                content=cls._content_from_dict(params['content']),
                check_period=params.get('check_period', None),
            )


class BsconfigResource(object):
    """
        ÐžÐ¿Ð¸ÑÐ°Ð½Ð¸Ðµ Ñ€ÐµÑÑƒÑ€ÑÐ° CMS
    """
    __slots__ = ['name', 'remote_path', 'local_path', 'chksum', 'arch', 'instances', 'short_name', 'is_dynamic',
                 'download_queue', 'download_speed', 'uuid', 'storage', 'extract_path', 'check_period']

    def __init__(self, name, remote_path, local_path, chksum, arch, instances=None, short_name=None, is_dynamic=False,
                 download_queue=None, download_speed=None, uuid=None, storage=None, extract_path=None,
                 check_period=None):
        self.name = name
        self.short_name = short_name
        self.remote_path = remote_path
        self.local_path = local_path
        self.chksum = chksum
        self.arch = arch
        self.instances = instances if instances else []
        self.is_dynamic = is_dynamic
        self.download_queue = download_queue
        self.download_speed = download_speed
        self.uuid = uuid
        self.storage = storage
        self.extract_path = extract_path
        self.check_period = check_period

    @classmethod
    def from_dict(cls, params):
        return cls(**params)

    def __eq__(self, other):
        return self.__slots__ == other.__slots__ and all(getattr(self, f) == getattr(other, f) for f in self.__slots__)

    def __repr__(self):
        r = ("BsconfigResource(name={!r}, short_name={!r}, remote_path={!r}, local_path={!r}, chksum={!r}, arch={!r}, "
             "instances={!r}, uuid={!r})".format(self.name, self.short_name, self.remote_path, self.local_path,
                                                 self.chksum, self.arch, self.instances, self.uuid))
        return r

    __str__ = __repr__

    def to_cms_format(self):
        return self.name, self.remote_path, self.local_path, self.chksum, self.arch


class BsconfigShard(object):
    """
        ÐžÐ¿Ð¸ÑÐ°Ð½Ð¸Ðµ ÑˆÐ°Ñ€Ð´Ð° CMS
    """
    __slots__ = ['name', 'chksum', 'mtime', 'resource_url', 'priority', 'shard_num', 'instances']

    def __init__(self, name, chksum, mtime, resource_url, priority=0, shard_num=-1, instances=None, **_):
        """

        :param name: Ð½Ð°Ð·Ð²Ð°Ð½Ð¸Ðµ ÑˆÐ°Ñ€Ð´Ð°
        :param chksum: Ñ‡ÐµÐºÑÑƒÐ¼Ð¼Ð° Ð´Ð»Ñ Ñ„Ð°Ð¹Ð»Ð¾Ð² ÑˆÐ°Ñ€Ð´Ð° (Ñ…Ð¸Ñ‚Ñ€Ð°Ñ ÑˆÑ‚ÑƒÐºÐ°, Ð»ÑƒÑ‡ÑˆÐµ Ð¸Ð·Ð²Ð½Ðµ Ð½Ðµ Ð¿Ð¾Ð»ÑŒÐ·Ð¾Ð²Ð°Ñ‚ÑŒ Ð¿Ñ€Ð¸ ÑÐ²Ð¾Ð¸Ñ… Ð²Ñ‹Ñ‡Ð¸ÑÐ»ÐµÐ½Ð¸ÑÑ… chksum)
            (Ð¿Ð¾ Ð¸Ð½Ñ„Ð¾Ñ€Ð¼Ð°Ñ†Ð¸Ð¸ Ð¾Ñ‚ @nekto0n ÑÑ‚Ð¾ md5 Ð¾Ñ‚ Ñ„Ð°Ð¹Ð»Ð° shard.conf)
            Ð›ÑƒÑ‡ÑˆÐµ Ð¸ÑÐ¿Ð¾Ð»ÑŒÐ·Ð¾Ð²Ð°Ñ‚ÑŒ checksum Ð¾Ñ‚ Ñ‚Ð¾Ñ€Ñ€ÐµÐ½Ñ‚Ð°.
        :param mtime: Ð²Ñ€ÐµÐ¼Ñ Ð¿Ð¾ÑÐ»ÐµÐ´Ð½ÐµÐ³Ð¾ Ð¸Ð·Ð¼ÐµÐ½ÐµÐ½Ð¸Ñ ÑˆÐ°Ñ€Ð´Ð° (ctime)
        :param resource_url: url Ð² Ð²Ð¸Ð´Ðµ Ñ‚Ð¾Ñ€Ñ€ÐµÐ½Ñ‚Ð°
        :param priority: Ð¿Ñ€Ð¸Ð¾Ñ€Ð¸Ñ‚ÐµÑ‚ ÑˆÐ°Ñ€Ð´Ð° Ð² Ð²Ð¸Ð´Ðµ Ñ‡Ð¸ÑÐ»Ð°, Ð¸Ð·Ð²ÐµÑÑ‚Ð½Ñ‹Ð¼Ð¸ ÑÐ¸ÑÑ‚ÐµÐ¼Ð°Ð¼Ð¸ Ð½Ðµ Ð¸ÑÐ¿Ð¾Ð»ÑŒÐ·ÑƒÐµÑ‚ÑÑ
        :param shard_num: Ð¿Ñ€Ð¸ÑÐ²Ð°Ð¸Ð²Ð°ÐµÑ‚ÑÑ Ð¿Ñ€Ð¸ Ð¸Ð½Ð¸Ñ†Ð¸Ð°Ð»Ð¸Ð·Ð°Ñ†Ð¸Ð¸ Ð² Ñ€Ð¾Ð±Ð¾Ñ‚Ðµ - ÑÐ¼Ñ‹ÑÐ» ÑÐºÑ€Ñ‹Ñ‚, Ð½Ð¸ÐºÐµÐ¼ Ð½Ðµ Ð¸ÑÐ¿Ð¾Ð»ÑŒÐ·ÑƒÐµÑ‚ÑÑ
        :param instances: ÑÐ¿Ð¸ÑÐ¾Ðº Ð¸Ð½ÑÑ‚Ð°Ð½ÑÐ¾Ð² ÑˆÐ°Ñ€Ð´Ð°
        :return:
        """
        self.name = name
        self.chksum = chksum
        self.mtime = mtime
        self.resource_url = resource_url
        self.priority = priority
        self.shard_num = shard_num
        self.instances = instances if instances else []

    def __repr__(self):
        return "BsconfigShard(name='{}', chksum='{}', mtime={}, resource_url={}, priority={}, shard_num={}, " \
               "instances={})".format(self.name, self.chksum, self.mtime, self.resource_url, self.priority,
                                      self.shard_num, self.instances)

    __str__ = __repr__


class SysctlParam(object):
    """
    :type name: six.text_type
    :type value: six.text_type
    """

    def __init__(self, name, value):
        self.name = name
        self.value = value


class ServiceInstances(object):
    """Service instances description"""

    __slots__ = ['chosen_type', 'instance_list', 'gencfg_groups', 'iss_settings',
                 'extended_gencfg_groups', 'yp_pods', 'yp_pod_ids']

    class GencfgGroup(object):

        __slots__ = ['name', 'release', 'tags', 'limits']

        def __init__(self, name, release='trunk', tags=None, limits=None):
            self.name = name
            self.release = release
            self.tags = tags or []
            self.limits = limits

        @classmethod
        def from_dict(cls, params):
            if 'limits' in params:
                params['limits'] = ContainerResourceLimits.from_nanny_response(params['limits'])
            else:
                params['limits'] = None
            kwargs = {f: params[f] for f in cls.__slots__}
            return cls(**kwargs)

        def __repr__(self):
            return 'GencfgGroup(release={!r}, group={!r})'.format(self.release, self.name)

    class NetworkSettings(object):

        def __init__(self, use_mtn, hbf_nat):
            """
            :type use_mtn: bool
            :type hbf_nat: six.text_type
            """
            self.use_mtn = use_mtn
            self.hbf_nat = hbf_nat

        @classmethod
        def from_dict(cls, params):
            return cls(
                use_mtn=params.get('use_mtn', False),
                hbf_nat=params.get('hbf_nat', 'enabled')
            )

    class GencfgVolumesSettings(object):

        def __init__(self, use_volumes):
            """
            :type use_volumes: bool
            """
            self.use_volumes = use_volumes

    class ContainersSettings(object):

        def __init__(self, set_slot_properties):
            """
            :type set_slot_properties: str | six.text_type
            """
            self.set_slot_properties = set_slot_properties

    class InstancePropertiesSettings(object):

        def __init__(self, tags):
            """
            :type tags: str | six.text_type
            """
            self.tags = tags

    class SysctlSettings(object):

        def __init__(self, params):
            self.params = params

        def to_porto_string(self):
            return ';'.join('{}:{}'.format(i.name, i.value) for i in self.params)

        @classmethod
        def from_dict(cls, d):
            p = d.get('params')
            if not p:
                return cls([])
            params = [SysctlParam(i['name'], i['value']) for i in p]
            return cls(params)

    class OrthogonalTags(object):

        __slots__ = ['itype', 'ctype', 'prj', 'metaprj']

        def __init__(self, itype, ctype, prj, metaprj):
            """
            :type itype: six.text_type
            :type ctype: six.text_type
            :type prj: six.text_type
            :type metaprj: six.text_type
            """
            self.itype = itype
            self.ctype = ctype
            self.prj = prj
            self.metaprj = metaprj

        @classmethod
        def from_dict(cls, d):
            DEFAULT_VALUE = 'unknown'
            return cls(
                d.get('itype', DEFAULT_VALUE),
                d.get('ctype', DEFAULT_VALUE),
                d.get('prj', DEFAULT_VALUE),
                d.get('metaprj', DEFAULT_VALUE),
            )

    class ExtendedGencfgGroups(object):
        __slots__ = ['tags', 'groups', 'network_settings', 'gencfg_volumes_settings', 'containers_settings',
                     'instance_properties_settings', 'sysctl_settings']

        def __init__(self, tags, groups, network_settings, gencfg_volumes_settings, containers_settings,
                     instance_properties_settings, sysctl_settings):
            """
            :type network_settings: ServiceInstances.NetworkSettings
            :type gencfg_volumes_settings: ServiceInstances.GencfgVolumesSettings
            :type containers_settings: ServiceInstances.ContainersSettings
            :type instance_properties_settings: ServiceInstances.InstancePropertiesSettings
            :type sysctl_settings: ServiceInstances.SysctlSettings
            """
            self.tags = tags
            self.groups = groups
            self.network_settings = network_settings
            self.gencfg_volumes_settings = gencfg_volumes_settings
            self.containers_settings = containers_settings
            self.instance_properties_settings = instance_properties_settings
            self.sysctl_settings = sysctl_settings

        @classmethod
        def from_dict(cls, d):
            """
            :type d: dict
            :rtype: ServiceInstances.ExtendedGencfgGroups
            """
            tags = d.get('tags', [])
            groups = [ServiceInstances.GencfgGroup.from_dict(r) for r in d.get('groups', [])]
            network = ServiceInstances.NetworkSettings.from_dict(d.get('network_settings', {}))
            use_volumes = d.get('gencfg_volumes_settings', {}).get('use_volumes', False)
            volumes = ServiceInstances.GencfgVolumesSettings(use_volumes=use_volumes)
            set_slot_container = d.get('containers_settings', {}).get('slot_porto_properties', 'NONE')
            containers_settings = ServiceInstances.ContainersSettings(set_slot_properties=set_slot_container)
            tags_settings = d.get('instance_properties_settings', {}).get('tags', 'ALL_STATIC')
            instance_properties = ServiceInstances.InstancePropertiesSettings(tags=tags_settings)
            sysctl = ServiceInstances.SysctlSettings.from_dict(d.get('sysctl_settings', {}))
            return cls(tags=tags,
                       groups=groups,
                       network_settings=network,
                       gencfg_volumes_settings=volumes,
                       containers_settings=containers_settings,
                       instance_properties_settings=instance_properties,
                       sysctl_settings=sysctl)

    class YpPodsAllocation(object):

        __slots__ = ['cluster', 'pod_filter']

        def __init__(self, cluster, pod_filter):
            """
            :type cluster: six.text_type
            :type pod_filter: six.text_type
            """
            self.cluster = cluster
            self.pod_filter = pod_filter

        @classmethod
        def from_dict(cls, d):
            """
            :type d: dict
            :rtype: ServiceInstances.YpPodsAllocation
            """
            return cls(
                cluster=d['cluster'],
                pod_filter=d['pod_filter'],
            )

        def __hash__(self):
            return hash((self.cluster, self.pod_filter))

    class YpPods(object):
        __slots__ = ['allocations', 'orthogonal_tags', 'sysctl_settings']

        def __init__(self, allocations, orthogonal_tags, sysctl_settings):
            """
            :type allocations: list[ServiceInstances.YpPodsAllocation]
            :type orthogonal_tags: ServiceInstances.OrthogonalTags
            :type sysctl_settings: ServiceInstances.SysctlSettings
            """
            self.allocations = allocations
            self.orthogonal_tags = orthogonal_tags
            self.sysctl_settings = sysctl_settings

        @classmethod
        def from_dict(cls, d):
            """
            :type d: dict
            :rtype: ServiceInstances.YpPods
            """
            allocations = [ServiceInstances.YpPodsAllocation.from_dict(i) for i in d.get('allocations', [])]
            orthogonal_tags = ServiceInstances.OrthogonalTags.from_dict(d.get('orthogonal_tags', {}))
            sysctl = ServiceInstances.SysctlSettings.from_dict(d.get('sysctl_settings', {}))
            return cls(
                allocations=allocations,
                orthogonal_tags=orthogonal_tags,
                sysctl_settings=sysctl,
            )

    class YpPod(object):

        __slots__ = ['cluster', 'pod_id']

        def __init__(self, cluster, pod_id):
            """
            :type cluster: six.text_type
            :type pod_id: six.text_type
            """
            self.cluster = cluster
            self.pod_id = pod_id

        @classmethod
        def from_dict(cls, d):
            """
            :type d: dict
            :rtype: ServiceInstances.YpPod
            """
            return cls(
                cluster=d['cluster'],
                pod_id=d['pod_id'],
            )

        def __hash__(self):
            return hash((self.cluster, self.pod_id))

    class YpPodIds(object):

        __slots__ = ['pods', 'orthogonal_tags', 'sysctl_settings']

        def __init__(self, pods, orthogonal_tags, sysctl_settings):
            """
            :type pods: list[ServiceInstances.YpPod]
            :type orthogonal_tags: ServiceInstances.OrthogonalTags
            :type sysctl_settings: ServiceInstances.SysctlSettings
            """
            self.pods = pods
            self.orthogonal_tags = orthogonal_tags
            self.sysctl_settings = sysctl_settings

        def group_pod_ids_by_cluster(self):
            """
            rtype: listiterator[(six.text_type, set[six.text_type])]
            """
            grouped = defaultdict(set)
            for pod in self.pods:
                grouped[pod.cluster].add(pod.pod_id)
            return six.iteritems(grouped)

        @classmethod
        def from_dict(cls, d):
            """
            :type d: dict
            :rtype: ServiceInstances.YpPodIds
            """
            pods = [ServiceInstances.YpPod.from_dict(i) for i in d.get('pods', [])]
            orthogonal_tags = ServiceInstances.OrthogonalTags.from_dict(d.get('orthogonal_tags', {}))
            sysctl = ServiceInstances.SysctlSettings.from_dict(d.get('sysctl_settings', {}))
            return cls(
                pods=pods,
                orthogonal_tags=orthogonal_tags,
                sysctl_settings=sysctl,
            )

    class IssSettings(object):

        __slots__ = ['instance_cls', 'hooks_time_limits', 'hooks_resource_limits']

        def __init__(self, instance_cls, hooks_time_limits=None, hooks_resource_limits=None):
            self.instance_cls = instance_cls
            self.hooks_time_limits = hooks_time_limits
            self.hooks_resource_limits = hooks_resource_limits

        @classmethod
        def from_dict(cls, params):
            return cls(
                instance_cls=params.get('instance_cls', 'ru.yandex.iss.Instance'),
                hooks_time_limits=Iss3HooksTimeLimits.from_dict(params.get('hooks_time_limits', {})),
                hooks_resource_limits=Iss3HooksResourceLimits.from_nanny_response(params.get('hooks_resource_limits',
                                                                                             {}))
            )

    class InstanceType(Enum):
        INSTANCE_LIST = 1
        GENCFG_GROUPS = 2
        EXTENDED_GENCFG_GROUPS = 6
        YP_PODS = 8
        YP_POD_IDS = 9

    INSTANCE_TYPE_MAPPING = {
        'INSTANCE_LIST': InstanceType.INSTANCE_LIST,
        'GENCFG_GROUPS': InstanceType.GENCFG_GROUPS,
        'EXTENDED_GENCFG_GROUPS': InstanceType.EXTENDED_GENCFG_GROUPS,
        'YP_PODS': InstanceType.YP_PODS,
        'YP_POD_IDS': InstanceType.YP_POD_IDS,
    }

    def __init__(self, chosen_type, instance_list=None, gencfg_groups=None,
                 extended_gencfg_groups=None, iss_settings=None, yp_pods=None, yp_pod_ids=None,
                 ):
        """
        :type extended_gencfg_groups: ServiceInstances.ExtendedGencfgGroups
        :type yp_pods: ServiceInstances.YpPods
        :type yp_pod_ids: ServiceInstances.YpPodIds
        """
        self.chosen_type = chosen_type
        self.instance_list = instance_list
        self.gencfg_groups = gencfg_groups
        self.extended_gencfg_groups = extended_gencfg_groups
        self.yp_pods = yp_pods
        self.yp_pod_ids = yp_pod_ids
        self.iss_settings = iss_settings

    @classmethod
    def from_dict(cls, d):
        chosen_type = ServiceInstances.INSTANCE_TYPE_MAPPING[d['chosen_type']]
        instance_list = [Instance.from_dict(dict(r, power=r.get('weight'))) for r in d.get('instance_list', [])]
        gencfg_groups = [ServiceInstances.GencfgGroup.from_dict(r) for r in d.get('gencfg_groups', [])]
        extended_gencfg_groups = ServiceInstances.ExtendedGencfgGroups.from_dict(d.get('extended_gencfg_groups', {}))
        yp_pods = ServiceInstances.YpPods.from_dict(d.get('yp_pods', {}))
        yp_pod_ids = ServiceInstances.YpPodIds.from_dict(d.get('yp_pod_ids', {}))
        iss_settings = ServiceInstances.IssSettings.from_dict(d.get('iss_settings', {}))
        return cls(
            chosen_type=chosen_type,
            instance_list=instance_list,
            gencfg_groups=gencfg_groups,
            extended_gencfg_groups=extended_gencfg_groups,
            iss_settings=iss_settings,
            yp_pods=yp_pods,
            yp_pod_ids=yp_pod_ids,
        )


class ServiceEngines(namedtuple('_ServiceEngines',
                                ['engine_type'])):
    @classmethod
    def from_dict(cls, params):
        return cls(engine_type=params['engine_type'])


class ServiceRuntimeAttributes(object):
    """Service attributes description"""

    __slots__ = ['service_id', 'snapshot_id', 'resources', 'instances', 'engines', 'instance_spec']

    def __init__(self, service_id, snapshot_id, resources, instances, engines, instance_spec=None, **_):
        """
        :type service_id: six.text_type
        :type snapshot_id: six.text_type
        :type resources: ServiceResources
        :type instances: ServiceInstances
        :type engines: ServiceEngines
        :type instance_spec: dict
        """
        self.service_id = service_id
        self.snapshot_id = snapshot_id
        self.resources = resources
        self.instances = instances
        self.engines = engines
        self.instance_spec = instance_spec

    @classmethod
    def from_dict(cls, params):
        """
        :type params: dict

        Create ServiceRuntimeAttributes class from :param params: received from Nanny
        """
        snapshot_id, service_id, content = params.get('_id'), params.get('service_id'), params['content']

        shard_dict = content['resources'].get('sandbox_bsc_shard')
        sandbox_bsc_shard = None
        if shard_dict:
            sandbox_bsc_shard = SandboxBsconfigShard.from_dict(shard_dict)

        static_files = [ServiceResources.StaticFile.from_dict(r)
                        for r in content['resources'].get('static_files', [])]
        sandbox_files = [ServiceResources.SandboxFile.from_dict(r)
                         for r in content['resources'].get('sandbox_files', [])]
        url_files = [ServiceResources.UrlFile.from_dict(r)
                     for r in content['resources'].get('url_files', [])]
        template_set_files = [ServiceResources.TemplateSetFile.from_dict(d)
                              for d in content['resources'].get('template_set_files', [])]
        l7_fast_balancer_configs = [ServiceResources.L7FastBalancerConfigTemplateFile.from_dict(r)
                                    for r in content['resources'].get('l7_fast_balancer_config_files', [])]

        resources = ServiceResources(
            sandbox_bsc_shard=sandbox_bsc_shard,
            static_files=static_files,
            sandbox_files=sandbox_files,
            url_files=url_files,
            template_set_files=template_set_files,
            l7_fast_balancer_config_template_files=l7_fast_balancer_configs,
        )
        instances = ServiceInstances.from_dict(content['instances'])
        engines = ServiceEngines.from_dict(content['engines'])
        instance_spec = content.get('instance_spec')
        return cls(service_id, snapshot_id, resources, instances, engines, instance_spec)


class INannyClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: NannyClient
        """
        return inject.instance(cls)


class NannyClient(INannyClient):
    _PRODUCTION_BASE_URL = 'http://nanny.yandex-team.ru/v2'
    _TESTING_BASE_URL = 'http://dev-nanny.yandex-team.ru/v2'
    _DEFAULT_REQ_TIMEOUT = 30
    _DEFAULT_ATTEMPTS = 2

    # TODO: at this moment we need only resources and instances section, so we ignore other sections in scheme
    # TODO: ignore_unknown=True is everywhere, need to remove it at some moment

    # we need both required and optional balancing options schemas for
    # sections and backends correspondingly
    REQUIRED_BALANCING_OPTIONS_SCHEMA = DictScheme({
        "backend_timeout": String(),
        "connection_timeout": String(),
        "retries_count": Integer(),
        "fail_on_5xx": Bool(optional=True),
        "keepalive_count": Integer(optional=True),
        "pinger": DictScheme({"url": String()}, optional=True, ignore_unknown=True),
        "balancing_type": DictScheme({
            "mode": String(
                choices=ServiceResources.L7FastBalancerConfigTemplateFile.BALANCING_TYPES),
            "weighted2": Dict(optional=True),
            "rr": DictScheme({}, optional=True),
            "hashing": Dict(optional=True),
        }),
    }, ignore_unknown=True)

    OPTIONAL_BALANCING_OPTIONS_SCHEMA = DictScheme({
        "backend_timeout": String(optional=True),
        "connection_timeout": String(optional=True),
        "retries_count": Integer(optional=True),
        "fail_on_5xx": Bool(optional=True),
        "keepalive_count": Integer(optional=True),
        "pinger": DictScheme({"url": String(optional=True)}, optional=True, ignore_unknown=True),
        "balancing_type": DictScheme({
            "mode": String(
                choices=ServiceResources.L7FastBalancerConfigTemplateFile.BALANCING_TYPES, optional=True),
            "weighted2": Dict(optional=True),
            "rr": DictScheme({}, optional=True),
            "hashing": Dict(optional=True),
        }, optional=True),
    }, optional=True, ignore_unknown=True)

    L7_FAST_BALANCER_GROUP_SCHEME = DictScheme({
        "name": String(),
        "service_id": String(),
        "snapshot_id": String(),
        "balancing_options": OPTIONAL_BALANCING_OPTIONS_SCHEMA,
    }, ignore_unknown=True)

    L7_FAST_BALANCER_TLD_SCHEME = DictScheme({
        "domains": List(String()),
        "groups": List(L7_FAST_BALANCER_GROUP_SCHEME),
    }, ignore_unknown=True)

    L7_FAST_BALANCER_CONFIG_TEMPLATE_SCHEME = DictScheme(
        {
            "local_path": String(),
            "is_dynamic": Bool(optional=True),
            "download_queue": String(optional=True),
            "content": DictScheme({
                "default_port": Integer(optional=True),
                "default_admin_port": Integer(optional=True),
                "main_domain": String(optional=True),
                "secaudited_section_ids": List(String(), optional=True),
                "sections": List(DictScheme(
                    {
                        "id": String(),
                        "prefix": String(),
                        "balancing_options": REQUIRED_BALANCING_OPTIONS_SCHEMA,
                        "https_required": Bool(),
                        "top_level_domains": List(L7_FAST_BALANCER_TLD_SCHEME),
                    }, ignore_unknown=True)),
            }, ignore_unknown=True),
        }, ignore_unknown=True)

    RESOURCES_SECTION_SCHEME = DictScheme(
        {
            "sandbox_bsc_shard": DictScheme(
                {
                    "local_path": String(optional=True),
                    "chosen_type": String(
                        choices=['SANDBOX_SHARD', 'REGISTERED_SHARD', 'SANDBOX_SHARDMAP'],
                        optional=True),
                    "resource_type": String(optional=True),
                    "task_id": String(optional=True),
                    "task_type": String(optional=True),
                    "registered_shard": DictScheme(
                        {
                            'shard_id': String()
                        }, optional=True),
                    "sandbox_shardmap": DictScheme(
                        {
                            'task_type': String(),
                            'task_id': String(),
                            'resource_type': String()
                        }, optional=True),
                    "download_queue": String(optional=True),
                }, ignore_unknown=True, optional=True),
            "static_files": List(DictScheme(
                {
                    "local_path": String(),
                    "is_dynamic": Bool(optional=True),
                    "download_queue": String(optional=True),
                    "content": String(),
                }, ignore_unknown=True), optional=True),
            "sandbox_files": List(DictScheme(
                {
                    "local_path": String(),
                    "is_dynamic": Bool(optional=True),
                    "download_queue": String(optional=True),
                    "task_id": String(),
                    "task_type": String(),
                    "resource_type": String(),
                }, ignore_unknown=True), optional=True),
            "url_files": List(DictScheme(
                {
                    "local_path": String(),
                    "is_dynamic": Bool(optional=True),
                    "download_queue": String(optional=True),
                    "url": String(),
                }, ignore_unknown=True), optional=True),
            "l7_fast_balancer_config_files": List(L7_FAST_BALANCER_CONFIG_TEMPLATE_SCHEME, optional=True),
            "template_set_files": List(DictScheme(
                {
                    "local_path": String(),
                    "is_dynamic": Bool(optional=True),
                    "download_queue": String(optional=True),
                    "layout": String(),
                    "templates": List(DictScheme(
                        {
                            "name": String(),
                            "content": String()
                        }
                    ))
                }, ignore_unknown=True), optional=True)
        }, ignore_unknown=True)

    INSTANCES_SECTION_SCHEME = DictScheme(
        {
            "chosen_type": String(
                choices=list(ServiceInstances.INSTANCE_TYPE_MAPPING)),
            "instance_list": List(DictScheme({
                "host": String(),
                "port": Integer(),
                "tags": List(String(), optional=True),
                "weight": Integer(optional=True),
                "ipv4_address": String(optional=True),
                "ipv6_address": String(optional=True),
            }, ignore_unknown=True), optional=True),
            "gencfg_groups": List(DictScheme({
                "name": String(),
                "release": String(),
                "tags": List(String(), optional=True)
            }, ignore_unknown=True), optional=True),
            "extended_gencfg_groups": DictScheme({
                "tags": List(String(), optional=True),
                "groups": List(DictScheme({
                    "name": String(),
                    "release": String(),
                    "tags": List(String(), optional=True)
                }, ignore_unknown=True), optional=True),
            }, ignore_unknown=True, optional=True),
            "yp_pods": DictScheme({
                "allocations": List(DictScheme({
                    "cluster": String(),
                    "pod_filter": String(),
                }, ignore_unknown=True)),
                "orthogonal_tags": DictScheme({
                    "metaprj": String(),
                    "itype": String(),
                    "ctype": String(),
                    "prj": String()
                }, optional=True, ignore_unknown=True),
                "sysctl_settings": DictScheme({
                    "params": List(DictScheme({
                        "name": String(),
                        "value": String(),
                    }, ignore_unknown=True)),
                }, optional=True, ignore_unknown=True),
            }, ignore_unknown=True, optional=True),
            "yp_pod_ids": DictScheme({
                "pods": List(DictScheme({
                    "cluster": String(),
                    "pod_id": String(),
                }, ignore_unknown=True)),
                "orthogonal_tags": DictScheme({
                    "metaprj": String(),
                    "itype": String(),
                    "ctype": String(),
                    "prj": String()
                }, optional=True, ignore_unknown=True),
                "sysctl_settings": DictScheme({
                    "params": List(DictScheme({
                        "name": String(),
                        "value": String(),
                    }, ignore_unknown=True)),
                }, optional=True, ignore_unknown=True),
            }, ignore_unknown=True, optional=True),
        }, ignore_unknown=True)

    ENGINES_SECTION_SCHEME = DictScheme(
        {
            'engine_type': String(optional=True)
        },
        ignore_unknown=True,
        optional=True)

    RUNTIME_ATTRIBUTES_SNAPSHOT_SCHEME = DictScheme(
        {
            "_id": String(),
            "service_id": String(),
            "content": DictScheme(
                {
                    "engines": ENGINES_SECTION_SCHEME,
                    "instances": INSTANCES_SECTION_SCHEME,
                    "resources": RESOURCES_SECTION_SCHEME
                },
                ignore_unknown=True)
        },
        ignore_unknown=True)

    SERVICE_EXCLUDE_RT_ATTRS_SCHEME = DictScheme({
        'runtime_attrs': DictScheme({
            '_id': String(),
        }, ignore_unknown=True)
    }, ignore_unknown=True)

    SERVICE_INSTANCES_SECTION_SCHEME = DictScheme({
        'snapshot_id': String(),
        'content': INSTANCES_SECTION_SCHEME,
    }, ignore_unknown=True)

    GET_ACTIVE_REVISION_ID_SCHEME = DictScheme({
        'active_revision_id': String(optional=True),
        'orthogonal_tags': DictScheme({
            "metaprj": String(optional=True),
            "prj": String(optional=True),
            "itype": String(optional=True),
            "ctype": String(optional=True),
        }, ignore_unknown=True, optional=True),
        'full_orthogonal_tags': DictScheme({
            "metaprj": List(List(String(optional=True), optional=True), optional=True),
            "prj": List(List(String(optional=True), optional=True), optional=True),
            "itype": List(List(String(optional=True), optional=True), optional=True),
            "ctype": List(List(String(optional=True), optional=True), optional=True),
        }, ignore_unknown=True, optional=True),
        'locations': List(String(), optional=True),
    }, ignore_unknown=True)

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def __init__(self, url=None, req_timeout=None, attempts=None, token=None, tvm_client_id=None):
        super(NannyClient, self).__init__()
        self.base_url = (url or self._PRODUCTION_BASE_URL).rstrip('/')
        self.token = token
        self._req_timeout = req_timeout or self._DEFAULT_REQ_TIMEOUT
        self._attempts = attempts or self._DEFAULT_ATTEMPTS
        self._session = InstrumentedSession('nanny')
        self._session.mount('https://', HTTPAdapter(max_retries=self._attempts))
        self._session.mount('http://', HTTPAdapter(max_retries=self._attempts))
        self.default_headers = self._make_headers()

    def _make_headers(self):
        headers = {'Content-Type': 'application/json'}
        if self.token:
            headers['Authorization'] = 'OAuth {}'.format(self.token)
        return headers

    def get_runtime_attributes_snapshot(self, service_id, snapshot_id):
        """
        Return info about service runtime attributes (resources, instances, etc)

        :type service_id: str | six.text_type
        :type snapshot_id: str | six.text_type
        :rtype: ServiceRuntimeAttributes
        """
        url = '{base_url}/services/{service_id}/history/runtime_attrs/{snapshot_id}/'.format(
            base_url=self.base_url, service_id=service_id, snapshot_id=snapshot_id
        )
        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, scheme=self.RUNTIME_ATTRIBUTES_SNAPSHOT_SCHEME, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )
        return ServiceRuntimeAttributes.from_dict(response)

    def get_service_info_attrs(self, service_id):
        url = '{base_url}/services/{service_id}/info_attrs/'.format(base_url=self.base_url,
                                                                    service_id=service_id)

        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def get_service(self, service_id):
        url = '{base_url}/services/{service_id}/'.format(base_url=self.base_url,
                                                         service_id=service_id)
        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def get_service_runtime_attrs(self, service_id, params=None):
        url = '{base_url}/services/{service_id}/runtime_attrs/'.format(base_url=self.base_url,
                                                                       service_id=service_id)
        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, params=params, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def get_service_auth_attrs(self, service_id):
        url = '{base_url}/services/{service_id}/auth_attrs/'.format(base_url=self.base_url,
                                                                    service_id=service_id)
        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def get_service_exclude_rt_attrs(self, service_id):
        # this method is used by awacsctl located in arcadia, please don't change it
        url = '{base_url}/services/{service_id}/?exclude_runtime_attrs=1'.format(
            base_url=self.base_url,
            service_id=service_id,
        )

        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, scheme=self.SERVICE_EXCLUDE_RT_ATTRS_SCHEME, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def get_service_instances_section(self, service_id):
        url = '{base_url}/services/{service_id}/runtime_attrs/instances/'.format(
            base_url=self.base_url,
            service_id=service_id,
        )
        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, scheme=self.SERVICE_INSTANCES_SECTION_SCHEME, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response['snapshot_id'], ServiceInstances.from_dict(response['content'])

    def get_snapshot_instances_section(self, snapshot_id):
        url = '{base_url}/history/services/runtime_attrs/{snapshot_id}/instances/'.format(
            base_url=self.base_url,
            snapshot_id=snapshot_id,
        )

        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, scheme=self.INSTANCES_SECTION_SCHEME, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return ServiceInstances.from_dict(response)

    def copy_service(self, service_id, new_service_id, description, category, yp_cluster, comment):
        url = '{base_url}/services/{service_id}/copies/'.format(base_url=self.base_url, service_id=service_id)
        req = {
            'id': new_service_id,
            'desc': description,
            'category': category,
            'type': 'AWACS_BALANCER',
            'yp_cluster': yp_cluster,
            'copy_infra_id': True,
            'copy_instances': False,
            'set_yp_lite_properties': 'true',  # nanny quirk
            'copy_secrets': False,  # we would need to store user's sessionId for this
            'copy_auth_attrs': True,
            'comment': comment,
        }
        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'post', url, json=req, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )
        return response

    def update_instances(self, service_id, snapshot_id, comment, instances_section):
        url = '{base_url}/services/{service_id}/runtime_attrs/instances/'.format(base_url=self.base_url,
                                                                                 service_id=service_id)

        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'put', url, json={
                    'comment': comment,
                    'snapshot_id': snapshot_id,
                    'content': instances_section,
                }, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def create_service(self, service_id, comment, info_attrs, runtime_attrs, auth_attrs):
        url = '{base_url}/services/'.format(base_url=self.base_url)

        with gevent.Timeout(self._req_timeout):
            req = {
                'id': service_id,
                'comment': comment,
                'info_attrs': info_attrs,
                'runtime_attrs': runtime_attrs,
                'auth_attrs': auth_attrs
            }
            response = json_request(
                'post', url, json=req, ok_statuses=[requests.codes.ok, requests.codes.created],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def set_snapshot_state(self, service_id, snapshot_id, state, comment, recipe='default'):
        url = '{base_url}/services/{service_id}/events/'.format(base_url=self.base_url, service_id=service_id)

        with gevent.Timeout(self._req_timeout):
            req = {
                'type': 'SET_SNAPSHOT_STATE',
                'content': {
                    'comment': comment,
                    'recipe': recipe,
                    'snapshot_id': snapshot_id,
                    'state': state,
                }
            }
            response = json_request(
                'post', url, json=req, ok_statuses=[requests.codes.ok,
                                                    requests.codes.created,
                                                    requests.codes.accepted],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def get_active_revision_id(self, service_id):
        """
        :rtype: dict
        """
        url = '{base_url}/services/_helpers/get_active_revision_id/{service_id}/'.format(
            base_url=self.base_url, service_id=service_id)

        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, scheme=self.GET_ACTIVE_REVISION_ID_SCHEME, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def list_current_instances(self, service_id):
        """
        :rtype: dict
        """
        url = '{base_url}/services/{service_id}/current_state/instances/'.format(
            base_url=self.base_url, service_id=service_id)

        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        return response

    def update_runtime_attrs_content(self, service_id, snapshot_id, snapshot_priority, runtime_attrs_content,
                                     comment=None):
        url = '{base_url}/services/{service_id}/runtime_attrs/'.format(base_url=self.base_url,
                                                                       service_id=service_id)
        with gevent.Timeout(self._req_timeout):
            return json_request(
                'put', url, ok_statuses=[requests.codes.ok], exception=NannyApiRequestException, session=self._session,
                headers=self.default_headers, json={
                    'snapshot_id': snapshot_id,
                    'content': runtime_attrs_content,
                    'comment': comment or 'Update config.lua',
                    'meta_info': {
                        'scheduling_config': {
                            'scheduling_priority': snapshot_priority,
                        },
                    },
                },
            )

    def update_info_attrs_content(self, service_id, snapshot_id, info_attrs_content, comment=None):
        url = '{base_url}/services/{service_id}/info_attrs/'.format(base_url=self.base_url,
                                                                    service_id=service_id)
        headers = {}
        if self.token:
            headers = {'Authorization': 'OAuth {}'.format(self.token)}
        with gevent.Timeout(self._req_timeout):
            return json_request(
                'put', url, ok_statuses=[requests.codes.ok], exception=NannyApiRequestException, session=self._session,
                headers=headers, json={
                    'snapshot_id': snapshot_id,
                    'content': info_attrs_content,
                    'comment': comment or '',
                },
            )

    def update_service(self, service_id, runtime_attrs, info_attrs, comment):
        url = '{base_url}/services/{service_id}/'.format(base_url=self.base_url, service_id=service_id)
        with gevent.Timeout(self._req_timeout):
            return json_request(
                'put', url, ok_statuses=[requests.codes.ok], exception=NannyApiRequestException, session=self._session,
                headers=self.default_headers, json={
                    'runtime_attrs': {'content': runtime_attrs['content'], 'snapshot_id': runtime_attrs['_id']},
                    'info_attrs': {'content': info_attrs['content'], 'snapshot_id': info_attrs['_id']},
                    'comment': comment,
                },
            )

    def get_current_runtime_attrs_id_and_ctime(self, service_id):
        response = self.get_service_runtime_attrs(service_id, params={'exclude_runtime_attrs': '1'})
        return response['_id'], response['change_info']['ctime']

    def get_target_runtime_attrs_id(self, service_id):
        """
        :type service_id: six.text_type
        :rtype: Optional[six.text_type]
        """
        url = '{base_url}/services/{service_id}/target_state/'.format(base_url=self.base_url,
                                                                      service_id=service_id)
        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )

        if response['content']['is_enabled']:
            return response['content']['snapshot_id']
        else:
            return None

    def get_current_runtime_attrs_id(self, service_id):
        """
        :type service_id: six.text_type
        :rtype: six.text_type
        """
        return self.get_service_instances_section(service_id)[0]

    def list_service_dashboards(self, service_id):
        """
        :rtype: dict
        """
        url = '{base_url}/services_dashboards/_services/{service_id}/'.format(base_url=self.base_url,
                                                                              service_id=service_id)

        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )
        return response

    def get_dashboard(self, dashboard_id):
        """
        :rtype: dict
        """
        url = '{base_url}/services_dashboards/catalog/{dashboard_id}/'.format(base_url=self.base_url,
                                                                              dashboard_id=dashboard_id)

        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )
        return response

    def update_dashboard(self, dashboard_id, content, comment):
        url = '{base_url}/services_dashboards/catalog/{dashboard_id}/'.format(base_url=self.base_url,
                                                                              dashboard_id=dashboard_id)
        with gevent.Timeout(self._req_timeout):
            return json_request(
                'put', url, ok_statuses=[requests.codes.ok], exception=NannyApiRequestException, session=self._session,
                headers=self.default_headers, json={
                    'content': content,
                    'comment': comment
                },
            )

    def shutdown_service(self, service_id, comment):
        url = '{base_url}/services/{service_id}/events/'.format(base_url=self.base_url, service_id=service_id)
        with gevent.Timeout(self._req_timeout):
            return json_request(
                'post', url, ok_statuses=[requests.codes.ok,
                                          requests.codes.accepted],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers, json={
                    'type': 'SET_TARGET_STATE',
                    'content': {
                        'comment': comment,
                        'is_enabled': False
                    }
                }
            )

    def get_service_state(self, service_id):
        url = '{base_url}/services/{service_id}/state/'.format(base_url=self.base_url, service_id=service_id)

        with gevent.Timeout(self._req_timeout):
            response = json_request(
                'get', url, ok_statuses=[requests.codes.ok],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers
            )
        return response

    def remove_service_snapshot(self, service_id, snapshot_id, comment):
        url = '{base_url}/services/{service_id}/events/'.format(base_url=self.base_url, service_id=service_id)
        with gevent.Timeout(self._req_timeout):
            return json_request(
                'post', url, ok_statuses=[requests.codes.ok,
                                          requests.codes.accepted],
                exception=NannyApiRequestException, session=self._session, headers=self.default_headers, json={
                    'type': 'SET_SNAPSHOT_STATE',
                    'content': {
                        'comment': comment,
                        'state': 'DESTROYED',
                        'recipe': 'common',
                        'snapshot_id': snapshot_id
                    }
                }
            )

    def remove_service(self, service_id):
        url = '{base_url}/services/{service_id}/'.format(base_url=self.base_url, service_id=service_id)
        with gevent.Timeout(self._req_timeout):
            try:
                json_request(
                    'delete', url, ok_statuses=[requests.codes.ok,
                                                requests.codes.accepted,
                                                requests.codes.no_content],
                    exception=NannyApiRequestException, session=self._session, headers=self.default_headers
                )
            except NannyApiRequestException as e:
                if (getattr(e, 'message', None) or str(e)) != 'JSON request error: The server returned an invalid JSON response.':
                    # For NO_CONTENT methods Nanny does not return "Content-type" header, let's ignore it
                    raise
