# -*- coding: utf-8 -*-
"""
Some basic alemate models like instance, instance state etc
"""
from __future__ import unicode_literals

import enum

from sepelib.core.constants import MEGABYTE
import six


TOPOLOGY_GROUP_TAG_PREFIX = 'a_topology_group-'
TOPOLOGY_VERSION_TAG_PREFIX = 'a_topology_version-'
DISK_QUOTA_POLICY_ENABLED = 'ENABLED'
DISK_QUOTA_POLICY_DISABLED = 'DISABLED'


class ContainerResourceLimits(object):
    """
    Instance limits description (like CPU, RAM, disk etc)
    """

    __slots__ = ['memory_guarantee', 'memory_limit',
                 'cpu_guarantee', 'cpu_limit', 'cpu_policy',
                 'io_limit', 'io_policy',
                 'net_guarantee', 'net_limit', 'net_priority', 'net_tos',
                 'disk_guarantee', 'disk_limit',
                 'dirty_limit', 'recharge_on_pgfault', 'ulimit', 'oom_is_fatal', 'anon_limit']

    BYTE_LIMITS = ['memory_guarantee',
                   'memory_limit',
                   'anon_limit',
                   'io_limit',
                   'net_guarantee',
                   'net_limit',
                   'dirty_limit']

    CPU_LIMITS = ['cpu_guarantee', 'cpu_limit']
    DISK_LIMITS = ['disk_guarantee', 'disk_limit']
    NET_BYTE_LIMITS = ['net_guarantee', 'net_limit']

    def __init__(self, memory_guarantee=None, memory_limit=None, cpu_guarantee=None, cpu_limit=None, cpu_policy=None,
                 io_limit=None, io_policy=None, net_guarantee=None, net_limit=None, net_priority=None, net_tos=None,
                 disk_guarantee=None, disk_limit=None, dirty_limit=None, recharge_on_pgfault=None,
                 ulimit=None, oom_is_fatal=None, anon_limit=None,
                 **_):
        """
        Instance limits description. For details see:
        https://wiki.yandex-team.ru/porto/limitsoverview
        https://wiki.yandex-team.ru/porto/propertiesanddata/
        https://wiki.yandex-team.ru/users/khlebnikov/yakernel/

        :param int memory_guarantee: Guaranteed memory in bytes for instance
        :param int memory_limit: Memory limit in bytes for instance
        :param int anon_limit: Anonymous mapped memory limit in bytes for instance
        :param float cpu_guarantee: Guaranteed CPU cores
        :param float cpu_limit: CPU cores
        :param cpu_policy: CPU policy
        :param int io_limit: Disk read/write speed limit (bytes/s)
        :param io_policy: Disk IO policy
        :param int net_guarantee: Guaranteed outgoing traffic (bytes/s)
        :param int net_limit: Outgoing traffic network limit (bytes/s)
        :param net_priority: Network packets priority
        :param net_tos: IP Type of Service
        :param disk_guarantee: Guaranteed disk space in Gb
        :param disk_limit: Disk space limit in Gb
        :param int dirty_limit: Dirty file cache limit (bytes)
        :param bool recharge_on_pgfault:
        :param six.text_type ulimit: ulimit specified as string for porto
        :type oom_is_fatal: bool | types.NoneType
        """
        self.memory_guarantee = memory_guarantee
        self.memory_limit = memory_limit
        self.anon_limit = anon_limit
        self.cpu_guarantee = cpu_guarantee
        self.cpu_limit = cpu_limit
        self.cpu_policy = cpu_policy
        self.io_limit = io_limit
        self.io_policy = io_policy
        self.net_guarantee = net_guarantee
        self.net_limit = net_limit
        self.net_priority = net_priority
        self.net_tos = net_tos
        self.disk_guarantee = disk_guarantee
        self.disk_limit = disk_limit
        self.dirty_limit = dirty_limit
        self.recharge_on_pgfault = recharge_on_pgfault
        self.ulimit = ulimit
        self.oom_is_fatal = oom_is_fatal

    @classmethod
    def from_dict(cls, params):
        return cls(**params)

    @classmethod
    def from_nanny_response(cls, params):
        # convert from Mb to bytes (from Mb/s to bytes/s)
        for limit in cls.BYTE_LIMITS:
            if limit in params:
                params[limit] *= MEGABYTE

        # convert from CPU milliseconds to cores
        for limit in cls.CPU_LIMITS:
            if limit in params:
                params[limit] /= 1000.0

        return ContainerResourceLimits.from_dict(params)

    @classmethod
    def from_gencfg_response(cls, params):
        params['cpu_limit'] = params.pop('cpu_cores_limit', None)
        params['cpu_guarantee'] = params.pop('cpu_cores_guarantee', None)
        return ContainerResourceLimits.from_dict(params)

    def __repr__(self):
        kwargs_str = []
        for attr in self.__slots__:
            kwargs_str.append('{}={!r}'.format(attr, getattr(self, attr, None)))
        return 'Instance({})'.format(kwargs_str)

    def update(self, other_limits):
        """
        Update limits with other limits
        """
        for attr in self.__slots__:
            limit = getattr(other_limits, attr, None)
            if limit is not None:
                setattr(self, attr, limit)

    def to_iss_dict(self):
        r = {}
        if self.memory_guarantee is not None:
            r['memory_guarantee'] = self.memory_guarantee
        if self.memory_limit is not None:
            r['memory_limit'] = self.memory_limit
        if self.anon_limit is not None:
            r['anon_limit'] = self.anon_limit

        if self.cpu_guarantee is not None:
            # porto expects CPU limits in core fraction format e.g. 1.5c for 1.5 cores
            r['cpu_guarantee'] = '{:f}c'.format(self.cpu_guarantee)
        if self.cpu_limit is not None:
            # porto expects CPU limits in core fraction format e.g. 1.5c for 1.5 cores
            r['cpu_limit'] = '{:f}c'.format(self.cpu_limit)

        if self.net_guarantee is not None:
            # porto expects network limits in format "eth0: 100500; vlan761: 100500" or "default: 100500"
            r['net_guarantee'] = 'default: {:d}'.format(self.net_guarantee)
        if self.net_limit is not None:
            # porto expects network limits in format "eth0: 100500; vlan761: 100500" or "default: 100500"
            r['net_limit'] = 'default: {:d}'.format(self.net_limit)

        remaining_limits = [
            'cpu_policy',
            'io_limit',
            'io_policy',
            'net_priority',
            'net_tos',
            'dirty_limit',
            'recharge_on_pgfault',
            'ulimit',
            'oom_is_fatal',
        ]
        for l in remaining_limits:
            value = getattr(self, l, None)
            if value is not None:
                r[l] = value
        return r

    def to_dict(self):
        return {k: getattr(self, k, None) for k in self.__slots__}

    @classmethod
    def from_dict(cls, params):
        return cls(**params)

    __str__ = __repr__


class Iss3HooksResourceLimits(object):
    __slots__ = ['iss_hook_stop', 'iss_hook_status', 'iss_hook_install', 'iss_hook_uninstall',
                 'iss_hook_validate', 'iss_hook_notify', 'iss_hook_reopenlogs']

    def __init__(self, iss_hook_stop=None, iss_hook_status=None, iss_hook_install=None, iss_hook_uninstall=None,
                 iss_hook_validate=None, iss_hook_notify=None, iss_hook_reopenlogs=None):
        self.iss_hook_stop = iss_hook_stop
        self.iss_hook_status = iss_hook_status
        self.iss_hook_install = iss_hook_install
        self.iss_hook_uninstall = iss_hook_uninstall
        self.iss_hook_validate = iss_hook_validate
        self.iss_hook_notify = iss_hook_notify
        self.iss_hook_reopenlogs = iss_hook_reopenlogs

    @classmethod
    def from_nanny_response(cls, params):
        return cls(**{f: ContainerResourceLimits.from_nanny_response(params.get(f, {})) for f in cls.__slots__})

    def to_iss_dict(self):
        """
        :rtype: dict
        """
        result = {}
        for hook in self.__slots__:
            hook_limits = getattr(self, hook)
            if hook_limits is not None:
                for limit_key, value in six.iteritems(hook_limits.to_iss_dict()):
                    result['{}.{}'.format(hook, limit_key)] = value
        return result


class MetaData(object):
    __slots__ = ['annotations']

    def __init__(self, annotations):
        """
        :type annotations: dict[six.text_type, six.text_type]
        """
        self.annotations = annotations

    @classmethod
    def from_nanny_response(cls, params):
        """
        :type params: dict
        :rtype: MetaData
        """
        annotations = params.get('annotations', [])
        return cls(annotations={l['key']: l['value'] for l in annotations})


class Iss3HooksTimeLimits(object):
    __slots__ = ['iss_hook_start', 'iss_hook_stop', 'iss_hook_status', 'iss_hook_install', 'iss_hook_uninstall',
                 'iss_hook_validate', 'iss_hook_notify', 'iss_hook_reopenlogs']

    def __init__(self, iss_hook_start=None, iss_hook_stop=None, iss_hook_status=None, iss_hook_install=None,
                 iss_hook_uninstall=None, iss_hook_validate=None, iss_hook_notify=None, iss_hook_reopenlogs=None):
        self.iss_hook_start = iss_hook_start
        self.iss_hook_stop = iss_hook_stop
        self.iss_hook_status = iss_hook_status
        self.iss_hook_install = iss_hook_install
        self.iss_hook_uninstall = iss_hook_uninstall
        self.iss_hook_validate = iss_hook_validate
        self.iss_hook_notify = iss_hook_notify
        self.iss_hook_reopenlogs = iss_hook_reopenlogs

    @classmethod
    def from_dict(cls, params):
        return cls(**{f: IssHookTimeLimits.from_dict(params.get(f, {})) for f in cls.__slots__})


class IssHookTimeLimits(object):
    __slots__ = [
        'min_restart_period',
        'max_execution_time',
        'restart_period_backoff',
        'max_restart_period',
        'restart_period_scale',
    ]

    def __init__(
        self,
        min_restart_period=None,
        max_execution_time=None,
        restart_period_backoff=None,
        max_restart_period=None,
        restart_period_scale=None,
    ):
        self.min_restart_period = min_restart_period
        self.max_execution_time = max_execution_time
        self.restart_period_backoff = restart_period_backoff
        self.max_restart_period = max_restart_period
        self.restart_period_scale = restart_period_scale

    @classmethod
    def from_dict(cls, params):
        return cls(**{f: params.get(f) for f in cls.__slots__})


class InvalidInstanceNetworkSettings(Exception):
    pass


class InstanceNetworkInterface(object):
    __slots__ = ['interface_name', 'address', 'hostname']

    def __init__(self, interface_name, address, hostname):
        """
        :type interface_name: six.text_type
        :type address: six.text_type
        :type hostname: six.text_type
        """
        self.interface_name = interface_name
        self.address = address
        self.hostname = hostname

    def __repr__(self):
        return 'InstanceNetworkInterface(interface_name={!r}, address={!r}, hostname={!r})'.format(
            self.interface_name, self.address, self.hostname
        )

    @classmethod
    def from_gencfg_response(cls, name, params):
        """
        :type name: six.text_type
        :type params: dict
        :rtype: InstanceNetworkInterface
        """
        address = params.get('ipv6addr')
        if not address:
            raise InvalidInstanceNetworkSettings('Invalid network settings in gencfg response: no ipv6addr given')
        hostname = params.get('hostname')
        if not hostname:
            raise InvalidInstanceNetworkSettings('Invalid network settings in gencfg response: no hostname given')

        return cls(interface_name=name,
                   address=address,
                   hostname=hostname)


class YasmAgentMonitoringSettings(object):
    __slots__ = ['port']

    def __init__(self, port):
        self.port = port


class JugglerAgentMonitoringSettings(object):
    __slots__ = ['port']

    def __init__(self, port):
        self.port = port


class ContainerMonitoringSettings(object):
    __slots__ = ['yasm', 'juggler']

    def __init__(self, yasm, juggler):
        """
        :type yasm: YasmAgentMonitoringSettings
        :type juggler: JugglerAgentMonitoringSettings
        """
        self.yasm = yasm
        self.juggler = juggler

    @classmethod
    def from_gencfg_response(cls, d):
        yasm = d.get('yasm', {}).get('agent', {}).get('port')
        juggler = d.get('juggler', {}).get('agent', {}).get('port')
        return cls(
            yasm=YasmAgentMonitoringSettings(port=yasm),
            juggler=JugglerAgentMonitoringSettings(port=juggler),
        )


class SlbIpV4Tunnel(object):
    __slots__ = ['ip', 'mtu', 'decapsulator_anycast_address']

    def __init__(self, ip, mtu, decapsulator_anycast_address):
        """
        :type ip: six.text_type
        :type mtu: int
        :type decapsulator_anycast_address: six.text_type
        """
        self.ip = ip
        self.mtu = mtu
        self.decapsulator_anycast_address = decapsulator_anycast_address


class SlbIpV6Tunnel(object):
    __slots__ = ['ip', 'mtu']

    def __init__(self, ip, mtu):
        """
        :type ip: six.text_type
        :type mtu: int
        """
        self.ip = ip
        self.mtu = mtu


class NetworkTunnel(object):
    __slots__ = ['type', 'slb_ipv4_tunnel', 'slb_ipv6_tunnel']

    def __init__(self, type_, slb_ipv4_tunnel, slb_ipv6_tunnel):
        """
        :type type_: six.text_type
        :type slb_ipv4_tunnel: SlbIpV4Tunnel
        :type slb_ipv6_tunnel: SlbIpV6Tunnel
        """
        self.type = type_
        self.slb_ipv4_tunnel = slb_ipv4_tunnel
        self.slb_ipv6_tunnel = slb_ipv6_tunnel

    @classmethod
    def from_gencfg_response(cls, d):
        t = d.get('type')
        if t == 'ipv4':
            return cls(
                type_='ipv4',
                slb_ipv4_tunnel=SlbIpV4Tunnel(
                    ip=d['ip'],
                    mtu=d['mtu'],
                    decapsulator_anycast_address=d['decapsulator_anycast_address'],
                ),
                slb_ipv6_tunnel=None,
            )
        if t == 'ipv6':
            return cls(
                type_='ipv6',
                slb_ipv4_tunnel=None,
                slb_ipv6_tunnel=SlbIpV6Tunnel(
                    ip=d['ip'],
                    mtu=d['mtu'],
                ),
            )
        raise InvalidInstanceNetworkSettings('Unknown SLB tunnel type: {}'.format(t))


class ExtIpV4Tunnel(object):
    __slots__ = ['name', 'ipv4_address', 'local_ipv6_address', 'remote_ipv6_address']

    def __init__(self, name, ipv4_address, local_ipv6_address, remote_ipv6_address):
        self.name = name
        self.ipv4_address = ipv4_address
        self.local_ipv6_address = local_ipv6_address
        self.remote_ipv6_address = remote_ipv6_address

    @classmethod
    def from_gencfg_response(cls, d):
        return cls(
            name=d['name'],
            ipv4_address=d['ipv4_address'],
            local_ipv6_address=d['local_ipv6_address'],
            remote_ipv6_address=d['remote_ipv6_address']
        )


class InstanceNetworkSettings(object):
    __slots__ = ['use_mtn', 'hostname', 'resolv_conf', 'interfaces', 'tunnels', 'backbone_ip', 'ext_tunnels', 'hbf_nat']

    def __init__(self, use_mtn, hostname, resolv_conf, interfaces, tunnels, backbone_ip,
                 ext_tunnels, hbf_nat):
        """
        :type use_mtn: bool
        :type hostname: six.text_type
        :type resolv_conf: six.text_type | None
        :type interfaces: list[InstanceNetworkInterface]
        :type tunnels: list[NetworkTunnel]
        :type backbone_ip: six.text_type
        :type ext_tunnels: list[ExtIpV4Tunnel]
        :type hbf_nat: six.text_type
        """
        self.hostname = hostname
        self.use_mtn = use_mtn
        self.hbf_nat = hbf_nat
        self.resolv_conf = resolv_conf
        self.interfaces = interfaces
        self.tunnels = tunnels
        self.backbone_ip = backbone_ip
        self.ext_tunnels = ext_tunnels

    @classmethod
    def from_gencfg_response(cls, d):
        """
        :type d: dict
        :rtype: InstanceNetworkSettings
        """
        resolv_conf = d.get('resolv.conf')
        dicts = d.get('interfaces', {})
        hostname = dicts.get('backbone', {}).get('hostname')
        backbone_ip = dicts.get('backbone', {}).get('ipv6addr')
        interfaces = [InstanceNetworkInterface.from_gencfg_response(n, i) for n, i in six.iteritems(dicts)]
        tunnel_dicts = d.get('slb', [])
        tunnels = [NetworkTunnel.from_gencfg_response(t) for t in tunnel_dicts]
        ext_tunnels_dicts = d.get('tunnels', [])
        ext_tunnels = [ExtIpV4Tunnel.from_gencfg_response(t) for t in ext_tunnels_dicts]
        return cls(
            use_mtn=False,
            hostname=hostname,
            resolv_conf=resolv_conf,
            interfaces=interfaces,
            tunnels=tunnels,
            backbone_ip=backbone_ip,
            ext_tunnels=ext_tunnels,
            hbf_nat='enabled',
        )


class ContainersSettings(object):
    __slots__ = ['set_slot_properties']

    def __init__(self, set_slot_properties):
        """
        :type set_slot_properties: six.text_type
        """
        self.set_slot_properties = set_slot_properties


class DiskQuotaSettings(object):
    __slots__ = ['policy', 'root_fs_snapshot_quota_bytes', 'work_dir_snapshot_quota_bytes']

    def __init__(self, policy, root_fs_snapshot_quota_bytes, work_dir_snapshot_quota_bytes):
        """
        :type policy: six.text_type
        :type root_fs_snapshot_quota_bytes: int
        :type work_dir_snapshot_quota_bytes: int
        """
        self.policy = policy
        self.root_fs_snapshot_quota_bytes = root_fs_snapshot_quota_bytes
        self.work_dir_snapshot_quota_bytes = work_dir_snapshot_quota_bytes


class GencfgVolume(object):
    __slots__ = ['uuid', 'volumes_storage', 'mount_point', 'quota_bytes', 'symlinks']

    def __init__(self, uuid, volumes_storage, mount_point, quota_bytes, symlinks):
        """
        :type uuid: six.text_type
        :type volumes_storage: six.text_type
        :type mount_point: six.text_type
        :type symlinks: list[six.text_type]
        :type quota_bytes: int
        """
        self.uuid = uuid
        self.volumes_storage = volumes_storage
        self.mount_point = mount_point
        self.quota_bytes = quota_bytes
        self.symlinks = symlinks

    def __repr__(self):
        return 'GencfgVolume(uuid={!r}, volumes_storage={!r}, mount_point={!r}, symlinks={!r})'.format(
            self.uuid, self.volumes_storage, self.mount_point, self.symlinks
        )

    @classmethod
    def from_gencfg_response(cls, d):
        """
        :type d: dict
        :rtype: GencfgVolume
        """
        return cls(uuid=d['uuid'],
                   volumes_storage=d['dom0_mount_point_root'],
                   mount_point=d['guest_mount_point'],
                   quota_bytes=d['quota'],
                   symlinks=d['symlinks'])


class InstanceVolumes(object):
    __slots__ = ['use_gencfg_volumes', 'gencfg_volumes', 'root_volume', 'work_dir_volume']

    def __init__(self, use_gencfg_volumes, gencfg_volumes, root_volume, work_dir_volume):
        """
        :type use_gencfg_volumes: bool
        :type gencfg_volumes: list[GencfgVolume]
        :type root_volume: GencfgVolume
        :type work_dir_volume: GencfgVolume
        """
        self.use_gencfg_volumes = use_gencfg_volumes
        self.gencfg_volumes = gencfg_volumes
        self.root_volume = root_volume
        self.work_dir_volume = work_dir_volume

    def __repr__(self):
        return (
            'InstanceVolumes(use_gencfg_volumes={!r}, gencfg_volumes={!r}, root_volume={!r}, work_dir={!r})'.format(
                self.use_gencfg_volumes, self.gencfg_volumes, self.root_volume, self.work_dir_volume)
        )

    @classmethod
    def from_gencfg_response(cls, d):
        """
        :type d: list[dict]
        :rtype: InstanceVolumes
        """
        volumes = []
        root_volume = None
        work_dir = None
        for i in d:
            v = GencfgVolume.from_gencfg_response(i)
            if i.get('mount_point_workdir'):
                work_dir = v
            elif v.mount_point == '/':
                root_volume = v
            else:
                volumes.append(v)
        return cls(use_gencfg_volumes=False,
                   gencfg_volumes=volumes,
                   root_volume=root_volume,
                   work_dir_volume=work_dir)


class GencfgTopology(object):
    __slots__ = ['release', 'revision', 'group_name']

    def __init__(self, release, revision, group_name):
        """
        :type release: six.text_type | types.NoneType
        :type revision: six.text_type | types.NoneType
        :type group_name: six.text_type | types.NoneType
        """
        self.release = release
        self.revision = revision
        self.group_name = group_name


class Instance(object):
    """
    Instance description
    """
    __slots__ = ['host', 'port', 'power', 'tags', 'short_hostname', 'shard', 'name', 'shard_info', 'virtual_shard',
                 'conf', 'gencfg_group', 'resource_limits', 'ipv4_address', 'ipv6_address', 'instance_cls',
                 'hooks_time_limits', 'annotated_ports', 'meta_data', 'hooks_resource_limits', 'location', 'dc',
                 'ports', 'network_settings', 'volumes', 'containers_settings', 'monitoring_settings',
                 'gencfg_topology', 'sysctl_settings', 'disk_quota_settings', 'slot', 'yp_allocation']

    def __init__(self, host, port, power=0, tags=None, shard=None, name=None, shard_info=None, virtual_shard=None,
                 conf=None, gencfg_group=None, limits=None, ipv4_address=None, ipv6_address=None, instance_cls=None,
                 hooks_time_limits=None, annotated_ports=None, meta_data=None, hooks_resource_limits=None,
                 location=None, dc=None, ports=None, network_settings=None, volumes=None, containers_settings=None,
                 monitoring_settings=None, gencfg_topology=None, sysctl_settings=None, disk_quota_settings=None,
                 slot='', yp_allocation=None, **_):
        """
        :param host: хост инстанса
        :type host: str | six.text_type
        :param int port: порт инстанса
        :param power: мощность инстанса
        :type power: int | float
        :param list tags: список тегов
        :param str shard: название шарда
        :param BsconfigShard shard_info: информация про шард
        :param str virtual_shard: "виртуальный шард" -- строка по которой инстансу сопоставляется реальный
         шард при разворачивании шардмапа, например, 'primus-Rus-1-3-5-0000000000' => 'primus-Rus-1-3-5-1431890942'
        :param conf: имя конфигурации, в которую входит инстанс
        :type conf: str | six.text_type
        :param str gencfg_group: имя группы gencfg, в которую входит инстанс
        :param ContainerResourceLimits limits: ограничения на использование инстансом ресурсов сервера
        :param str ipv4_address: IPv4-адрес хоста
        :param str ipv6_address: IPv6-адрес хоста
        :param str instance_cls: название класса ISS инстанса
        :param hooks_time_limits: таймауты запуска status hook'а
        :type hooks_time_limits: Iss3HooksTimeLimits
        :param hooks_resource_limits: ограничения ресурсов и настройки porto для iss-хуков
        :type hooks_resource_limits: Iss3HooksResourceLimits
        :param annotated_ports: словарь именованных портов инстанса
        :type annotated_ports: dict[str | six.text_type, sepelib.yandex.nanny.AnnotatedPort]
        :type meta_data: MetaData
        :type ports: list[sepelib.yandex.nanny.AnnotatedPort]
        :type network_settings: InstanceNetworkSettings
        :type monitoring_settings: ContainerMonitoringSettings
        :type volumes: InstanceVolumes
        :type topology: GencfgTopology
        :type sysctl_settings: alemate.lib.nanny.ServiceInstances.SysctlSettings
        :type slot: six.text_type
        :type yp_allocation: alemate.lib.nanny.ServiceInstances.YpPodsAllocation
        """
        self.host = host
        self.short_hostname = host.split('.', 1)[0]
        self.port = port
        self.slot = slot or '{}:{}'.format(self.host, self.port)
        self.power = power
        self.tags = tags if tags is not None else []
        self.shard = shard
        self.shard_info = shard_info
        self.virtual_shard = virtual_shard
        self.conf = conf
        self.gencfg_group = gencfg_group
        self.resource_limits = limits or ContainerResourceLimits()
        self.instance_cls = instance_cls
        self.hooks_time_limits = hooks_time_limits
        self.hooks_resource_limits = hooks_resource_limits
        self.ipv4_address = ipv4_address
        self.ipv6_address = ipv6_address
        self.name = '{conf}@{host}:{port}'.format(conf=conf, host=host, port=port)
        self.annotated_ports = annotated_ports or {}
        self.meta_data = meta_data or MetaData(annotations={})
        self.location = location
        self.dc = dc
        self.ports = ports or []
        self.volumes = volumes or InstanceVolumes(use_gencfg_volumes=False,
                                                  gencfg_volumes=[],
                                                  root_volume=GencfgVolume(
                                                      uuid=None,
                                                      volumes_storage=None,
                                                      mount_point='/',
                                                      quota_bytes=0,
                                                      symlinks=[]
                                                  ),
                                                  work_dir_volume=GencfgVolume(
                                                      uuid=None,
                                                      volumes_storage=None,
                                                      mount_point='',
                                                      quota_bytes=0,
                                                      symlinks=[]
                                                  ))
        self.containers_settings = containers_settings or ContainersSettings(set_slot_properties='NONE')
        self.gencfg_topology = gencfg_topology or GencfgTopology(release=None, revision=None, group_name=None)
        self.network_settings = network_settings or InstanceNetworkSettings(use_mtn=False,
                                                                            hostname='',
                                                                            resolv_conf=None,
                                                                            interfaces=[],
                                                                            tunnels=[],
                                                                            backbone_ip='',
                                                                            ext_tunnels=[],
                                                                            hbf_nat='enabled')
        self.monitoring_settings = monitoring_settings or ContainerMonitoringSettings(
            YasmAgentMonitoringSettings(port=None),
            JugglerAgentMonitoringSettings(port=None)
        )
        self.sysctl_settings = sysctl_settings
        self.disk_quota_settings = disk_quota_settings or DiskQuotaSettings(policy=DISK_QUOTA_POLICY_DISABLED,
                                                                            root_fs_snapshot_quota_bytes=0,
                                                                            work_dir_snapshot_quota_bytes=0)
        self.yp_allocation = yp_allocation

    def as_string(self):
        return "{}:{}@{}".format(self.short_hostname, self.port, self.conf)

    @classmethod
    def from_dict(cls, params):
        if 'limits' in params:
            params['limits'] = ContainerResourceLimits.from_nanny_response(params['limits'])
        return cls(**params)

    @staticmethod
    def make_topology_from_tags(tags):
        """
        :type tags: list[six.text_type]
        :rtype: GencfgTopology
        """
        revision = None
        release = None
        group = None
        for t in tags:
            if t.startswith(TOPOLOGY_GROUP_TAG_PREFIX):
                group = t[len(TOPOLOGY_GROUP_TAG_PREFIX):]
            if t.startswith(TOPOLOGY_VERSION_TAG_PREFIX):
                version = t[len(TOPOLOGY_VERSION_TAG_PREFIX):]
                if version.startswith('trunk-'):
                    release, revision = version.split('-', 1)
                else:
                    release = version
                    revision = None
        return GencfgTopology(release=release, revision=revision, group_name=group)

    @classmethod
    def from_gencfg_response(cls, params):
        volumes = InstanceVolumes.from_gencfg_response(params.get('volumes', []))
        network = InstanceNetworkSettings.from_gencfg_response(params.get('hbf', {}))
        monitoring = ContainerMonitoringSettings.from_gencfg_response(params.get('monitoring', {}))
        limits = params.get('porto_limits') or params.get('limits') or {}
        virtual_shard = params.get('shard_name')
        location_str = params.get('location', '').upper()
        location = Location.__members__.get(location_str)
        dc = params.get('dc', '')
        tags = params.get('tags', [])
        topology = cls.make_topology_from_tags(tags)
        return cls(
            host=params['hostname'],
            port=params['port'],
            ipv4_address=params.get('ipv4addr'),
            ipv6_address=params.get('ipv6addr'),
            tags=tags,
            power=float(params.get('power', 1.0)),
            virtual_shard=virtual_shard if virtual_shard != "none" else None,
            limits=ContainerResourceLimits.from_gencfg_response(limits),
            dc=dc,
            location=location,
            network_settings=network,
            volumes=volumes,
            monitoring_settings=monitoring,
            gencfg_topology=topology,
        )

    def __repr__(self):
        kwargs_str = []
        for attr in self.__slots__:
            kwargs_str.append('{}={!r}'.format(attr, getattr(self, attr, None)))
        return 'Instance({})'.format(kwargs_str)

    def __cmp__(self, other):
        return cmp(self.slot, other.slot)

    def __hash__(self):
        return hash(self.slot)

    def __eq__(self, other):
        return self.slot == other.slot

    __str__ = as_string


class Location(enum.Enum):
    MSK = 1
    SAS = 2
    MAN = 3
    VLA = 4
