# coding: utf-8
import inject
from six.moves.urllib import parse as urlparse

from nanny_repo import repo_api_stub, repo_api_pb2
from nanny_rpc_client import RequestsRpcClient


class INannyRpcClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: NannyRpcClient
        """
        return inject.instance(cls)


class NannyRpcClient(INannyRpcClient):
    DEFAULT_REQUEST_TIMEOUT = 30

    @classmethod
    def from_config(cls, d):
        return cls(url=d['url'], req_timeout=d.get('req_timeout'), token=d.get('token'))

    def __init__(self, url, req_timeout=None, token=None):
        super(NannyRpcClient, self).__init__()
        self.client = RequestsRpcClient(urlparse.urljoin(url, '/api/repo/'),
                                        request_timeout=req_timeout or self.DEFAULT_REQUEST_TIMEOUT,
                                        oauth_token=token)
        self.repo_stub = repo_api_stub.RepoServiceStub(self.client)

    def list_snapshot_state_changes(self, service_id, snapshot_id, limit=100):
        """
        :param six.text_type service_id:
        :param six.text_type  snapshot_id:
        :param int limit:
        :rtype: repo_api_pb2.ListSnapshotStateChangeResponse
        """
        return self.repo_stub.list_snapshot_state_changes(repo_api_pb2.ListSnapshotStateChangeRequest(
            service_id=service_id,
            snapshot_id=snapshot_id,
            limit=limit
        ))

    def get_cleanup_policy(self, service_id):
        """
        :param six.text_type service_id:
        :rtype: repo_api_pb2.GetCleanupPolicyResponse
        """
        return self.repo_stub.get_cleanup_policy(repo_api_pb2.GetCleanupPolicyRequest(
            policy_id=service_id
        ))

    def update_cleanup_policy(self, meta_pb, spec_pb):
        """
        :param repo_pb2.CleanupPolicyMeta meta_pb:
        :param repo_pb2.CleanupPolicySpec spec_pb:
        :rtype: repo_api_pb2.UpdateCleanupPolicyResponse
        """
        return self.repo_stub.update_cleanup_policy(repo_api_pb2.UpdateCleanupPolicyRequest(
            meta=meta_pb,
            spec=spec_pb
        ))

    def get_replication_policy(self, service_id):
        """
        :param six.text_type service_id:
        :rtype: repo_api_pb2.GetReplicationPolicyResponse
        """
        return self.repo_stub.get_replication_policy(repo_api_pb2.GetReplicationPolicyRequest(
            policy_id=service_id
        ))

    def update_replication_policy(self, meta_pb, spec_pb):
        """
        :param repo_pb2.ReplicationPolicyMeta meta_pb:
        :param repo_pb2.ReplicationPolicySpec spec_pb:
        :rtype: repo_api_pb2.UpdateReplicationPolicyResponse
        """
        return self.repo_stub.update_replication_policy(repo_api_pb2.UpdateReplicationPolicyRequest(
            meta=meta_pb,
            spec=spec_pb
        ))

    def has_snapshot_been_active(self, service_id, snapshot_id):
        resp = self.list_snapshot_state_changes(
            service_id=service_id,
            snapshot_id=snapshot_id,
            limit=50
        )
        for state_change in resp.value:
            if state_change.current_state.state == 'ACTIVE':
                return True
        return False
