import json
import logging

import abc
import attr
import enum
import six
import ujson
from boltons import strutils
from google.protobuf import message as proto_message
from typing import Optional, Dict

from awacs.model.errors import ConflictError
from infra.awacs.proto import model_pb2


class OverallStatus(enum.Enum):
    CREATED = 1
    IN_PROGRESS = 2
    FINISHED = 3
    CANCELLED = 4


@attr.s(slots=True, weakref_slot=True, cmp=False)
class Result(object):
    # required
    next_state = attr.ib(type=enum.Enum, repr=True, kw_only=True)

    # optional
    state_descriptions = attr.ib(type=Dict[enum.Enum, six.text_type], default=None, repr=False, kw_only=True)
    description = attr.ib(type=Optional[six.text_type], default=None, repr=True, kw_only=True)
    severity = attr.ib(  # noqa
        type=int,
        default=model_pb2.FB_SEVERITY_INFO,
        converter=lambda s: s if s is not None else model_pb2.FB_SEVERITY_INFO,
        repr=model_pb2.FeedbackMessageSeverity.Name,
        kw_only=True)
    content_pb = attr.ib(type=Optional[proto_message.Message], default=None, repr=True, kw_only=True)

    # computed
    json = attr.ib(type=six.text_type, init=False, default=None, repr=False)

    def __attrs_post_init__(self):
        if self.description is None:
            if self.state_descriptions is not None:
                self.description = self.state_descriptions.get(self.next_state)
        if self.description is None:
            self.description = self.next_state.name.lower().replace(u'_', u' ').capitalize()
        self.json = json_dumps(self.description)

    def translate_state(self, states):
        if self.next_state.name not in states.__members__:
            raise RuntimeError(u"State {} doesn't exist in states {}".format(self.next_state.name, type(states)))
        self.next_state = states[self.next_state.name]
        return self


# old-style, use Result instead
class FeedbackMessage(object):
    def __init__(self, message, pb_error_type=0, content=None):
        """
        :type message: six.text_type
        :type pb_error_type: int
        :type content: dict[six.text_type, Any] | None
        """
        self.message = message
        self.json_message = json_dumps(message)
        self.pb_error_type = pb_error_type
        self.content = content or {}


def json_loads(val):
    try:
        return ujson.loads(val)
    except Exception as e:
        # Remove fallback if this exception doesn't trigger for some time: SWAT-6113
        logging.getLogger(u'ujsonError').exception(u'ujson.loads() failed with "{}", value "{}"'.format(e, val))
        return json.loads(val)


def json_dumps(val):
    try:
        return ujson.dumps(val, sort_keys=True, escape_forward_slashes=False)
    except Exception as e:
        # Remove fallback if this exception doesn't trigger for some time: SWAT-6113
        logging.getLogger(u'ujsonError').exception(u'ujson.dumps() failed with "{}", value "{}"'.format(e, val))
        return json.dumps(val, sort_keys=True)


class ProcessableEntity(six.with_metaclass(abc.ABCMeta, object)):
    __slots__ = (u'pb', u'id', u'context')

    type = None  # type: six.text_type # must be set in inheritors
    state_descriptions = None

    def __init__(self, pb):
        self.pb = pb
        self.id = pb.meta.id
        self.context = self._load_context()

    @property
    def namespace_id(self):
        """
        :rtype: six.text_type
        """
        # property instead of a regular attribute to deal with entities without namespace_id (e.g. namespace)
        return self.pb.meta.namespace_id

    @abc.abstractmethod
    def _get_pb_field(self, pb):
        raise NotImplementedError

    @property
    def _pb_field(self):
        return self._get_pb_field(self.pb)

    def _load_context(self):
        if not self._pb_field.HasField('progress'):
            return {}
        return {k: json_loads(v) for k, v in six.iteritems(self._pb_field.progress.context)}

    @abc.abstractproperty
    def states(self):
        raise NotImplementedError

    @abc.abstractproperty
    def name(self):
        raise NotImplementedError

    @abc.abstractmethod
    def zk_update(self):
        raise NotImplementedError

    def make_result(self, next_state, description=None, severity=None, content_pb=None):
        """
        :type next_state: enum.Enum
        :type description: six.text_type
        :type severity: int
        :type content_pb: proto_message.Message
        :rtype: Result
        """
        return Result(next_state=next_state,
                      description=description,
                      severity=severity,
                      content_pb=content_pb,
                      state_descriptions=getattr(self, u'state_descriptions', None))

    def should_be_cancelled(self):
        """
        :rtype bool
        """
        return self._pb_field.cancelled.value

    def get_cancellation_info(self):
        """
        :rtype model_pb2.CancelledCondition
        """
        return self._pb_field.cancelled

    @property
    def attempts(self):
        """
        :rtype: int
        """
        return self._pb_field.progress.state.attempts

    @property
    def current_progress_state(self):
        """
        :rtype: enum.Enum
        """
        return self.states[self._pb_field.progress.state.id]

    @property
    def current_overall_status(self):
        """
        :rtype: enum.Enum
        """
        if not self._pb_field.status.status:
            return None
        return OverallStatus[self._pb_field.status.status]

    def is_started(self):
        """
        :rtype bool
        """
        return self._pb_field.HasField('progress') and self._pb_field.progress.HasField('state')

    def is_finished(self):
        """
        :rtype bool
        """
        return self.current_overall_status == OverallStatus.FINISHED

    def is_cancelled(self):
        """
        :rtype bool
        """
        return self.current_overall_status == OverallStatus.CANCELLED

    def start(self, initial_state):
        """
        :type initial_state: enum.Enum
        """
        for pb in self.zk_update():
            pb_field = self._get_pb_field(pb)
            pb_field.progress.state.id = initial_state.name
            pb_field.progress.state.entered_at.GetCurrentTime()
            self.pb = pb

    def update_context_and_states(self, status, comment, next_state, feedback_message):
        """
        :type status: enum.Enum | None
        :type comment: six.text_type | None
        :type next_state: enum.Enum | None
        :type feedback_message: FeedbackMessage | Result | None
        """
        updated = False
        from_state = self.current_progress_state
        for pb in self.zk_update():
            pb_field = self._get_pb_field(pb)
            updated = self._set_context(pb_field)
            updated |= self._set_result(pb_field, feedback_message)
            if status and comment:
                updated |= self._set_overall_status(pb_field, status, comment)
            if next_state:
                updated |= self._change_progress_state(pb_field, from_state=from_state, to_state=next_state)
            if not updated:
                break
            self.pb = pb
        return updated

    def save_context(self):
        for pb in self.zk_update():
            pb_field = self._get_pb_field(pb)
            updated = self._set_context(pb_field)
            if not updated:
                break
            self.pb = pb

    @property
    def attempt_number(self):
        return self._get_pb_field(self.pb).progress.state.attempts

    @staticmethod
    def _change_progress_state(pb_field, from_state, to_state):
        """
        :type from_state: enum.Enum
        :type to_state: enum.Enum
        """
        state_pb = pb_field.progress.state
        if state_pb.id != from_state.name:
            raise ConflictError(u'Expected state {}, got {}'.format(from_state.name, state_pb.id))

        if to_state.name == from_state.name:
            state_pb.attempts += 1
        else:
            state_pb.id = to_state.name
            state_pb.entered_at.GetCurrentTime()
            state_pb.attempts = 0
        return True

    def _set_context(self, pb_field):
        updated = False
        for key, value in sorted(six.iteritems(self.context)):
            json_value = json_dumps(value)
            if key not in pb_field.progress.context or pb_field.progress.context[key] != json_value:
                pb_field.progress.context[key] = json_value
                updated = True
        for key in list(pb_field.progress.context):
            if key not in self.context:
                del pb_field.progress.context[key]
                updated = True
        return updated

    @staticmethod
    def _set_result(pb_field, result):
        """
        :type result: FeedbackMessage | Result | None
        """
        if not result:
            # clean up old messages
            if pb_field.feedback.messages:
                del pb_field.feedback.messages[:]
                return True
            return False  # nothing to do

        # clean up old messages
        del pb_field.feedback.messages[:]

        # replace old message with new message
        message_pb = pb_field.feedback.messages.add()
        message_pb.ctime.GetCurrentTime()
        if isinstance(result, Result):
            message_pb.text = result.json
            if result.content_pb is not None:
                error_field_name = strutils.camel2under(result.content_pb.__class__.__name__)
                getattr(message_pb, error_field_name).CopyFrom(result.content_pb)
            if hasattr(message_pb, u'severity'):
                message_pb.severity = result.severity
        else:
            message_pb.text = result.json_message
            if result.pb_error_type:
                message_pb.type = result.pb_error_type
            if result.content:
                for key, value in six.iteritems(result.content):
                    getattr(message_pb, key).CopyFrom(value)
        return True

    @staticmethod
    def _set_overall_status(pb_field, status, comment):
        """
        :type status: enum.Enum
        :type comment: six.text_type
        """
        status_pb = pb_field.status
        if status_pb.status == status.name:
            return False
        status_pb.status = status.name
        status_pb.message = comment
        status_pb.last_transition_time.GetCurrentTime()
        return True


class WithOrder(six.with_metaclass(abc.ABCMeta, ProcessableEntity)):
    __slots__ = ()

    type = u'order'

    def _get_pb_field(self, pb):
        return pb.order


class WithRemoval(six.with_metaclass(abc.ABCMeta, ProcessableEntity)):
    __slots__ = ()

    type = u'removal'

    def _get_pb_field(self, pb):
        return pb.removal


class BaseProcessor(six.with_metaclass(abc.ABCMeta, object)):
    __slots__ = (u'entity',)

    def __init__(self, entity):
        """
        :type entity: ProcessableEntity
        """
        self.entity = entity

    @abc.abstractmethod
    def process(self, ctx):
        """
        :type ctx: context.OpCtx
        :rtype: enum.Enum | FeedbackMessage
        """
        raise NotImplementedError

    @abc.abstractproperty
    def state(self):
        """
        :rtype: enum.Enum
        """
        raise NotImplementedError

    @abc.abstractproperty
    def next_state(self):
        """
        :rtype: enum.Enum
        """
        raise NotImplementedError

    @abc.abstractproperty
    def cancelled_state(self):
        """
        :rtype: enum.Enum
        """
        raise NotImplementedError


class BaseStandaloneProcessorMeta(abc.ABCMeta):
    def __new__(mcs, name, bases, attrs):
        slots = set()
        if u'__slots__' in attrs:
            slots.update(attrs[u'__slots__'])  # preserve slots defined by inheritors
        else:
            attrs[u'__slots__'] = []
        if bases == (object,):
            slots.add(u'__dict__')  # preserve ability for inheritors to be mockable
        attrs[u'__slots__'] = sorted(slots)
        klass = abc.ABCMeta.__new__(mcs, name, bases, attrs)
        return klass


class BaseStandaloneProcessor(six.with_metaclass(BaseStandaloneProcessorMeta, object)):
    def __init__(self):
        raise RuntimeError(u"You don't need to instantiate this class")

    @classmethod
    @abc.abstractmethod
    def process(cls, ctx, entity):
        """
        :type ctx: context.OpCtx
        :type entity: ProcessableEntity
        :rtype: enum.Enum | FeedbackMessage
        """
        raise NotImplementedError

    @classmethod
    @abc.abstractproperty
    def state(cls):  # noqa
        """
        :rtype: enum.Enum
        """
        raise NotImplementedError

    @classmethod
    @abc.abstractproperty
    def next_state(cls):  # noqa
        """
        :rtype: enum.Enum
        """
        raise NotImplementedError

    @classmethod
    @abc.abstractproperty
    def cancelled_state(cls):  # noqa
        """
        :rtype: enum.Enum
        """
        raise NotImplementedError


def is_spec_complete(pb):
    """
    :param pb: from model_pb2
    :rtype: bool
    """
    return not pb.spec.incomplete


def is_order_in_progress(pb, field='order'):
    """
    :type pb: from model_pb2
    :type field: six.text_type
    :rtype: bool
    """
    if not pb.HasField(field):
        return False
    pb_field = getattr(pb, field)
    return pb_field.status.status not in (OverallStatus.FINISHED.name, OverallStatus.CANCELLED.name)


def is_order_cancelled(pb):
    """
    :type pb: from model_pb2
    :rtype: bool
    """
    return pb.HasField('order') and pb.order.status.status == OverallStatus.CANCELLED.name


def can_be_cancelled(pb, processors, field='order'):
    if not pb.HasField(field):
        return False
    field_pb = getattr(pb, field)
    state = field_pb.progress.state.id
    for p in processors:
        if state == p.state.name:
            if p.cancelled_state is not None:
                return True
            return False
    return False


def cancel_order(pb, author, comment, forced=False, field='order'):
    if not pb.HasField(field):
        return False
    pb_field = getattr(pb, field)
    pb_field.cancelled.value = True
    pb_field.cancelled.author = author
    pb_field.cancelled.comment = comment
    pb_field.cancelled.mtime.GetCurrentTime()
    pb_field.cancelled.forced = forced


def needs_removal(pb):
    """
    :type pb: from model_pb2
    :rtype: bool
    """
    if isinstance(pb, (model_pb2.Certificate, model_pb2.CertificateRenewal)):
        return pb.spec.state != model_pb2.CertificateSpec.PRESENT
    if isinstance(pb, model_pb2.L7HeavyConfig):
        return pb.spec.state != model_pb2.L7HeavyConfigSpec.PRESENT
    if isinstance(pb, model_pb2.L3Balancer):
        return False
    if isinstance(pb, (model_pb2.Balancer,
                       model_pb2.Backend,
                       model_pb2.Domain,
                       model_pb2.Namespace,
                       model_pb2.DnsRecord,
                       model_pb2.WeightSection,
                       )):
        return pb.spec.deleted
    if isinstance(pb, (model_pb2.DomainOperation,
                       model_pb2.BalancerOperation,
                       model_pb2.DnsRecordOperation,
                       model_pb2.NamespaceOperation,
                       )):
        return is_spec_complete(pb)
    raise RuntimeError(u'Unknown object: {}'.format(type(pb)))


def has_actionable_spec(pb):
    """
    :type pb: from model_pb2
    :rtype: bool
    """
    return ((is_spec_complete(pb) and not is_order_cancelled(pb))
            or (needs_removal(pb) and not is_order_in_progress(pb)))
