import abc
import enum
import gevent
import six
from google.protobuf.message import Message
from typing import List, Type, Union

from awacs.lib import context
from awacs.lib.order_processor.model import (
    OverallStatus,
    BaseProcessor,
    ProcessableEntity,
    FeedbackMessage,
    BaseStandaloneProcessor,
    Result,
)
from infra.awacs.proto import model_pb2


class StateRunner(object):
    __slots__ = (u'entity_class', u'processing_interval', u'processors', u'initial_state', u'final_state',
                 u'final_cancelled_state', u'states', u'_processors_states')

    MAX_BACKOFF_ATTEMPTS_COUNT = 50
    MAX_PROCESSING_DELAY = 60
    BACKOFF_COEFFICIENT = 1.1

    def __init__(self, entity_class, processing_interval, processors, initial_state, final_state,
                 final_cancelled_state):
        self.entity_class = entity_class  # type: Type[ProcessableEntity]
        self.states = set(entity_class.states)
        self.initial_state = initial_state  # type: enum.Enum
        self.final_state = final_state  # type: enum.Enum
        self.final_cancelled_state = final_cancelled_state  # type: enum.Enum
        self.processors = processors  # type: List[Union[Type[BaseProcessor], Type[BaseStandaloneProcessor]]]
        self.processing_interval = processing_interval
        self._check_processors()
        self._processors_states = self._map_states_to_processors()
        self._check_states()

    def process(self, ctx, pb):
        """
        :type ctx: context.OpCtx
        :type pb: Message
        """
        entity = self.entity_class(pb)
        if entity.is_finished() or entity.is_cancelled():
            ctx.log.debug(u'Status is already %s, nothing to process', entity.current_overall_status.name)
            return
        if not entity.is_started():
            entity.start(self.initial_state)
            entity.update_context_and_states(
                OverallStatus.CREATED,
                comment=u'Assigned initial state "%s"' % self.initial_state.name,
                next_state=self.initial_state,
                feedback_message=Result(next_state=self.initial_state, state_descriptions=entity.state_descriptions))
            ctx.log.debug(u'Assigned initial state "%s"', self.initial_state.name)
            return
        processor_class = self._get_processor_class(ctx, entity)
        processor_name = processor_class.__name__
        if issubclass(processor_class, BaseStandaloneProcessor):
            processor = processor_class  # type: Type[BaseStandaloneProcessor]
        else:
            processor = processor_class(entity)  # type: BaseProcessor

        status = comment = next_progress_state = result = None
        try:
            ctx.log.debug(u'Current state: %s, running %s.process()...',
                          entity.current_progress_state.name, processor_name)
            with ctx.with_forced_timeout(120):
                if issubclass(processor_class, BaseStandaloneProcessor):
                    result = processor.process(ctx, entity)
                else:
                    result = processor.process(ctx)
            if isinstance(result, (Result, entity.states)):
                if isinstance(result, Result):
                    next_progress_state = result.next_state
                else:
                    next_progress_state = result
                    result = None
                assert (next_progress_state in self._processors_states
                        or next_progress_state in (self.final_cancelled_state, self.final_state)), \
                    u'Unknown next state "{}"'.format(next_progress_state)
                ctx.log.debug(u'Processed, next state: %s', next_progress_state.name)
                old_overall_status = entity.current_overall_status.name if entity.current_overall_status else None
                status, comment = self._calculate_overall_status(entity, processor, next_progress_state)
                if status:
                    ctx.log.debug(u'Overall status will be %s (was %s)', status.name, old_overall_status)
            elif isinstance(result, FeedbackMessage):
                ctx.log.debug(u'Got FeedbackMessage, next state remains the same: %s. Message: %s',
                              entity.current_progress_state.name, six.text_type(result.message))
                next_progress_state = entity.current_progress_state
            else:
                raise RuntimeError(u'Processor {} returned unknown value {} with type {}'
                                   .format(processor_name, result, type(result)))
        except (context.CtxTimeoutCancelled, context.CtxTimeoutExceeded, gevent.Timeout, Exception) as e:
            # these specific exceptions are inherited from BaseException, catch them in addition to Exception
            result = self._create_exc_result(ctx, entity, e)
            if entity.should_be_cancelled():
                next_progress_state = processor.state
            raise
        finally:
            entity.update_context_and_states(status, comment, next_progress_state, result)
            sleep_duration = self._get_processing_delay(entity.attempts)
            ctx.log.debug(u'Updated %s: generation=%s, attempt=%s; sleeping for %s seconds before next iteration',
                          self.entity_class.__name__,
                          entity.pb.meta.generation,
                          entity.attempt_number,
                          round(sleep_duration, 2))
            gevent.sleep(sleep_duration)

    def _get_processing_delay(self, attempts):
        backoff = self.BACKOFF_COEFFICIENT ** min(self.MAX_BACKOFF_ATTEMPTS_COUNT, attempts)
        delay = self.processing_interval * backoff
        return min(self.MAX_PROCESSING_DELAY, delay)

    @staticmethod
    def _create_exc_result(ctx, entity, e):
        if hasattr(e, u'response') and hasattr(e.response, u'content'):
            if six.PY2:
                error_message = u'{}; Reason: {}'.format(e.message.decode('utf-8'),
                                                         e.response.content[:500].decode('utf-8'))
            else:
                error_message = u'{}; Reason: {}'.format(e, e.response.content[:500])
        else:
            if six.PY2:
                error_message = six.text_type(e.message.decode('utf-8')) or six.text_type(e)
            else:
                error_message = str(e)
        error = Result(
            description=error_message,
            next_state=entity.current_progress_state,
            severity=model_pb2.FB_SEVERITY_ERROR)
        ctx.log.exception(u'Unexpected exception while running processor for %s: %s: %s',
                          entity.current_progress_state.name, type(e).__name__, error_message)
        return error

    def _check_processors(self):
        for proc in self.processors:
            if isinstance(proc.state, abc.abstractproperty):
                raise AssertionError(u'Processor "{}" must implement attribute "state"'.format(proc.__name__))
            if isinstance(proc.next_state, abc.abstractproperty):
                raise AssertionError(u'Processor "{}" must implement attribute "next_state"'.format(proc.__name__))
            if isinstance(proc.cancelled_state, abc.abstractproperty):
                raise AssertionError(u'Processor "{}" must implement attribute "cancelled_state"'.format(proc.__name__))
            if getattr(proc.process, u'__isabstractmethod__', False):
                raise AssertionError(u'Processor "{}" must implement method "process"'.format(proc.__name__))
            assert proc.state in self.states, u'Processor "{}" has unknown state "{}"'.format(
                proc.__name__, proc.state)
            assert proc.next_state in self.states, u'Processor "{}" has unknown next_state "{}"'.format(
                proc.__name__, proc.next_state)
            assert proc.cancelled_state is None or proc.cancelled_state in self.states, \
                u'Processor "{}" has unknown cancelled_state "{}"'.format(proc.__name__, proc.cancelled_state)

    def _map_states_to_processors(self):
        rv = {}
        for proc in self.processors:
            state = proc.state  # type: enum.Enum  # noqa
            assert state not in rv, u'Duplicate processors for state "{}": "{}" and "{}"'.format(
                state.name, proc.__name__, rv[state].__name__
            )
            rv[state] = proc
        return rv

    def _check_states(self):
        assert self.initial_state in self.states, u'Unknown initial state "{}"'.format(self.initial_state)
        assert self.final_state in self.states, u'Unknown final state "{}"'.format(self.final_state)
        assert self.final_cancelled_state in self.states, \
            u'Unknown final cancelled state "{}"'.format(self.final_cancelled_state)
        assert len({self.initial_state, self.final_state, self.final_cancelled_state}) == 3, \
            u'Initial, final, and final cancelled states cannot match: {} / {} / {}'.format(
                self.initial_state, self.final_state, self.final_cancelled_state)

        processors_states = set(self._processors_states.keys()).union({self.final_state, self.final_cancelled_state})
        states_without_processors = self.states - processors_states
        if states_without_processors:
            raise AssertionError(u'Some states have no corresponding processors: {}'.format(
                u', '.join(sorted(s.name for s in states_without_processors))))

        for proc in self.processors:
            if proc.cancelled_state is None:
                continue
            cancel_processor = self._processors_states[proc.cancelled_state]  # noqa
            if cancel_processor.cancelled_state is not None:
                raise AssertionError(u'Cancel processor "{}" cannot have a cancelled state'.format(
                    cancel_processor.__name__))

    def _calculate_overall_status(self, entity, processor, next_progress_state):
        """
        :type entity: ProcessableEntity
        :type processor: BaseProcessor | BaseStandaloneProcessor
        :type next_progress_state: enum.Enum
        """
        if processor.state == next_progress_state:
            return None, None

        status = comment = None
        if next_progress_state == self.final_cancelled_state:
            status = OverallStatus.CANCELLED
            comment = u'{} {} has been cancelled'.format(entity.name, entity.type)
        elif next_progress_state == self.final_state:
            status = OverallStatus.FINISHED
            comment = u'{} {} has been processed'.format(entity.name, entity.type)
        elif entity.current_overall_status == OverallStatus.CREATED:
            status = OverallStatus.IN_PROGRESS
            comment = u'{} {} is being processed'.format(entity.name, entity.type)

        return status, comment

    def _get_processor_class(self, ctx, entity):
        """
        :type ctx: context.OpCtx
        :type entity: ProcessableEntity
        :rtype: type[BaseProcessor | BaseStandaloneProcessor]
        """
        processor_class = self._processors_states[entity.current_progress_state]
        if entity.should_be_cancelled():
            return self._handle_cancellation(ctx, processor_class, entity)
        else:
            return processor_class

    def _handle_cancellation(self, ctx, processor_class, entity):
        """
        :type ctx: context.OpCtx
        :type entity: ProcessableEntity
        :rtype: type[BaseProcessor | BaseStandaloneProcessor]
        """
        cancelled_cond_pb = entity.get_cancellation_info()
        if cancelled_cond_pb.forced:
            try:
                cancelled_state = entity.states[u'CANCELLING']
            except:
                cancelled_state = None
            action = u'force-cancelled'
        else:
            cancelled_state = processor_class.cancelled_state
            action = u'cancelled'
        if cancelled_state is None:
            ctx.log.debug(u'%s %s was marked as %s by "%s" with comment "%s", '
                          u'but order processor "%s" cannot be cancelled',
                          entity.name, entity.type, action, cancelled_cond_pb.author, cancelled_cond_pb.comment,
                          processor_class.__name__)
            return processor_class
        ctx.log.debug(u'%s %s was %s by "%s" with comment "%s"',
                      entity.name, entity.type, action, cancelled_cond_pb.author, cancelled_cond_pb.comment)
        return self._processors_states[cancelled_state]
