import gevent
import inject
import ipaddress  # noqa
import six
from cachetools.func import ttl_cache
from collections import defaultdict
from six.moves.urllib import parse as urlparse

from awacs.lib import validators
from infra.swatlib.metrics import InstrumentedSession


PUBLIC_NETWORK_MACRO = u'_SLBPUBLICSUPERNETS_'


class IRacktablesClient(object):
    @classmethod
    def instance(cls):
        """
        :rtype: RacktablesClient
        """
        return inject.instance(cls)


class RacktablesClient(IRacktablesClient):
    DEFAULT_URL = u'https://ro.racktables.yandex.net/'
    DEFAULT_TIMEOUT = 10

    def __init__(self, url=None, timeout=None):
        """
        :type url: str
        :type timeout: int
        """
        self.url = url or self.DEFAULT_URL
        self.timeout = timeout or self.DEFAULT_TIMEOUT
        self._session = InstrumentedSession(u'rt')

    def _call(self, method, url, **kwargs):
        with gevent.Timeout(self.timeout):
            resp = self._session.request(method, url, **kwargs)
        resp.raise_for_status()
        return resp

    def ip_belongs_to_macro(self, ip, macro):
        if isinstance(ip, six.binary_type):
            ip = ip.decode('utf-8')
        ip_address = ipaddress.ip_address(ip)
        url = urlparse.urljoin(self.url, '/export/expand-fw-macro.php')
        params = {u'macro': macro}
        ip_nets = self._call(u'GET', url, params=params).text
        for ip_net in ip_nets.splitlines():
            if ip_net and (ip_address in ipaddress.ip_network(ip_net)):
                return True
        return False

    def is_public_ip(self, ip):
        return self.ip_belongs_to_macro(ip, PUBLIC_NETWORK_MACRO)

    @ttl_cache(ttl=300)
    def list_services(self, external_only=False):
        url = urlparse.urljoin(self.url, '/export/slb-api.php')
        params = {}
        if external_only:
            params['external'] = True

        rv = defaultdict(set)
        for ipv in ('v4', 'v6'):
            params['services-list'] = ipv
            content = self._call(u'GET', url, params=params).text
            """
            5.45.196.0/24	id: 5040, name: public VIPs vec.maps.yandex.net
            	5.45.196.61 - vec.maps.yandex.net
            	5.45.196.62 - vec.maps.yandex.net
            	5.45.196.63 - vec.maps.yandex.net
            	5.45.196.64 - vec.maps.yandex.net
            	...
            5.45.202.0/24	id: 8620, name: private Cogent VIPs
            	5.45.202.1 - uaproxy-static.yandex.net
            	5.45.202.2 - uaproxy-static.yandex.net
            	5.45.202.3 - uaproxy-static.yandex.net
            	5.45.202.4 - uaproxy-static.yandex.net
            	5.45.202.5 - mc.yandex.ru
            ...
            """

            for line in content.splitlines():
                splitted = line.strip().split()
                if len(splitted) != 3:
                    continue
                ip, _, fqdn = splitted
                if validators.is_ip_address(ip):
                    rv[fqdn].add(ip)
        return rv
