# coding: utf-8
import inject
import requests

from infra.swatlib.auth import staff


class IStaffClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: StaffClient
        """
        return inject.instance(cls)


class StaffClient(object):
    _USE_EXPIRED_CACHE_ERRORS = (requests.HTTPError, requests.ConnectionError,
                                 requests.Timeout, staff.StaffError)

    _LOGIN_TO_GROUP_IDS = 'login_to_group_ids'

    def __init__(self, staff_client, cache=None):
        """
        :type staff_client: sepelib.yandex.staff.StaffClient
        """
        self._staff_client = staff_client
        self._cache = cache

    @classmethod
    def from_config(cls, d, cache=None):
        return cls(staff_client=staff.StaffClient.from_config(d), cache=cache)

    def _do_resolve_login_to_group_ids(self, login):
        return staff.get_group_ids(self._staff_client, login)

    def resolve_login_to_group_ids(self, login, use_cache=True, use_expired_cache=False):
        use_cache = use_cache and self._cache is not None
        cached_group_ids = None

        if use_cache:
            cached_group_ids, expired = self._cache.get(self._LOGIN_TO_GROUP_IDS, login)
            if cached_group_ids is not None and not expired:
                return cached_group_ids

        try:
            group_ids = self._do_resolve_login_to_group_ids(login)
        except self._USE_EXPIRED_CACHE_ERRORS:
            if use_expired_cache and cached_group_ids is not None:
                return cached_group_ids
            else:
                raise
        else:
            if self._cache:
                self._cache.set(self._LOGIN_TO_GROUP_IDS, login, group_ids)
            return group_ids

    def get_staff_id_for_abc_role(self, role_slug):
        """
        :param str role_slug:
        :rtype: str
        """
        spec = {'url': role_slug}
        data = self._staff_client.list_groups(spec, fields=('id',), one=True)
        return str(data['id'])

    def get_groups_by_ids(self, group_ids, fields=('id', 'url', 'type')):
        """

        :type group_ids: list[int]
        :type fields: list[str] | tuple[str]
        :rtype: dict[int, dict]
        """
        spec = {
            'id': ",".join([str(g_id) for g_id in group_ids]),
            '_limit': len(group_ids)
        }
        data = self._staff_client.list_groups(spec, fields=fields)
        return {int(v['id']): v for v in data.get('result', [])}

    def get_group_members(self, group_id):
        """
        :param str group_id: staff group or abc role id
        :rtype list[six.text_type]
        :return: human (or at least non-robot) members of target staff group or abc group or role
        """
        return staff.get_group_member_logins(self._staff_client, group_id, skip_robots=True)

    def list_persons(self, logins, fields=('id', 'name')):
        spec = {
            'login': ','.join(logins)
        }
        return self._staff_client.list_persons(spec, fields)['result']

    def get_abc_roles(self, abc_service_id):
        """
        :type abc_service_id: int
        """
        spec = {
            'parent.service.id': abc_service_id,
            'type': 'servicerole',
        }
        fields = ('id', 'role_scope')
        return self._staff_client.list_groups(spec, fields=fields)['result']

    def filter_dismissed(self, logins):
        """
        :type logins: list[six.text_type]
        """
        spec = {
            'official.is_dismissed': True,
            'login': logins
        }
        fields = ('login',)
        return self._staff_client.list_persons(spec, fields=fields)['result']
