import six
from typing import Iterable, Tuple


def quote_join_sorted(iterable):
    """
    :type iterable: Iterable
    :rtype: six.text_type
    """
    return u'", "'.join(sorted(iterable))


def removeprefix(text, prefix):
    """
    str.removeprefix() from python3.9

    :type text: six.text_type
    :type prefix: six.text_type
    :rtype: six.text_type
    """
    if text.startswith(prefix):
        return text[len(prefix):]
    return text


def removesuffix(text, suffix):
    """
    str.removesuffix() from python3.9

    :type text: six.text_type
    :type suffix: six.text_type
    :rtype: six.text_type
    """
    if text.endswith(suffix):
        return text[:-len(suffix)]
    return text


def flatten_full_id(current_namespace_id, full_id):
    """
    :type current_namespace_id: six.text_type
    :type full_id: Tuple[six.text_type, six.text_type]
    :rtype: six.text_type
    """
    assert isinstance(current_namespace_id, six.string_types)
    assert isinstance(full_id, tuple) and len(full_id) == 2
    if full_id[0] == current_namespace_id:
        return full_id[1]
    else:
        return u'/'.join(full_id)


def flatten_full_id2(full_id):
    """
    :type full_id: Tuple[six.text_type, six.text_type]
    :rtype: six.text_type
    """
    assert isinstance(full_id, tuple) and len(full_id) == 2
    return u'{}/{}'.format(full_id[0], full_id[1])


def join_full_uids(iterable):
    """
    :type iterable: Iterable[tuple[six.text_type, six.text_type]]
    :rtype: six.text_type
    """
    return quote_join_sorted(flatten_full_id2(uid) for uid in iterable)


def flatten_full_ids(current_namespace_id, full_ids):
    """
    :type current_namespace_id: six.text_type
    :type full_ids: Iterable[Tuple[six.text_type, six.text_type]]
    :rtype: Iterable[six.text_type]
    """
    for full_id in full_ids:
        yield flatten_full_id(current_namespace_id, full_id)


def join_full_uids2(namespace_id, ids):
    return quote_join_sorted(flatten_full_ids(namespace_id, ids))


def to_full_ids(current_namespace_id, ids):
    """
    :type current_namespace_id: six.text_type
    :type ids: Iterable[six.text_type]
    :rtype: Iterable[six.text_type]
    """
    for id_ in ids:
        yield to_full_id(current_namespace_id, id_)


def to_full_id(current_namespace_id, id_):
    """
    :type current_namespace_id: six.text_type
    :type id_: six.text_type
    :rtype: Tuple[six.text_type, six.text_type]
    """
    assert isinstance(current_namespace_id, six.string_types)
    assert isinstance(id_, six.string_types)
    if u'/' in id_:
        assert id_.count(u'/') == 1, u'More than one slash: {}'.format(id_)
        return tuple(id_.split(u'/'))
    else:
        return current_namespace_id, id_
