import six

from awacs.lib.vectors.version import L3BalancerVersion, DnsRecordVersion, BackendVersion, EndpointSetVersion
from awacs.model import db
from infra.awacs.proto import model_pb2


def must_get_dns_record_version(cache, namespace_id, dns_record_id):
    """
    :type cache: AwacsCache
    :type namespace_id: six.text_type
    :type dns_record_id: six.text_type
    :rtype: DnsRecordVersion
    """
    return DnsRecordVersion.from_pb(cache.must_get_dns_record(namespace_id=namespace_id,
                                                              dns_record_id=dns_record_id))


def must_get_l3_balancer_version(cache, namespace_id, l3_balancer_id):
    """
    :type cache: AwacsCache
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :rtype: L3BalancerVersion
    """
    return L3BalancerVersion.from_pb(cache.must_get_l3_balancer(namespace_id=namespace_id,
                                                                l3_balancer_id=l3_balancer_id))


def get_backend_versions(cache, namespace_id):
    """
    :type cache: AwacsCache
    :param six.text_type namespace_id:
    :rtype: dict[(six.text_type, six.text_type), BackendVersion]
    """
    return {(backend_pb.meta.namespace_id, backend_pb.meta.id): BackendVersion.from_pb(backend_pb)
            for backend_pb in cache.list_all_backends(namespace_id=namespace_id)}


def get_endpoint_set_versions(cache, namespace_id):
    """
    :type cache: AwacsCache
    :param six.text_type namespace_id:
    :rtype: dict[(six.text_type, six.text_type), EndpointSetVersion]
    """
    return {(endpoint_set_pb.meta.namespace_id, endpoint_set_pb.meta.id): EndpointSetVersion.from_pb(endpoint_set_pb)
            for endpoint_set_pb in cache.list_all_endpoint_sets(namespace_id=namespace_id)}


def must_get_dns_record_revision_spec_with_cache(cache, namespace_id, dns_record_id, version):
    """
    :type cache: AwacsCache
    :type namespace_id: six.text_type
    :type dns_record_id: six.text_type
    :type version: six.text_type
    :rtype: model_pb2.DnsRecordSpec
    """
    return db.find_dns_record_revision_spec(version,
                                            current_pb=cache.must_get_dns_record(namespace_id, dns_record_id))


def must_get_l3_balancer_revision_spec_with_cache(cache, namespace_id, l3_balancer_id, version):
    """
    :type cache: AwacsCache
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :type version: six.text_type
    :rtype: model_pb2.L3BalancerSpec
    """
    return db.find_l3_balancer_revision_spec(version,
                                             current_pb=cache.must_get_l3_balancer(namespace_id, l3_balancer_id))


def must_get_endpoint_set_revision_with_cache(cache, namespace_id, endpoint_set_id, version):
    """
    :type cache: AwacsCache
    :type namespace_id: six.text_type
    :type endpoint_set_id: six.text_type
    :type version: six.text_type
    :rtype: model_pb2.EndpointSet
    """
    return db.find_endpoint_set_revision(version,
                                         current_pb=cache.must_get_endpoint_set(namespace_id, endpoint_set_id))
