import abc
import inject
import six

from awacs.lib.vectors.vector import Vector, VectorMeta
from awacs.model import cache


class DiscoveredVector(six.with_metaclass(VectorMeta, Vector)):
    __slots__ = ()

    __main_version_class__ = None
    __version_classes__ = None

    @classmethod
    @abc.abstractmethod
    def from_cache(cls, namespace_id, main_id):
        raise NotImplementedError

    def find_versions_to_update_in_state(self, vectors):
        """
        Inheritors must call super() on this method to collect all versions

        :type vectors: state_handler.vectors
        :rtype: set[Versions], set[Versions]
        """
        versions_to_add = set()
        versions_to_delete = set()

        # add the latest main version
        if self.main_version > vectors.current.main_version:
            versions_to_add.add(self.main_version)

        return versions_to_add, versions_to_delete


class DiscoveredVectorWithBackends(six.with_metaclass(VectorMeta, DiscoveredVector)):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    __main_version_class__ = None
    __version_classes__ = None

    def __init__(self, backend_versions, endpoint_set_versions, *args, **kwargs):
        """
        :type backend_versions: dict[(six.text_type, six.text_type), BackendVersion]
        :type endpoint_set_versions: dict[(six.text_type, six.text_type), EndpointSetVersion]
        """
        self.backend_versions = backend_versions
        self.endpoint_set_versions = endpoint_set_versions

        # super() resolves to Vector instead of DiscoveredVector here, this is a workaround
        DiscoveredVector.__init__(self, *args, **kwargs)

    @classmethod
    @abc.abstractmethod
    def from_cache(cls, namespace_id, main_id):
        raise NotImplementedError

    @abc.abstractmethod
    def get_included_backends(self, version):
        """
        Specific to inheritor (L3, L7, etc)
        """
        raise NotImplementedError

    def find_versions_to_update_in_state(self, vectors):
        """
        Inheritors must call super() on this method to collect all versions

        This finds versions of objects that should be added/deleted from the state (corresponds to vectors.current)

        :type vectors: state_handler.vectors
        :rtype: set[Versions], set[Versions]
        """
        versions_to_add, versions_to_delete = \
            super(DiscoveredVectorWithBackends, self).find_versions_to_update_in_state(vectors)

        latest_backend_versions, latest_es_versions = self._get_latest_included_backend_and_es_versions(vectors)
        versions_to_add |= self._get_backend_and_es_versions_to_add(vectors,
                                                                    latest_backend_versions,
                                                                    latest_es_versions)
        versions_to_delete |= self._get_backend_and_es_versions_to_delete(vectors, latest_backend_versions)

        return versions_to_add, versions_to_delete

    def _get_latest_included_backend_and_es_versions(self, vectors):
        """
        Collect backends and ES that are included in any of the vectors, and match them with the latest version

        :type vectors: state_handler.vectors
        """
        # find all backends that are included in all available main versions
        included_backend_ids = set()
        seen_versions = set()
        for version in (self.main_version,
                        vectors.current.main_version,
                        vectors.valid.main_version,
                        vectors.in_progress.main_version,
                        vectors.active.main_version):
            if version is not None and version not in seen_versions:
                included_backend_ids.update(self.get_included_backends(version))
                seen_versions.add(version)

        all_included_backend_versions = {}
        all_included_endpoint_set_versions = {}
        for full_backend_id in included_backend_ids:
            if full_backend_id in self.backend_versions:
                all_included_backend_versions[full_backend_id] = self.backend_versions[full_backend_id]
            if full_backend_id in self.endpoint_set_versions:  # ES id always matches backend id
                all_included_endpoint_set_versions[full_backend_id] = self.endpoint_set_versions[full_backend_id]
        return all_included_backend_versions, all_included_endpoint_set_versions

    @staticmethod
    def _get_backend_and_es_versions_to_add(vectors, latest_backend_versions, latest_es_versions):
        versions = set()
        for full_backend_id, latest_backend_version in six.iteritems(latest_backend_versions):
            current_backend_version = vectors.current.backend_versions.get(full_backend_id)
            if current_backend_version is None:
                if not latest_backend_version.deleted and not latest_backend_version.incomplete:
                    versions.add(latest_backend_version)
            elif latest_backend_version > current_backend_version:
                versions.add(latest_backend_version)

            if full_backend_id not in latest_es_versions:  # ES id always matches backend id
                continue
            latest_es_version = latest_es_versions.get(full_backend_id)
            current_es_version = vectors.current.endpoint_set_versions.get(full_backend_id)
            if current_es_version is None:
                if (not latest_es_version.deleted and
                        not latest_es_version.incomplete and
                        not latest_backend_version.deleted):
                    versions.add(latest_es_version)
            elif latest_es_version > current_es_version:
                versions.add(latest_es_version)
        return versions

    def _get_backend_and_es_versions_to_delete(self, vectors, latest_included_backend_versions):
        versions = set()
        if not vectors.active.main_version:
            # until any vector is active, we can't remove anything from state
            return versions

        # If a vector is neither in progress nor activated, we don't want to touch it yet --
        # the transport might be processing it right now. Let's wait until it's activated.
        # See https://st.yandex-team.ru/AWACS-828#60b8d40fb2e6956bdbdf812f for details.
        non_active_vectors = []
        for vector in (vectors.current, vectors.valid, vectors.in_progress):
            if vector != vectors.active:
                non_active_vectors.append(vector)

        # first, collect backends that are included in active spec
        included_active_backend_ids = self.get_included_backends(vectors.active.main_version)
        # then go through all active backend and ES versions and find something to delete
        for field in (u'backend_versions', u'endpoint_set_versions'):
            active_versions = vectors.active.must_get_version_dict(field)
            for active_id, active_version in six.iteritems(active_versions):

                # if it's present in active spec, it's useful
                if active_id in included_active_backend_ids:
                    continue

                # if it's present in the latest vector and not deleted there, it's also useful
                if (active_id in latest_included_backend_versions and
                        not latest_included_backend_versions[active_id].deleted):
                    continue

                # otherwise, check whether any non-active vector is using it
                for vector in non_active_vectors:
                    non_active_versions = vector.must_get_version_dict(field)
                    # if version is different from active
                    if active_id in non_active_versions and non_active_versions[active_id] != active_version:
                        break  # then it's useful
                else:  # otherwise it means none of the non-active vectors use it, so it's safe to delete from state
                    versions.add(active_version)

        # TODO: move logic for finding orphan ES here (StateHandler._remove_orphan_revs)

        return versions
