import functools

import abc

import six
from six.moves import zip

from awacs.lib.strutils import flatten_full_id2


REGISTRY = {}


def choose_newer_version(version_1, version_2):
    """
    :type version_1: Optional[Version]
    :type version_2: Optional[Version]
    :rtype: Optional[Version]
    """
    if not version_2:
        return version_1
    if not version_1:
        return version_2
    if version_1.ctime >= version_2.ctime:
        return version_1
    if version_1 == version_2:
        return version_1
    return version_2


class VersionMeta(abc.ABCMeta):
    def __new__(mcs, name, bases, attrs):
        if name in REGISTRY:
            raise RuntimeError('Version {} already exists: {!r}'.format(name, REGISTRY[name]))

        if object not in bases:
            assert attrs['name']
            assert attrs['pb_field_name']
            assert attrs['vector_field_name']
            assert attrs['can_be_deleted'] is not None
            assert attrs['can_be_incomplete'] is not None

        slots = set()
        if '__slots__' in attrs:
            slots.update(attrs['__slots__'])  # we also want to preserve slots defined by users
        if bases == (object,):
            slots.add('__dict__')  # and we want our instances to be mockable (very useful in tests)
        attrs['__slots__'] = sorted(slots)

        klass = REGISTRY[name] = abc.ABCMeta.__new__(mcs, name, bases, attrs)

        return klass


@functools.total_ordering
class Version(six.with_metaclass(VersionMeta, object)):
    __slots__ = ('ctime', 'id', 'version', 'deleted', 'incomplete', '_weak_hash', '_hash')
    name = None
    pb_field_name = None
    vector_field_name = None
    can_be_deleted = None
    can_be_incomplete = None

    def __init__(self, ctime, version_id, version, deleted, incomplete):
        """
        :type ctime: int
        :type version_id: tuple[six.text_type, six.text_type]
        :type version: six.text_type
        :type deleted: bool
        :type incomplete: bool
        """
        self.ctime = ctime
        self.id = version_id
        self.version = version
        self.deleted = deleted
        self.incomplete = incomplete
        self._weak_hash = None
        self._hash = None

    def _get_eq_fields(self):
        yield self.ctime
        yield self.id
        yield self.version
        if self.can_be_deleted:
            yield self.deleted
        if self.can_be_incomplete:
            yield self.incomplete

    def __eq__(self, other):
        """
        :rtype: bool
        """
        if other is None:
            return False
        if not isinstance(other, type(self)):
            return False
        return all(v1 == v2 for (v1, v2) in zip(self._get_eq_fields(), other._get_eq_fields()))

    def __ne__(self, other):
        # https://bugs.python.org/issue25732
        return not self.__eq__(other)

    def __lt__(self, other):
        """
        :rtype: bool
        """
        if other is None:
            return False
        if not isinstance(other, type(self)):
            raise NotImplementedError
        if self.ctime == other.ctime:
            return self.id < other.id
        else:
            return self.ctime < other.ctime

    def __hash__(self):
        """
        :rtype: int
        """
        if self._hash is None:
            self._hash = hash(tuple(self._get_eq_fields()))
        return self._hash

    def get_weak_hash(self):
        """
        :rtype: six.text_type
        """
        if self._weak_hash is None:
            self._weak_hash = u'{}:{}'.format(u':'.join(self.id), self.version).encode('utf-8')
        return self._weak_hash

    @classmethod
    def is_deleted(cls, pb):
        """
        :rtype: bool
        """
        assert cls.can_be_deleted
        return pb.spec.deleted

    @classmethod
    def from_pb(cls, pb):
        """
        :rtype: cls
        """
        return cls(ctime=pb.meta.mtime.ToMicroseconds(),
                   version_id=(pb.meta.namespace_id, pb.meta.id),
                   version=pb.meta.version,
                   deleted=cls.is_deleted(pb) if cls.can_be_deleted else False,
                   incomplete=pb.spec.incomplete if cls.can_be_incomplete else False,
                   )

    @classmethod
    def from_rev_status_pb(cls, version_id, pb):
        """
        :rtype: cls
        """
        assert isinstance(version_id, tuple) and len(version_id) == 2
        return cls(ctime=pb.ctime.ToMicroseconds(),
                   version_id=version_id,
                   version=pb.revision_id,
                   deleted=pb.deleted if cls.can_be_deleted else False,
                   incomplete=pb.incomplete if cls.can_be_incomplete else False,
                   )

    def __repr__(self):
        """
        :rtype: six.text_type
        """
        suffix = ''
        if self.can_be_deleted:
            suffix += ', d={}'.format(self.deleted)
        if self.can_be_incomplete:
            suffix += ', i={}'.format(self.incomplete)
        return u'{}(id={}, v={}, ctime={}{})'.format(self.name,
                                                     flatten_full_id2(self.id),
                                                     self.version[:8],
                                                     self.ctime,
                                                     suffix)


class BackendVersion(Version):
    __slots__ = ()
    name = u'Back'
    pb_field_name = u'backends'
    vector_field_name = u'backend_versions'
    can_be_deleted = True
    can_be_incomplete = False


class EndpointSetVersion(Version):
    __slots__ = ()
    name = u'ESet'
    pb_field_name = u'endpoint_sets'
    vector_field_name = u'endpoint_set_versions'
    can_be_deleted = True
    can_be_incomplete = False


class L3BalancerVersion(Version):
    __slots__ = ()
    name = u'L3Bal'
    pb_field_name = u'l3_balancer'
    vector_field_name = u'l3_balancer_version'
    can_be_deleted = False
    can_be_incomplete = False


class L7BalancerVersion(Version):
    __slots__ = ()
    name = u'L7Bal'
    pb_field_name = u'balancer'
    vector_field_name = u'balancer_version'
    can_be_deleted = True
    can_be_incomplete = True


class UpstreamVersion(Version):
    __slots__ = ()
    name = u'Up'
    pb_field_name = u'upstreams'
    vector_field_name = u'upstream_versions'
    can_be_deleted = True
    can_be_incomplete = False


class KnobVersion(Version):
    __slots__ = ()
    name = u'Knob'
    pb_field_name = u'knobs'
    vector_field_name = u'knob_versions'
    can_be_deleted = True
    can_be_incomplete = False


class CertVersion(Version):
    __slots__ = ()
    name = u'Cert'
    pb_field_name = u'certs'
    vector_field_name = u'cert_versions'
    can_be_deleted = True
    can_be_incomplete = True


class DomainVersion(Version):
    __slots__ = ()
    name = u'Domain'
    pb_field_name = u'domains'
    vector_field_name = u'domain_versions'
    can_be_deleted = True
    can_be_incomplete = True


class DnsRecordVersion(Version):
    __slots__ = ()
    name = u'DnsRec'
    pb_field_name = u'dns_record'
    vector_field_name = u'dns_record_version'
    can_be_deleted = True
    can_be_incomplete = True


class WeightSectionVersion(Version):
    __slots__ = ()
    name = u'WeiSec'
    pb_field_name = u'weight_sections'
    vector_field_name = u'weight_section_versions'
    can_be_deleted = True
    can_be_incomplete = False


class L7HeavyConfigVersion(Version):
    __slots__ = ()
    name = u'L7Heavy'
    pb_field_name = u'l7heavy_config'
    vector_field_name = u'l7heavy_config_version'
    can_be_deleted = True
    can_be_incomplete = True

    @classmethod
    def is_deleted(cls, pb):
        return pb.spec.state != pb.spec.PRESENT
