import base64

import inject
import six


try:
    import vault_client
except ImportError:
    # arcadia compatibility
    import library.python.vault_client as vault_client

from infra.swatlib.metrics import InstrumentedSession


class IYaVaultClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: YaVaultClient
        """
        return inject.instance(cls)


class YaVaultClient(object):
    def __init__(self, yav_url, nanny_tvm_id, rsa_login, rsa_key_path, connection_timeout=None):
        with open(rsa_key_path, mode='r' if six.PY3 else 'rb') as f:
            rsa_key = f.read()
        self._yav = vault_client.VaultClient(
            host=yav_url,
            native_client=InstrumentedSession('ya_vault', connection_timeout=connection_timeout),
            rsa_login=rsa_login,
            rsa_auth=rsa_key,
        )
        self._nanny_tvm_id = nanny_tvm_id

    @classmethod
    def from_config(cls, config):
        return cls(
            yav_url=config.get_value('ya_vault.url'),
            nanny_tvm_id=config.get_value('nanny.tvm_client_id'),
            rsa_login=config.get_value('rsa.login'),
            rsa_key_path=config.get_value('rsa.private_key_path'),
        )

    def get_token(self, service_id, secret_id):
        """
        Get delegation token for secret

        :type service_id: six.text_type
        :type secret_id: six.text_type
        :rtype: six.text_type
        """
        token, token_uuid = self._yav.create_token(signature=service_id,
                                                   secret_uuid=secret_id,
                                                   tvm_client_id=self._nanny_tvm_id)
        return token

    def check_token(self, secret_id, service_id, token):
        resp = self._yav.get_token_info(token=token)
        token_info = resp.get('token_info')
        if not (token_info.get('state_name') == 'normal' and
                str(token_info.get('tvm_client_id')) == str(self._nanny_tvm_id)):
            return False
        all_tokens = self._yav.list_tokens(secret_uuid=secret_id)
        token_uuid = token_info.get('token_uuid')
        for t in all_tokens:
            if token_uuid and token_uuid == t.get('token_uuid') and t.get('signature') == service_id:
                return True
        return False

    def get_version(self, version):
        """
        :type version: six.text_type
        :rtype: dict
        """
        return self._yav.get_version(version=version,
                                     decode_files=True)

    def remove_secret(self, secret_id):
        """
        :type secret_id: six.text_type
        :rtype: dict
        """
        return self._yav.update_secret(secret_uuid=secret_id,
                                       state='hidden')

    def add_certs_to_secret(self, secret_ver, certs_tarball, comment=''):
        """
        :type secret_ver: six.text_type
        :type certs_tarball: bytes
        :type comment: six.text_type
        """
        diff = [{
            'key': 'secrets.tgz',
            'value': base64.b64encode(certs_tarball).decode('ascii'),
            'encoding': 'base64'
        }]
        return self._yav.create_diff_version(parent_version_uuid=secret_ver,
                                             diff=diff,
                                             comment=comment)

    def delete_user_role_from_secret(self, secret_id, abc_service_id, abc_scope, role):
        """
        :type secret_id: six.text_type
        :type abc_service_id: six.text_type
        :type abc_scope: six.text_type
        :type role: six.text_type
        :rtype: dict
        """
        self._yav.delete_user_role_from_secret(secret_uuid=secret_id,
                                               abc_id=abc_service_id,
                                               abc_scope=abc_scope,
                                               role=role)

    def get_owners(self, secret_id):
        """
        :type secret_id: six.text_type
        :rtype: dict
        """
        return self._yav.get_owners(secret_uuid=secret_id)

    def get_readers(self, secret_id):
        """
        :type secret_id: six.text_type
        :rtype: dict
        """
        return self._yav.get_owners(secret_uuid=secret_id)

    def create_secret(self, name):
        return self._yav.create_secret(name)

    def create_secret_version(self, secret_uuid, value):
        return self._yav.create_secret_version(secret_uuid, value)
