# coding: utf-8
import six
import yaml


def is_endline(c):
    return c in '\0\n'


def get_snippet(buffer, pointer, max_length, max_lines=5):
    start = pointer

    endlines_count = 0
    last_line_start = 0
    while start > 0 and endlines_count < max_lines:
        if is_endline(buffer[start - 1]):
            if endlines_count == 0:
                last_line_start = start
            endlines_count += 1
        start -= 1
        if pointer - start > max_length // 2 - 1:
            if endlines_count == 0:
                last_line_start = start
            break
    end = pointer
    while end < len(buffer) and not is_endline(buffer[end]):
        end += 1
        if end - pointer > max_length // 2 - 1:
            break
    if is_endline(buffer[start]):
        start += 1
    snippet = buffer[start:end]
    return snippet + '\n' + ' ' * (pointer - last_line_start) + '^'


class Error(Exception):
    mark = None

    def attach_mark(self, mark):
        """
        :type mark: yaml.Mark
        """
        assert isinstance(mark, yaml.Mark)
        self.mark = mark

    def get_snippet(self, max_length=1024, max_lines=5):
        if not self.mark:
            return None

        if not self.mark.buffer:
            return None

        return get_snippet(buffer=self.mark.buffer, pointer=self.mark.pointer,
                           max_length=max_length, max_lines=max_lines)

    def to_dict(self):
        if six.PY3:
            message = Exception.__str__(self)
        else:
            message = self.message
        rv = {'message': message}
        snippet = self.get_snippet()
        if snippet:
            rv['snippet'] = snippet
        if self.mark:
            rv['position'] = {'line': self.mark.line + 1, 'column': self.mark.column + 1}
        return rv

    def __str__(self):
        if six.PY3:
            message = Exception.__str__(self)
        else:
            message = self.message
        if self.mark:
            position = 'line {}, column {}'.format(self.mark.line + 1, self.mark.column + 1)
            return '{}: {}'.format(position, message)
        else:
            return message

    __repr__ = __unicode__ = __str__


class InternalError(Error):
    pass


class YamlSyntaxError(Error):
    pass


class SchemaError(Error):
    pass


class NonAsciiCharsPresent(Error):
    @staticmethod
    def from_unicode_encode_error(event, error):
        mark = event.start_mark
        mark.column += error.start
        err = NonAsciiCharsPresent('Non-ascii characters present')
        err.message = 'Non-ascii characters present'
        err.attach_mark(event.start_mark)
        return err
