# coding: utf-8
import yaml.cyaml
from yaml import ScalarEvent

from . import errors


def _check_unicode_symbols(event):
    try:
         event.value.encode('ascii')
    except UnicodeEncodeError as e:
        raise errors.NonAsciiCharsPresent.from_unicode_encode_error(event, e)


def parse(stream, loader_cls=yaml.Loader, ensure_ascii=False):
    """
    Parses a YAML stream and yields YAML events.

    :type stream: str or a file-like object with `read` method
    :type loader_cls: yaml.Loader | yaml.cyaml.CLoader
    :type ensure_ascii: bool
    :rtype: iterable[yaml.events.Event]
    """
    loader = loader_cls(stream)
    try:
        while loader.check_event():
            event = loader.get_event()
            if ensure_ascii and isinstance(event, ScalarEvent):
                _check_unicode_symbols(event)
            yield event
    except yaml.error.YAMLError as e:
        exc = errors.YamlSyntaxError(e.problem)
        if e.problem_mark:
            exc.attach_mark(e.problem_mark)
        raise exc
    finally:
        loader.dispose()


def feed_events(builder, events):
    """
    :type builder:
    :type events: iterable[yaml.events.Event]
    """
    for event in events:
        try:
            builder.process(event)
        except errors.Error as e:
            if event.start_mark:
                e.attach_mark(event.start_mark)
            raise
