# coding: utf-8
import re

from google.protobuf import descriptor
from google.protobuf.internal import type_checkers


FD = descriptor.FieldDescriptor

_INTEGER_CHECKERS = (type_checkers.Uint32ValueChecker(),
                     type_checkers.Int32ValueChecker(),
                     type_checkers.Uint64ValueChecker(),
                     type_checkers.Int64ValueChecker())

_FLOAT_INFINITY = re.compile('-?inf(?:inity)?f?', re.IGNORECASE)
_FLOAT_NAN = re.compile('nanf?', re.IGNORECASE)

_FLOAT_TYPES = frozenset([FD.TYPE_FLOAT, FD.TYPE_DOUBLE])
_INT32_TYPES = frozenset([FD.TYPE_INT32, FD.TYPE_SINT32, FD.TYPE_SFIXED32])
_UINT32_TYPES = frozenset([FD.TYPE_UINT32, FD.TYPE_FIXED32])
_INT64_TYPES = frozenset([FD.TYPE_INT64, FD.TYPE_SINT64, FD.TYPE_SFIXED64])
_UINT64_TYPES = frozenset([FD.TYPE_UINT64, FD.TYPE_FIXED64])


def parse_int32(text):
    return parse_integer(text, is_signed=True, is_long=False)


def parse_uint32(text):
    return parse_integer(text, is_signed=False, is_long=False)


def parse_int64(text):
    return parse_integer(text, is_signed=True, is_long=True)


def parse_uint64(text):
    return parse_integer(text, is_signed=False, is_long=True)


def parse_integer(text, is_signed=False, is_long=False):
    try:
        if is_long:
            result = long(text, 0)
        else:
            result = int(text, 0)
    except ValueError:
        raise ValueError('couldn\'t parse integer: {}'.format(text))

    # check if the integer is sane
    checker = _INTEGER_CHECKERS[2 * int(is_long) + int(is_signed)]
    checker.CheckValue(result)
    return result


def parse_float(text):
    try:
        # assume Python compatible syntax
        return float(text)
    except ValueError:
        # check alternative spellings
        if _FLOAT_INFINITY.match(text):
            if text[0] == '-':
                return float('-inf')
            else:
                return float('inf')
        elif _FLOAT_NAN.match(text):
            return float('nan')
        else:
            # assume '1.0f' format
            try:
                return float(text.rstrip('f'))
            except ValueError:
                raise ValueError('couldn\'t parse float: {}'.format(text))


def parse_bool(text):
    if text in ('true', 't', '1', 'True'):
        return True
    elif text in ('false', 'f', '0', 'False'):
        return False
    else:
        raise ValueError('expected "true" or "false"')


def parse_enum(field, value):
    enum_descriptor = field.enum_type
    try:
        number = int(value, 0)
    except ValueError:
        # identifier
        enum_value = enum_descriptor.values_by_name.get(value, None)
        if enum_value is None:
            raise ValueError('enum type "{}" has no value named {}'.format(enum_descriptor.full_name, value))
    else:
        # numeric value
        enum_value = enum_descriptor.values_by_number.get(number, None)
        if enum_value is None:
            raise ValueError('enum type "{}" has no value with number {}'.format(enum_descriptor.full_name, number))
    return enum_value.number


def get_scalar_field_value(field_desc, text):
    if field_desc.type in _INT32_TYPES:
        rv = parse_int32(text)
    elif field_desc.type in _INT64_TYPES:
        rv = parse_int64(text)
    elif field_desc.type in _UINT32_TYPES:
        rv = parse_uint32(text)
    elif field_desc.type in _UINT64_TYPES:
        rv = parse_uint64(text)
    elif field_desc.type in _FLOAT_TYPES:
        rv = parse_float(text)
    elif field_desc.type == FD.TYPE_BOOL:
        rv = parse_bool(text)
    elif field_desc.type == FD.TYPE_STRING:
        rv = text
    elif field_desc.type == FD.TYPE_BYTES:
        rv = text
    elif field_desc.type == FD.TYPE_ENUM:
        rv = parse_enum(field_desc, text)
    else:
        raise RuntimeError('Unknown field type {}'.format(field_desc.type))
    return rv


class OrderedMapProxy(object):
    __slots__ = ('entries_pb', 'DESCRIPTOR')

    def __init__(self, entry_desc, entries_pb):
        self.entries_pb = entries_pb
        self.DESCRIPTOR = entry_desc

    def __getitem__(self, key):
        entry_pb = self.entries_pb.add()
        entry_pb.key = key
        return entry_pb.value

    def __setitem__(self, key, value):
        self.entries_pb.add(key=key, value=value)
