# coding: utf-8
import six
import yaml
from collections import OrderedDict
from six.moves import map

from infra.awacs.proto import modules_pb2
from awacs.yamlparser.core import pb_to_dict, parse
from awacs.yamlparser.util import AwacsYamlDumper


AWACS_HOLDER_MESSAGE_FULL_NAME = modules_pb2.Holder.DESCRIPTOR.full_name


class WrappersDumper(AwacsYamlDumper):
    def increase_indent(self, flow=False, indentless=False):
        return super(WrappersDumper, self).increase_indent(flow, False)


class UpstreamEasyModeYamlDumper(WrappersDumper):
    pass


class TopLevelEasyModeYamlDumper(WrappersDumper):
    pass


class HeaderActionData(object):
    def __init__(self, data):
        self.data = data


class HeaderAction(object):
    def __init__(self, data):
        assert len(data) == 1
        self.data = {k: HeaderActionData(v) for k, v in six.iteritems(data)}


def header_action_representer(dumper, action):
    node = dumper.represent_data(action.data)
    node.flow_style = False
    return node


def header_action_data_representer(dumper, action_data):
    node = dumper.represent_data(action_data.data)
    node.flow_style = True
    return node


def literal_representer(dumper, data):
    if isinstance(data, six.string_types) and (any(c in data for c in ' #:;=!@$&^*()+\\') or data.isdigit()):
        return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='\'')
    return dumper.represent_scalar('tag:yaml.org,2002:str', data)


class FlatSchemeData(object):
    def __init__(self, data):
        data['balancer'] = BalancerData(data['balancer'])
        self.data = data


class DcData(object):
    def __init__(self, data):
        name = data.pop('name')
        self.data = OrderedDict([('name', name)] + list(data.items()))


class ByDcSchemeData(object):
    def __init__(self, data):
        data['balancer'] = BalancerData(data['balancer'])
        data['dc_balancer'] = DcBalancerData(data['dc_balancer'])
        data['dcs'] = [DcData(dc) for dc in data['dcs']]
        self.data = data


class NoQuotesScalar(object):
    def __init__(self, value):
        self.value = value


class InlineList(object):
    def __init__(self, items, no_quotes=False):
        if no_quotes:
            items = list(map(NoQuotesScalar, items))
        self.items = items


class RetryHttpResponsesData(object):
    def __init__(self, data):
        data['codes'] = InlineList(data['codes'], no_quotes=True)
        if 'exceptions' in data:
            data['exceptions'] = InlineList(data['exceptions'], no_quotes=True)
        self.data = data


class BalancerData(object):
    def __init__(self, data):
        if 'retry_http_responses' in data:
            data['retry_http_responses'] = RetryHttpResponsesData(data['retry_http_responses'])
        self.data = data


class DcBalancerData(object):
    def __init__(self, data):
        self.data = data


def flat_scheme_representer(dumper, action):
    node = dumper.represent_data(action.data)
    node.flow_style = False
    return node


def by_dc_scheme_representer(dumper, action):
    node = dumper.represent_data(action.data)
    node.flow_style = False
    return node


def balancer_representer(dumper, action):
    node = dumper.represent_data(action.data)
    node.flow_style = False
    return node


def dc_balancer_representer(dumper, action):
    node = dumper.represent_data(action.data)
    node.flow_style = False
    return node


def dc_representer(dumper, action):
    node = dumper.represent_data(action.data)
    node.flow_style = False
    return node


def retry_http_responses_representer(dumper, action):
    node = dumper.represent_data(action.data)
    node.flow_style = False
    return node


def float_representer(dumper, value):
    if value == int(value):
        return dumper.represent_scalar(u'tag:yaml.org,2002:int', six.text_type(int(value)))
    return dumper.represent_scalar(u'tag:yaml.org,2002:float', u'%.2g' % value)


def inline_list_representer(dumper, o):
    node = dumper.represent_data(o.items)
    node.flow_style = True
    return node


def no_quotes_scalar_representer(dumper, o):
    if o.value.isdigit():
        return dumper.represent_scalar(u'tag:yaml.org,2002:int', o.value, style=u'')
    return dumper.represent_scalar(u'tag:yaml.org,2002:str', o.value, style=u'')


WrappersDumper.add_representer(six.string_types, literal_representer)
WrappersDumper.add_representer(six.text_type, literal_representer)
WrappersDumper.add_representer(str, literal_representer)
WrappersDumper.add_representer(float, float_representer)

UpstreamEasyModeYamlDumper.add_representer(six.string_types, literal_representer)
UpstreamEasyModeYamlDumper.add_representer(six.text_type, literal_representer)
UpstreamEasyModeYamlDumper.add_representer(str, literal_representer)
UpstreamEasyModeYamlDumper.add_representer(float, float_representer)
UpstreamEasyModeYamlDumper.add_representer(HeaderAction, header_action_representer)
UpstreamEasyModeYamlDumper.add_representer(HeaderActionData, header_action_data_representer)
UpstreamEasyModeYamlDumper.add_representer(BalancerData, balancer_representer)
UpstreamEasyModeYamlDumper.add_representer(DcBalancerData, dc_balancer_representer)
UpstreamEasyModeYamlDumper.add_representer(ByDcSchemeData, by_dc_scheme_representer)
UpstreamEasyModeYamlDumper.add_representer(FlatSchemeData, flat_scheme_representer)
UpstreamEasyModeYamlDumper.add_representer(RetryHttpResponsesData, retry_http_responses_representer)
UpstreamEasyModeYamlDumper.add_representer(InlineList, inline_list_representer)
UpstreamEasyModeYamlDumper.add_representer(NoQuotesScalar, no_quotes_scalar_representer)
UpstreamEasyModeYamlDumper.add_representer(DcData, dc_representer)

TopLevelEasyModeYamlDumper.add_representer(HeaderAction, header_action_representer)
TopLevelEasyModeYamlDumper.add_representer(HeaderActionData, header_action_data_representer)
TopLevelEasyModeYamlDumper.add_representer(six.string_types, literal_representer)
TopLevelEasyModeYamlDumper.add_representer(six.text_type, literal_representer)
TopLevelEasyModeYamlDumper.add_representer(str, literal_representer)


def dump_uem_pb(pb):
    assert pb.DESCRIPTOR.full_name == AWACS_HOLDER_MESSAGE_FULL_NAME
    assert pb.HasField('l7_upstream_macro')
    d = pb_to_dict(pb)
    l7_upstream_macro_d = d['l7_upstream_macro']
    headers = l7_upstream_macro_d.get('headers')
    if headers:
        l7_upstream_macro_d['headers'] = list(map(HeaderAction, headers))
    response_headers = l7_upstream_macro_d.get('response_headers')
    if response_headers:
        l7_upstream_macro_d['response_headers'] = list(map(HeaderAction, response_headers))
    flat_scheme = l7_upstream_macro_d.get('flat_scheme')
    if flat_scheme:
        l7_upstream_macro_d['flat_scheme'] = FlatSchemeData(flat_scheme)
    by_dc_scheme = l7_upstream_macro_d.get('by_dc_scheme')
    if by_dc_scheme:
        l7_upstream_macro_d['by_dc_scheme'] = ByDcSchemeData(by_dc_scheme)
    yml = yaml.dump(d, default_flow_style=False, Dumper=UpstreamEasyModeYamlDumper, width=200)
    assert parse(modules_pb2.Holder, yml) == pb
    return yml


def dump_tlem_pb(pb):
    assert pb.DESCRIPTOR.full_name == AWACS_HOLDER_MESSAGE_FULL_NAME
    assert pb.HasField('l7_macro')
    d = pb_to_dict(pb)
    l7_macro_d = d['l7_macro']
    if 'headers' in l7_macro_d:
        l7_macro_d['headers'] = list(map(HeaderAction, l7_macro_d['headers']))
    if 'response_headers' in l7_macro_d:
        l7_macro_d['response_headers'] = list(map(HeaderAction, l7_macro_d['response_headers']))
    yml = yaml.dump(d, default_flow_style=False, Dumper=TopLevelEasyModeYamlDumper, width=200)
    assert parse(modules_pb2.Holder, yml) == pb
    return yml
