import six
import inject
import requests
import time
from collections import namedtuple

from infra.yasm.yasmapi import GolovanRequest, Transport


class IYasmClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: YasmClient
        """
        return inject.instance(cls)


class YasmReplaceAlertsResult(namedtuple('YasmReplaceAlertsResult', ['updated', 'created', 'deleted'])):
    pass


class YasmClient(object):
    DEFAULT_URL = 'https://yasm.yandex-team.ru'

    REPLACE_ALERTS_PATH = '/srvambry/alerts/replace'
    REPLACE_ALERTS_MAX_COUNT = 1000

    LIST_ALERTS_PATH = '/srvambry/alerts/list'

    RPS_SIGNAL_NAME = 'balancer_report-report-service_total-succ_summ'

    @classmethod
    def from_config(cls, c):
        return cls(c['name_prefix'], yasm_url=c.get('url'))

    def __init__(self, name_prefix, yasm_url=None):
        self._name_prefix = name_prefix
        self._yasm_url = yasm_url or self.DEFAULT_URL

    def replace_alerts(self, prefix, alerts):
        """
        Docs:
            - https://wiki.yandex-team.ru/golovan/userdocs/alerts/api/?from=%252Fgolovan%252Falerts%252Fapi%252F#post../srvambry/alerts/replace

        :type prefix: six.text_type
        :type alerts: list[dict]
        :rtype: YasmReplaceAlertsResult
        """
        prefix = prefix.rstrip('.')
        # TODO: remove after fix https://st.yandex-team.ru/GOLOVAN-7098

        if len(alerts) > self.REPLACE_ALERTS_MAX_COUNT:
            raise ValueError('replace_alerts does not support more then {} alerts for replace'.format(
                self.REPLACE_ALERTS_MAX_COUNT))

        if not prefix.startswith(self._name_prefix):
            raise ValueError('Wrong alerts prefix: "{}", should starts with "{}"'.format(prefix, self._name_prefix))

        if any([not a['name'].startswith(prefix) for a in alerts]):
            raise ValueError('All alerts names should starts with passed prefix: "{}"'.format(prefix))

        resp = requests.post(
            url=self._yasm_url + self.REPLACE_ALERTS_PATH,
            json={
                'prefix': prefix,
                'alerts': alerts
            })
        if resp.status_code == 400:
            raise self.BadRequestError(resp.text)

        if resp.status_code == 500:
            raise self.BadRequestError(resp.text)

        resp.raise_for_status()
        data = resp.json()
        result = data.get('response', {}).get('result', {})
        return YasmReplaceAlertsResult(
            updated=int(result.get('updated', 0)),
            created=int(result.get('created', 0)),
            deleted=int(result.get('deleted', 0)),
        )

    @staticmethod
    def _balancer_yasm_tags(location_pb, instance_tags_pb):
        geo = (location_pb.yp_cluster or location_pb.gencfg_dc).lower()
        if geo in ('iva', 'myt'):
            geo = 'msk'
        return 'itype={};ctype={};prj={};geo={}'.format(instance_tags_pb.itype, instance_tags_pb.ctype,
                                                        instance_tags_pb.prj, geo)

    def get_last_balancer_rps_link(self, location_pb, instance_tags_pb):
        return (self._yasm_url + '/chart/hosts=ASEARCH;{};signals={}/?range=3600000'
                .format(self._balancer_yasm_tags(location_pb, instance_tags_pb), self.RPS_SIGNAL_NAME))

    @staticmethod
    def get_screenshot_link(link):
        return link.replace('://', '://s.')

    def get_last_balancer_rps(self, location_pb, instance_tags_pb):
        period = 3600
        end_time = int(time.time()) - 60
        start_time = end_time - period
        full_signal_name = '{}:{}'.format(self._balancer_yasm_tags(location_pb, instance_tags_pb), self.RPS_SIGNAL_NAME)
        vals = []
        for timestamp, values in GolovanRequest(host='ASEARCH', period=period, st=start_time, et=end_time,
                                                fields=[full_signal_name],
                                                transport=Transport(connect_timeout=100)):
            for signal, val in six.iteritems(values):
                vals.append(float(val) / period if val else 0)
        return sum(vals) // len(vals) if vals else 0

    class YasmApiError(Exception):
        pass

    class BadRequestError(YasmApiError):
        pass
