"""
Intends to be a drop-in replacement for
https://a.yandex-team.ru/arc/trunk/arcadia/infra/yp_service_discovery/python/resolver/resolver.py.
"""
import time
import uuid

import inject
from six.moves import http_client as httplib
from requests.adapters import HTTPAdapter
from infra.swatlib.metrics import InstrumentedSession
from infra.swatlib.util.retry import RetrySleeper, RetryWithTimeout

from infra.awacs.proto import internals_pb2


DEFAULT_SD_HTTP_ADDRESS = u'http://sd.yandex.net:8080/'
DEFAULT_CLIENT_NAME = u'awacs'
DEFAULT_REQ_TIMEOUT = 60
DEFAULT_ATTEMPTS = 2


def generate_reqid():
    return u'{}-{}'.format(int(time.time() * 1000), uuid.uuid4())


class IResolver(object):
    @classmethod
    def instance(cls):
        """
        :rtype: Resolver
        """
        return inject.instance(cls)

    def resolve_endpoints(self, req_pb, req_id=None):
        """
        :type req_pb: internals_pb2.TReqResolveEndpoints
        :type req_id: six.text_type
        :rtype: internals_pb2.TRspResolveEndpoints
        """
        raise NotImplementedError


class Resolver(IResolver):
    OVERRIDE_CLUSTER_NAMES = {
        'test_sas': 'sas-test',
        'sas_test': 'sas-test',
        'man_pre': 'man-pre',
    }

    def __init__(self, client_name=DEFAULT_CLIENT_NAME, http_address=DEFAULT_SD_HTTP_ADDRESS,
                 req_timeout=DEFAULT_REQ_TIMEOUT, attempts=DEFAULT_ATTEMPTS):
        """A client for resolving YP endpoints.
        :param client_name: A name to identify you on YP Service Discovery side,
            it is desirable to specify hostname.
        """
        self._client_name = client_name
        self._http_address = http_address.rstrip(u'/')
        self._req_timeout = req_timeout
        if self._req_timeout < 1.0:
            raise ValueError(u'req_timeout must be >= 1.0, got {}'.format(self._req_timeout))
        self._attempts = attempts
        if self._attempts < 1:
            raise ValueError(u'attempts must be >= 1, got {}'.format(self._attempts))
        self._session = InstrumentedSession(u'yp-sd-resolver')
        self._session.mount(self._http_address, self._make_http_adapter())

    def _make_http_adapter(self):
        """
        Make custom http adapter.
        We need maximum pooled connection to be less than attempts.
        Thus we work around situation when we have actually dead connections at hand.
        We achieve this by exhausting connection pool if connections are dead.
        E.g. they can be dropped by ipvs or/and http balancer due to inactivity.

        Proper ways to fix can be:
          * Use SO_KEEPALIVE to minimize having broken connections in the pool.
          * Throw connections coming from pool if they were idle for too long.
        But these ways are hard to implement, there aren't enough hooks in requests library.
        This fix should do it. See SEPE-8393 for details.
        """
        max_conn = max(1, self._attempts // 2)  # We need at least 1 connection
        return HTTPAdapter(pool_connections=max_conn, pool_maxsize=max_conn)

    def _call_remote(self, func, *args, **kwargs):
        r = RetryWithTimeout(self._req_timeout,
                             RetrySleeper(max_tries=self._attempts, max_delay=30))
        return r(func, *args, **kwargs)

    @classmethod
    def from_config(cls, d):
        return cls(client_name=d.get(u'client_name', DEFAULT_CLIENT_NAME),
                   http_address=d[u'url'])

    def resolve_endpoints(self, req_pb, req_id=None):
        """
        :type req_pb: internals_pb2.TReqResolveEndpoints
        :type req_id: six.text_type
        :rtype: internals_pb2.TRspResolveEndpoints
        """
        url = self._http_address + u'/resolve_endpoints'

        if req_pb.cluster_name in self.OVERRIDE_CLUSTER_NAMES:
            req_pb.cluster_name = self.OVERRIDE_CLUSTER_NAMES[req_pb.cluster_name]
        if not req_pb.client_name:
            req_pb.client_name = self._client_name
        req_pb.ruid = req_id or generate_reqid()
        resp = self._call_remote(self._session.post, url, data=req_pb.SerializeToString())
        if resp.status_code != httplib.NOT_FOUND:
            # https://st.yandex-team.ru/YP-1163#5ef353262e1dc57cc88d6f81
            resp.raise_for_status()
        rv_pb = internals_pb2.TRspResolveEndpoints()
        rv_pb.ParseFromString(resp.content)
        return rv_pb
