from __future__ import unicode_literals

import grpc
import gevent
import inject
import requests
import ujson
from nanny_rpc_client import requests_client, exceptions
from infra.swatlib.util import retry

try:
    from awacs.lib.ypclient import object_service_stub
except ImportError:
    # this is a correct import for awacs in arcadia (see https://a.yandex-team.ru/arc/commit/3529257):
    from yp_proto.yp.client.api.proto import object_service_pb2_grpc as object_service_stub

from . import interceptors


class YpError(Exception):
    """
    YP-related error
    """
    pass


def _yp_exception_parser(resp):
    """
    :type resp: requests.models.Response
    :rtype: exceptions.HttpError
    """
    if resp.status_code == requests.codes.GATEWAY_TIMEOUT:
        # We can't be sure that it's exactly "balancer max retries exceeded" error.
        # But I've never seen the other reason for this error, so consider this error as
        # "balancer max retries exceeded"
        return exceptions.BalancerRetriesExceeded('Balancer max retries for request processing exceeded, response: '
                                                  '{0}: "{1}"'.format(resp.status_code, resp.content))

    h = resp.headers.get('X-Yt-Error')
    if not h:
        raise exceptions.InvalidResponseError('Invalid YP response, no X-Yt-Error header given')

    parsed = ujson.loads(h)
    # We should put code -> exception class map here at some point
    if parsed['code'] == 100002:
        raise exceptions.NotFoundError(parsed.get('message', 'Not found'))

    raise exceptions.BadRequestError('Unknown YP error: {}'.format(h))


class IYpObjectServiceClientFactory(object):
    @classmethod
    def instance(cls):
        return inject.instance(cls)

    def get(self, cluster):
        """
        :type cluster: six.text_type
        :rtype: yp.client.api.proto.object_service_stub.ObjectServiceStub
        """
        raise NotImplementedError


class YpObjectServiceClientFactory(IYpObjectServiceClientFactory):
    DEFAULT_RETRY_SLEEPER_CONFIG = {
        'max_tries': 3,
        'delay': 1,
    }

    def __init__(self, clients):
        """
        :type clients: dict[six.text_type, requests_client.RetryingRpcClient]
        """
        self.clients = clients

    @staticmethod
    def _grpc_client_from_config(cfg, oauth_token=None):
        """
        WARNING
        This method is a temporary implementation, only works in arcadia, where library.python.resource is present.
        /WARNING

        :param dict cfg:
        :param str oauth_token:
        :rtype: object_service_stub.ObjectServiceStub
        """
        from library.python import resource
        cfg = cfg.copy()
        crt = resource.find('YandexInternalRootCA.crt')
        channel = grpc.secure_channel(cfg['rpc_url'], grpc.ssl_channel_credentials(crt))
        # https://wiki.yandex-team.ru/yp/accesscontrol/#tokenyblackbox
        # https://github.yandex-team.ru/yt/yt/blob/dfd1cf2b77308f9ea9ba90503c941be26b699ffc/yp/python/yp/client.py#L169
        if oauth_token:
            header_adder_interceptor = interceptors.MetadataAppenderInterceptor([(b'yt-auth-token', oauth_token)])
            channel = grpc.intercept_channel(channel, header_adder_interceptor)

        return object_service_stub.ObjectServiceStub(channel)

    @staticmethod
    def _create_http_client_from_config(retry_sleeper):
        def http_client_from_config(cfg, oauth_token=None):
            cfg = cfg.copy()
            cfg.pop('cluster', None)
            rpc = requests_client.RetryingRpcClient(
                retry_sleeper=retry_sleeper,
                exception_parser=_yp_exception_parser,
                path_trailing_slash=False,
                oauth_token=oauth_token,
                **cfg
            )
            return object_service_stub.ObjectServiceStub(rpc)

        return http_client_from_config

    @classmethod
    def _clients_from_configs(cls, configs, client_from_config, oauth_token=None):
        clients = {}
        for cfg in configs:
            clients[cfg['cluster'].lower()] = client_from_config(cfg, oauth_token=oauth_token)
        return clients

    @classmethod
    def from_config(cls, config):
        use_grpc = config.get('use_grpc', True)
        oauth_token = config.get('oauth_token', None)
        if use_grpc:
            client_from_config = cls._grpc_client_from_config
        else:
            client_from_config = cls._create_http_client_from_config(retry.RetrySleeper(
                sleep_func=gevent.sleep,
                **config.get('retry', cls.DEFAULT_RETRY_SLEEPER_CONFIG)
            ))
        clients = cls._clients_from_configs(config['clusters'], client_from_config, oauth_token=oauth_token)
        return cls(clients=clients)

    def get(self, cluster):
        c = self.clients.get(cluster)
        if c:
            return c
        raise YpError('Cannot get YP client for cluster "{}"'.format(cluster))
