"""
Please see the following example for details:
https://github.com/grpc/grpc/blob/master/examples/python/interceptors/headers/
"""
import collections

import grpc


class BaseClientInterceptor(grpc.UnaryUnaryClientInterceptor, grpc.UnaryStreamClientInterceptor,
                            grpc.StreamUnaryClientInterceptor, grpc.StreamStreamClientInterceptor):

    def _do_intercept(self, client_call_details, request_iterator, request_streaming, response_streaming):
        raise NotImplementedError

    def intercept_unary_unary(self, continuation, client_call_details, request):
        new_details, new_request_iterator, postprocess = self._do_intercept(
            client_call_details, iter((request,)), False, False)
        response = continuation(new_details, next(new_request_iterator))
        return postprocess(response) if postprocess else response

    def intercept_unary_stream(self, continuation, client_call_details,
                               request):
        new_details, new_request_iterator, postprocess = self._do_intercept(
            client_call_details, iter((request,)), False, True)
        response_it = continuation(new_details, next(new_request_iterator))
        return postprocess(response_it) if postprocess else response_it

    def intercept_stream_unary(self, continuation, client_call_details,
                               request_iterator):
        new_details, new_request_iterator, postprocess = self._do_intercept(
            client_call_details, request_iterator, True, False)
        response = continuation(new_details, new_request_iterator)
        return postprocess(response) if postprocess else response

    def intercept_stream_stream(self, continuation, client_call_details,
                                request_iterator):
        new_details, new_request_iterator, postprocess = self._do_intercept(
            client_call_details, request_iterator, True, True)
        response_it = continuation(new_details, new_request_iterator)
        return postprocess(response_it) if postprocess else response_it


class _ClientCallDetails(collections.namedtuple('_ClientCallDetails', ('method', 'timeout', 'metadata', 'credentials')),
                         grpc.ClientCallDetails):
    pass


class MetadataUpdaterInterceptor(BaseClientInterceptor):
    def _do_intercept(self, client_call_details, request_iterator, request_streaming, response_streaming):
        metadata = []
        if client_call_details.metadata is not None:
            metadata = list(client_call_details.metadata)
        updated_metadata = self._update_metadata(metadata)
        client_call_details = _ClientCallDetails(
            client_call_details.method, client_call_details.timeout, updated_metadata, client_call_details.credentials)
        return client_call_details, request_iterator, None

    def _update_metadata(self, metadata):
        """
        :type metadata: list[(str, str)]
        :rtype: list[(str, str)]
        """
        raise NotImplementedError


class MetadataAppenderInterceptor(MetadataUpdaterInterceptor):
    def __init__(self, metadata_to_append):
        """
        :type metadata_to_append: list[(str, str)]
        """
        self._metadata_to_append = metadata_to_append

    def _update_metadata(self, metadata):
        """
        :type metadata: list[(str, str)]
        :rtype: list[(str, str)]
        """
        return metadata + self._metadata_to_append
