# coding: utf-8
from six.moves.urllib import parse as urlparse

import inject

import nanny_rpc_client


try:
    from yp_lite_ui_repo import pod_sets_api_stub, endpoint_sets_api_stub, pod_sets_api_pb2
except ImportError:
    # for Arcadia:
    from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
    from infra.nanny.yp_lite_api.py_stubs import pod_sets_api_stub, endpoint_sets_api_stub


class IYpLiteRpcClient(object):
    @classmethod
    def instance(cls):
        """
        :rtype: YpLiteRpcClient
        """
        return inject.instance(cls)


class YpLiteRpcClient(IYpLiteRpcClient):
    DEFAULT_REQUEST_TIMEOUT = 60
    LIST_PODS_REQ_LIMIT = 100

    @classmethod
    def from_config(cls, d):
        return cls(url=d['url'], req_timeout=d.get('req_timeout'), token=d.get('token'))

    def __init__(self, url, req_timeout=None, token=None):
        super(YpLiteRpcClient, self).__init__()
        self.pod_sets_client = nanny_rpc_client.RequestsRpcClient(
            urlparse.urljoin(url, '/api/yplite/pod-sets/'),
            request_timeout=req_timeout or self.DEFAULT_REQUEST_TIMEOUT,
            oauth_token=token)
        self.endpoint_sets_client = nanny_rpc_client.RequestsRpcClient(
            urlparse.urljoin(url, '/api/yplite/endpoint-sets/'),
            request_timeout=req_timeout or self.DEFAULT_REQUEST_TIMEOUT,
            oauth_token=token)
        self.pod_sets_stub = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(self.pod_sets_client)
        self.endpoint_sets_stub = endpoint_sets_api_stub.YpLiteUIEndpointSetsServiceStub(self.endpoint_sets_client)

    def get_pod(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.pod_sets_api_pb2.GetPodRequest
        :rtype: yp_lite_ui_repo.pod_sets_api_pb2.GetPodResponse
        """
        return self.pod_sets_stub.get_pod(req_pb)

    def create_pod_set(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.pod_sets_api_pb2.CreatePodSetRequest
        :rtype: yp_lite_ui_repo.pod_sets_api_pb2.CreatePodSetResponse
        """
        return self.pod_sets_stub.create_pod_set(req_pb)

    def get_pod_set(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.pod_sets_api_pb2.GetPodSetRequest
        :rtype: yp_lite_ui_repo.pod_sets_api_pb2.GetPodSetResponse
        """
        return self.pod_sets_stub.get_pod_set(req_pb)

    def remove_pod_set(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.pod_sets_api_pb2.RemovePodSetRequest
        :rtype: yp_lite_ui_repo.pod_sets_api_pb2.RemovePodSetResponse
        """
        return self.pod_sets_stub.remove_pod_set(req_pb)

    def get_endpoint_set(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.endpoint_sets_api_pb2.GetEndpointSetRequest
        :rtype: yp_lite_ui_repo.endpoint_sets_api_pb2.GetEndpointSetResponse
        """
        return self.endpoint_sets_stub.get_endpoint_set(req_pb)

    def create_endpoint_set(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.endpoint_sets_api_pb2.CreateEndpointSetRequest
        :rtype: yp_lite_ui_repo.endpoint_sets_api_pb2.CreateEndpointSetResponse
        """
        return self.endpoint_sets_stub.create_endpoint_set(req_pb)

    def list_endpoint_sets(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.endpoint_sets_api_pb2.ListEndpointSetsRequest
        :rtype: yp_lite_ui_repo.endpoint_sets_api_pb2.ListEndpointSetsResponse
        """
        return self.endpoint_sets_stub.list_endpoint_sets(req_pb)

    def remove_endpoint_sets(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.endpoint_sets_api_pb2.RemoveEndpointSetsRequest
        :rtype: yp_lite_ui_repo.endpoint_sets_api_pb2.RemoveEndpointSetsResponse
        """
        return self.endpoint_sets_stub.remove_endpoint_sets(req_pb)

    def validate_network_macro(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.pod_sets_api_pb2.ValidateNetworkMacroRequest
        :rtype: yp_lite_ui_repo.pod_sets_api_pb2.ValidateNetworkMacroResponse
        """
        return self.pod_sets_stub.validate_network_macro(req_pb)

    def validate_quota_settings(self, req_pb):
        """
        :type req_pb: yp_lite_ui_repo.pod_sets_api_pb2.ValidateQuotaSettingsRequest
        :rtype: yp_lite_ui_repo.pod_sets_api_pb2.ValidateQuotaSettingsResponse
        """
        return self.pod_sets_stub.validate_quota_settings(req_pb)

    def list_all_pod_ids(self, service_id, cluster):
        """
        :type req_pb: six.text_type
        :rtype: six.text_type
        """
        limit = self.LIST_PODS_REQ_LIMIT
        req_pb = pod_sets_api_pb2.ListPodsRequest(
            service_id=service_id,
            cluster=cluster,
            limit=limit
        )
        req_pb.field_mask.paths.append('meta.id')
        rv = []
        while True:
            resp_pb = self.pod_sets_stub.list_pods(req_pb)
            if not resp_pb.pods:
                break
            rv.extend(pod_pb.meta.id for pod_pb in resp_pb.pods)
            req_pb.offset += limit
        return rv
