# coding: utf-8
import re
import six
import abc
import string
from itertools import chain


try:
    from string import lowercase  # py2
except ImportError:
    from string import ascii_lowercase as lowercase  # py3

import yaml
import juggler_sdk
import semantic_version

from awacs.model import cache, errors, external_clusters
from awacs.model.balancer import endpoints
from awacs.model.util import make_system_endpoint_set_id
from awacs.lib import juggler_client, staffclient
from infra.awacs.proto import model_pb2


BALANCER_GROUP = model_pb2.NamespaceSpec.AlertingSettings.BALANCER_GROUP
PLATFORM_GROUP = model_pb2.NamespaceSpec.AlertingSettings.PLATFORM_GROUP

NOTIFY_GROUP_TO_FIELD_NAME = {
    BALANCER_GROUP: 'balancer',
    PLATFORM_GROUP: 'platform',
}

ENV_TYPE_TO_SUFFIX = {
    model_pb2.BalancerSpec.L7_ENV_UNKNOWN: '',
    model_pb2.BalancerSpec.L7_ENV_PRODUCTION: '',
    model_pb2.BalancerSpec.L7_ENV_PRESTABLE: '_prestable',
    model_pb2.BalancerSpec.L7_ENV_TESTING: '_testing',
}

YASM_ALERTS_NAMESPACE_PANEL_URL_TPL = ('https://yasm.yandex-team.ru/template/panel/awacs-balancers-alerts/'
                                       'awacs={alerting_prefix};'
                                       'namespace={namespace_id};'
                                       'locations={locations_coma_sep}/')

WIKI_MONITORING_DOC_URL = 'https://wiki.yandex-team.ru/cplb/awacs/monitoring/alerting/'
WIKI_MONITORING_ACTION_ITEMS_URL = 'https://wiki.yandex-team.ru/cplb/awacs/monitoring/alerting/actions/'

DEFAULT_BALANCER_NOTIFY_RULE_NAME = 'on_status_change'
DEFAULT_WARN_BALANCER_NOTIFY_RULE_KWARGS_TEMPLATE = """status:
    - from: OK
      to: WARN
login: {}
method:
    - email
"""

DEFAULT_CRIT_BALANCER_NOTIFY_RULE_TEMPLATE = """status:
    - from: WARN
      to: CRIT
    - from: OK
      to: CRIT
login: {notify_group_slugs}
method:
    - {notify_method}
    - email
"""

REQUIRED_BALANCER_NOTIFY_RULE_NAME = 'on_status_change'

notify_rule_status_change_key = lambda status: (status.get('from'), status.get('to'))
REQUIRED_CRIT_BALANCER_NOTIFY_RULE_STATUS_CHANGE = sorted(
    yaml.safe_load(DEFAULT_CRIT_BALANCER_NOTIFY_RULE_TEMPLATE)['status'],
    key=notify_rule_status_change_key)

SMS_RECEIVERS_THRESHOLD = 4
MIN_TELEGRAM_RECEIVERS = 2
MIN_TELEGRAM_RECEIVERS_RATIO = 0.75


def get_crit_notifications_method(notify_group_ids):
    logins = set()
    for notify_group_id in notify_group_ids:
        logins.update(staffclient.IStaffClient.instance().get_group_members(notify_group_id))

    logins = list(logins)

    if len(logins) < SMS_RECEIVERS_THRESHOLD:
        return 'sms'

    telegram_enabled_logins = juggler_client.IJugglerClient.instance().get_telegram_subscribers(logins)
    if (
        len(telegram_enabled_logins) >= MIN_TELEGRAM_RECEIVERS
        and float(len(telegram_enabled_logins)) / len(logins) >= MIN_TELEGRAM_RECEIVERS_RATIO
    ):
        return 'telegram'

    return 'sms'


def get_notify_group_slug(staff_group_id):
    group_info = staffclient.IStaffClient.instance().get_groups_by_ids(
        group_ids=[staff_group_id],
        fields=('id', 'url')
    )
    if staff_group_id not in group_info:
        raise RuntimeError(u'Failed to enable namespace alerting: staff group with id "{}" not found'
                           .format(staff_group_id))
    return group_info[staff_group_id][u'url']


def fill_default_notify_rules(alerting_pb, staff_group_ids):
    if not staff_group_ids:
        raise ValueError("staff_group_ids must be specified")

    staff_group_slugs = [get_notify_group_slug(group_id) for group_id in staff_group_ids]

    notify_method = get_crit_notifications_method(staff_group_ids)

    notify_targets = [
        repr('@' + str(slug)) for slug in staff_group_slugs
    ]
    slugs_str = '[{}]'.format(", ".join(notify_targets))

    # add CRIT rule
    alerting_pb.juggler_raw_notify_rules.balancer.add(
        template_name=DEFAULT_BALANCER_NOTIFY_RULE_NAME,
        template_kwargs=DEFAULT_CRIT_BALANCER_NOTIFY_RULE_TEMPLATE.format(
            notify_group_slugs=slugs_str,
            notify_method=notify_method,
        )
    )

    # add WARN rule
    alerting_pb.juggler_raw_notify_rules.balancer.add(
        template_name=DEFAULT_BALANCER_NOTIFY_RULE_NAME,
        template_kwargs=DEFAULT_WARN_BALANCER_NOTIFY_RULE_KWARGS_TEMPLATE.format(slugs_str),
    )


def apply_alerting_preset(spec_pb, preset):
    if preset == model_pb2.NamespaceSpec.PR_DEFAULT:
        spec_pb.alerting.juggler_raw_notify_rules.SetInParent()
        spec_pb.alerting.balancer_checks_disabled = False
    elif preset == model_pb2.NamespaceSpec.PR_WITHOUT_NOTIFICATIONS:
        spec_pb.alerting.notify_rules_disabled = True
        spec_pb.alerting.balancer_checks_disabled = False
    elif preset == model_pb2.NamespaceSpec.PR_PLATFORM_CHECKS_ONLY:
        spec_pb.alerting.juggler_raw_notify_rules.SetInParent()
        spec_pb.alerting.balancer_checks_disabled = True
    else:
        raise errors.ValidationError(
            u'Namespace modification conflict: unsupported preset {!r}'.format(
                model_pb2.NamespaceSpec.NamespaceSettingsPreset.Name(preset)
                if preset in model_pb2.NamespaceSpec.NamespaceSettingsPreset.values()
                else preset
            )
        )

    spec_pb.preset = preset


def notify_group_str(notify_group):
    return model_pb2.NamespaceSpec.AlertingSettings.NotifyRuleGroup.Name(notify_group)


def normalize_namespace_id(namespace_id):
    return cache.AwacsCache.normalise_namespace_name(namespace_id).strip('-')


def make_yasm_alerts_panel_url(alerting_prefix, namespace_id, locations):
    """

    :type alerting_prefix: six.text_type
    :type namespace_id: six.text_type
    :type locations: list[six.text_type]
    :rtype: six.text_type
    """
    return YASM_ALERTS_NAMESPACE_PANEL_URL_TPL.format(
        alerting_prefix=alerting_prefix,
        namespace_id=cache.AwacsCache.normalise_namespace_name(namespace_id),
        locations_coma_sep=','.join([location.lower() for location in locations])
    )


def make_juggler_check_meta_urls(balancer_ui_url, actions_wiki_page_anchor, padding_right):
    """

    :type balancer_ui_url: six.text_type
    :type actions_wiki_page_anchor: six.text_type
    :type padding_right: int
    :rtype: list[dict]
    """
    return [
        {
            'url': balancer_ui_url,
            'type': 'nanny',
            'title': u'⚠️Балансер в awacs'
        },
        {
            'url': "{}#{}".format(WIKI_MONITORING_ACTION_ITEMS_URL, actions_wiki_page_anchor),
            'type': 'wiki',
            'title': u'⚠️Что делать, если алерт сработал'
                     + (u'\u00A0' * padding_right)
        },
    ]


def make_juggler_check_host(alerting_prefix, namespace_id, location, env_type, alerting_suffix):
    return "{}.{}.{}{}{}".format(
        alerting_prefix,
        normalize_namespace_id(namespace_id),
        location.lower(),
        ENV_TYPE_TO_SUFFIX[env_type],
        alerting_suffix,
    )


class Alert(six.with_metaclass(abc.ABCMeta, object)):
    @abc.abstractproperty
    def name(self):
        pass

    def __init__(self, notify_group):
        """

        :type notify_group: int
        """
        self.notify_group = notify_group


class YasmAlert(Alert):
    NAME_RE = re.compile('[a-zA-Z0-9-_]{,40}$')

    def __init__(self, notify_group, warn_threshold=None, crit_threshold=None, flaps_stable_time=None):
        """

        :type notify_group: int
        :type warn_threshold: tuple[int, int] | None
        :type crit_threshold: tuple[int, int] | None
        :type flaps_stable_time: int | None
        """
        super(YasmAlert, self).__init__(notify_group)
        assert self.NAME_RE.match(self.name), 'yasm alert name should match: {}'.format(self.NAME_RE.pattern)

        self.warn_threshold = warn_threshold or self.default_warn_threshold
        self.crit_threshold = crit_threshold or self.default_crit_threshold
        self.flaps_stable_time = flaps_stable_time or self.default_flaps_stable_time

    @abc.abstractproperty
    def name(self):
        """

        :rtype: six.text_type
        """

    @abc.abstractproperty
    def signal(self):
        """

        :rtype: six.text_type
        """

    @abc.abstractproperty
    def description(self):
        """

        :rtype: six.text_type
        """

    @abc.abstractproperty
    def default_warn_threshold(self):
        """

        :rtype: tuple[int | float, int | float]
        """

    @abc.abstractproperty
    def default_crit_threshold(self):
        """

        :rtype: tuple[int | float, int | float]
        """

    @property
    def default_flaps_stable_time(self):
        return 30

    @property
    def value_modify(self):
        """
        https://wiki.yandex-team.ru/golovan/userdocs/alerts/api/#valuemodify

        :rtype: dict | None
        """
        return None

    def to_json(self, alerting_prefix, alert_name, juggler_namespace, juggler_check_tags,
                itype, ctype, prj, geo, location, balancer_ui_url, balancer_pb, namespace_id, abc_service_slug,
                alerting_suffix):
        """

        :type alerting_prefix: six.text_type
        :type alert_name: six.text_type
        :type juggler_namespace: six.text_type
        :type juggler_check_tags: list[six.text_type]
        :type itype: six.text_type
        :type ctype: six.text_type
        :type prj: six.text_type
        :type geo: six.text_type
        :type location: six.text_type
        :type balancer_ui_url: six.text_type
        :type balancer_pb: model_pb2.Balancer
        :type namespace_id: six.text_type
        :type abc_service_slug: six.text_type
        :type alerting_suffix: six.text_type
        :rtype: dict
        """
        tags = {
            'itype': [itype],
            'ctype': [ctype],
            'prj': [prj],
            'geo': [geo],
        }
        juggler_check = {
            'host': make_juggler_check_host(alerting_prefix, namespace_id,
                                            location, balancer_pb.spec.env_type,
                                            alerting_suffix),
            'service': self.name,
            'namespace': juggler_namespace,
            'tags': juggler_check_tags,
            'flaps': {
                'stable': self.flaps_stable_time,
                'critical': self.flaps_stable_time * 5,
            },
            'meta': {
                'urls': make_juggler_check_meta_urls(
                    balancer_ui_url=balancer_ui_url,
                    actions_wiki_page_anchor=self.name.replace('_', '-'),
                    padding_right=30
                )
            }
        }

        yasm_alert_data = {
            'name': alert_name,
            'abc': abc_service_slug,
            'signal': self.signal,
            'tags': tags,
            'warn': self.warn_threshold if self.warn_threshold else [None, None],
            'crit': self.crit_threshold if self.crit_threshold else [None, None],
            'juggler_check': juggler_check,
            'mgroups': ['ASEARCH'],
        }
        if self.value_modify is not None:
            yasm_alert_data['value_modify'] = self.value_modify

        return yasm_alert_data


class JugglerCheck(Alert):
    @abc.abstractproperty
    def name(self):
        """

        :rtype: six.text_type
        """

    def _find_yp_endpoint(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: Optional[Text]
        """

        if balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.YP_CLUSTER:
            yp_cluster = balancer_pb.meta.location.yp_cluster
        elif balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.AZURE_CLUSTER:
            yp_cluster = external_clusters.AZURE_CLUSTERS_BY_NAME[balancer_pb.meta.location.azure_cluster].yp_cluster
        else:
            return None

        cluster = yp_cluster.lower().replace('_', '-')
        nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        endpoint_candidate = make_system_endpoint_set_id(nanny_service_id)
        if endpoints.endpoint_set_exists(endpoint_candidate, yp_cluster):
            return '{}@cluster={}'.format(endpoint_candidate, cluster)

        return None

    def make_aggregate_children(self, balancer_pb, name=None, instance=''):
        """
        :type balancer_pb: model_pb2.Balancer
        :type name: Optional[Text]
        :type instance: str
        :rtype: List[juggler_sdk.Child]
        """

        nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        yp_endpoint = self._find_yp_endpoint(balancer_pb)
        name = name or self.name
        if yp_endpoint is not None:
            return [
                juggler_sdk.Child(
                    host=yp_endpoint,
                    service=name,
                    group_type='YP_ENDPOINT',
                )
            ]
        else:
            return [
                juggler_sdk.Child(
                    host=nanny_service_id,
                    service=name,
                    instance=instance,
                    group_type='NANNY',
                )
            ]

    @abc.abstractmethod
    def to_juggler_sdk_check(self, alerting_prefix, juggler_namespace, namespace_id, location, juggler_check_tags,
                             balancer_pb, balancer_ui_url, alerting_suffix):
        """
        :type alerting_prefix: six.text_type
        :type juggler_namespace: six.text_type
        :type namespace_id: six.text_type
        :type balancer_pb: model_pb2.Balancer
        :type location: six.text_type
        :type juggler_check_tags: list[six.text_type]
        :type balancer_ui_url: six.text_type
        :type alerting_suffix: six.text_type
        :rtype: juggler_sdk.Check
        """


class CPUGuaranteeUsageYasmAlert(YasmAlert):
    @property
    def name(self):
        return 'cpu_usage'

    @property
    def signal(self):
        return 'quant(portoinst-cpu_guarantee_usage_perc_hgram, 80)'

    @property
    def description(self):
        return "CPU usage"

    @property
    def default_warn_threshold(self):
        return [60, 80]

    @property
    def default_crit_threshold(self):
        return [80, None]


class CPULimitUsageYasmAlert(CPUGuaranteeUsageYasmAlert):
    @property
    def signal(self):
        return 'quant(portoinst-cpu_limit_usage_perc_hgram, 80)'


class CPUWaitCoresYasmAlert(YasmAlert):
    @property
    def name(self):
        return 'cpu_wait_cores'

    @property
    def signal(self):
        return 'quant(portoinst-cpu_wait_slot_hgram, 90)'

    @property
    def description(self):
        return "CPU wait cores"

    @property
    def default_warn_threshold(self):
        return [0.3, 0.4]

    @property
    def default_crit_threshold(self):
        return [0.4, None]


class MemoryUsageYasmAlert(YasmAlert):
    @property
    def name(self):
        return 'mem_usage'

    @property
    def signal(self):
        return 'quant(portoinst-anon_limit_usage_perc_hgram, 90)'

    @property
    def description(self):
        return "Memory usage"

    @property
    def default_warn_threshold(self):
        return [80, 90]

    @property
    def default_crit_threshold(self):
        return [90, None]


class LogsVolumeUsageYasmAlert(YasmAlert):
    @property
    def name(self):
        return 'logs_vol_usage'

    @property
    def signal(self):
        return 'portoinst-volume_/logs_usage_perc_txxx'

    @property
    def description(self):
        return "Logs volume usage"

    @property
    def default_warn_threshold(self):
        return [80, 90]

    @property
    def default_crit_threshold(self):
        return [90, None]


class WorkerCPUUsageYasmAlert(YasmAlert):
    @property
    def name(self):
        return 'worker_cpu_usage'

    @property
    def signal(self):
        return 'max(balancer_report-worker-cpu_usage_hgram)'

    @property
    def description(self):
        return "Worker CPU usage"

    @property
    def default_warn_threshold(self):
        return [70, 90]

    @property
    def default_crit_threshold(self):
        return [90, None]


class WorkerCPUUsageWindowAverYasmAlert(WorkerCPUUsageYasmAlert):

    @property
    def value_modify(self):
        return {
            'type': 'aver',
            'window': 30
        }


class FileDescriptorsUsageYasmAlert(YasmAlert):
    @property
    def name(self):
        return 'fd_usage'

    @property
    def signal(self):
        return 'or(perc(balancer_report-fd_size_ammv, balancer_report-no_file_limit_ammv), 0)'

    @property
    def description(self):
        return "File descriptors usage"

    @property
    def default_warn_threshold(self):
        return [50, 75]

    @property
    def default_crit_threshold(self):
        return [75, None]


class FrozenThreadsYasmAlert(YasmAlert):
    @property
    def name(self):
        return 'frozen_threads'

    @property
    def signal(self):
        return 'or(balancer_report-threads-froze_ammv, 0)'

    @property
    def description(self):
        return "Frozen threads"

    @property
    def default_warn_threshold(self):
        return None

    @property
    def default_crit_threshold(self):
        return [1, None]


class CoredumpsTotalYasmAlert(YasmAlert):
    @property
    def name(self):
        return 'coredumps_total'

    @property
    def signal(self):
        return 'or(hsum(portoinst-cores_total_hgram), 0)'

    @property
    def value_modify(self):
        return {
            'type': 'summ',
            'window': 30
        }

    @property
    def description(self):
        return "Balancer crashes"

    @property
    def default_warn_threshold(self):
        return None

    @property
    def default_crit_threshold(self):
        return [1, None]


class NannyDeployStatusJugglerCheck(JugglerCheck):
    """

    https://wiki.yandex-team.ru/sm/juggler/activechecks/#nannydeploystatus
    """
    DEFAULT_REFRESH_TIME = 300
    DEFAULT_TTL = 1500

    def __init__(self, notify_group, warn_time, crit_time, refresh_time=None, ttl=None):
        """

        :type notify_group: int
        :type warn_time: int
        :type crit_time: int
        :type refresh_time: int
        :type ttl: int
        """
        super(NannyDeployStatusJugglerCheck, self).__init__(notify_group)
        self._warn_time = warn_time
        self._crit_time = crit_time
        self._refresh_time = refresh_time or self.DEFAULT_REFRESH_TIME
        self._ttl = ttl or self.DEFAULT_TTL

    @property
    def name(self):
        return 'deploy_status'

    def to_juggler_sdk_check(self, alerting_prefix, juggler_namespace, namespace_id, location, juggler_check_tags,
                             balancer_pb, balancer_ui_url, alerting_suffix):
        """

        :type alerting_prefix: six.text_type
        :type juggler_namespace: six.text_type
        :type namespace_id: six.text_type
        :type balancer_pb: model_pb2.Balancer
        :type location: six.text_type
        :type juggler_check_tags: list[six.text_type]
        :type balancer_ui_url: six.text_type
        :type alerting_suffix: six.text_type
        :rtype: juggler_sdk.Check
        """
        assert balancer_pb.spec.config_transport.type == model_pb2.NANNY_STATIC_FILE

        nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        nanny_service_ui_url = 'https://nanny.yandex-team.ru/ui/#/services/catalog/{}/'.format(nanny_service_id)
        host = make_juggler_check_host(
            alerting_prefix=alerting_prefix,
            namespace_id=namespace_id,
            location=location,
            env_type=balancer_pb.spec.env_type,
            alerting_suffix=alerting_suffix,
        )

        default_meta_urls = make_juggler_check_meta_urls(
            balancer_ui_url=balancer_ui_url,
            actions_wiki_page_anchor=self.name.replace('_', '-'),
            padding_right=0
        )

        return juggler_sdk.Check(
            service=self.name,
            host=host,
            namespace=juggler_namespace,
            tags=juggler_check_tags,
            active='nanny_deploy_status',
            active_kwargs={
                'service': nanny_service_id,
                'warn': self._warn_time,
                'crit': self._crit_time
            },
            ttl=self._ttl,
            refresh_time=self._refresh_time,
            meta={
                'urls': [{
                    'url': nanny_service_ui_url,
                    'type': 'nanny',
                    'title': u'⚠️Сервис в nanny'
                }] + default_meta_urls
            }
        )


class EnddateCertificateJugglerCheck(JugglerCheck):
    DEFAULT_REFRESH_TIME = 300
    DEFAULT_TTL = 1500
    DEFAULT_STABLE_TIME = 60
    DEFAULT_CRITICAL_TIME = 600

    def __init__(
        self,
        notify_group,
        warn,
        crit,
        refresh_time=None,
        ttl=None,
        stable_time=None,
        critical_time=None,
        depend_on_meta=False,
        crit_days=None,
        warn_days=None,
    ):
        """

        :type notify_group: int
        :type warn: str
        :type crit: str
        :type refresh_time: int
        :type ttl: int
        :type stable_time: int
        :type critical_time: int
        :type depend_on_meta: bool
        :type crit_days: Optional[int]
        :type warn_days: Optional[int]
        """
        super(EnddateCertificateJugglerCheck, self).__init__(notify_group)
        self._warn = warn
        self._crit = crit
        self._refresh_time = refresh_time or self.DEFAULT_REFRESH_TIME
        self._ttl = ttl or self.DEFAULT_TTL
        self._stable_time = stable_time or self.DEFAULT_STABLE_TIME
        self._critical_time = critical_time or self.DEFAULT_CRITICAL_TIME
        self._depend_on_meta = depend_on_meta
        self._crit_days = crit_days
        self._warn_days = warn_days

    @property
    def name(self):
        return 'check_enddate_certificate'

    def get_check_options(self):
        if self._crit_days is None and self._warn_days is None:
            return None

        options = {'env': {}}
        if self._crit_days is not None:
            options['env']['CRIT_THRESHOLD'] = str(self._crit_days)
        if self._warn_days is not None:
            options['env']['WARNING_THRESHOLD'] = str(self._warn_days)

        return options

    def to_juggler_sdk_check(self, alerting_prefix, juggler_namespace, namespace_id, location, juggler_check_tags,
                             balancer_pb, balancer_ui_url, alerting_suffix):
        """

        :type alerting_prefix: six.text_type
        :type juggler_namespace: six.text_type
        :type namespace_id: six.text_type
        :type balancer_pb: model_pb2.Balancer
        :type location: six.text_type
        :type juggler_check_tags: list[six.text_type]
        :type balancer_ui_url: six.text_type
        :type alerting_suffix: six.text_type
        :rtype: juggler_sdk.Check
        """
        assert balancer_pb.spec.config_transport.type == model_pb2.NANNY_STATIC_FILE

        nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        nanny_service_ui_url = 'https://nanny.yandex-team.ru/ui/#/services/catalog/{}/'.format(nanny_service_id)
        host = make_juggler_check_host(
            alerting_prefix=alerting_prefix,
            namespace_id=namespace_id,
            location=location,
            env_type=balancer_pb.spec.env_type,
            alerting_suffix=alerting_suffix,
        )

        default_meta_urls = make_juggler_check_meta_urls(
            balancer_ui_url=balancer_ui_url,
            # TODO write corresponding alert chapter in wiki
            actions_wiki_page_anchor=self.name.replace('_', '-'),
            padding_right=0
        )

        aggregator_kwargs = {
            'ignore_nodata': False,
            'mode': 'normal',
            'crit_limit': self._crit,
            'warn_limit': self._warn,
        }

        if self._depend_on_meta:
            aggregator_kwargs['unreach_mode'] = 'skip'
            aggregator_kwargs['unreach_service'] = [{
                'check': ':META',
                'hold': 600,
            }]

        return juggler_sdk.Check(
            namespace=juggler_namespace,
            host=host,
            service=self.name,
            refresh_time=self._refresh_time,
            ttl=self._ttl,
            aggregator='more_than_limit_is_problem',
            aggregator_kwargs=aggregator_kwargs,
            children=self.make_aggregate_children(balancer_pb),
            flaps_config=juggler_sdk.FlapOptions(
                stable=self._stable_time,
                critical=self._critical_time,
                boost=0,
            ),
            check_options=self.get_check_options(),
            tags=juggler_check_tags,
            meta={
                'urls': [{
                    'url': nanny_service_ui_url,
                    'type': 'nanny',
                    'title': u'⚠️Сервис в nanny'
                }] + default_meta_urls
            }
        )


class MetaJugglerCheck(JugglerCheck):
    DEFAULT_REFRESH_TIME = 90
    DEFAULT_TTL = 900

    def __init__(self, notify_group, warn='20', crit='50', refresh_time=None, ttl=None):
        """

        :type notify_group: int
        :type warn: Union[int, Text]
        :type crit: Union[int, Text]
        :type refresh_time: Optional[int]
        :type ttl: Optional[int]
        """
        super(MetaJugglerCheck, self).__init__(notify_group)
        self._warn = warn
        self._crit = crit
        self._refresh_time = refresh_time or self.DEFAULT_REFRESH_TIME
        self._ttl = ttl or self.DEFAULT_TTL

    @property
    def name(self):
        return 'META'

    def to_juggler_sdk_check(self, alerting_prefix, juggler_namespace, namespace_id, location, juggler_check_tags,
                             balancer_pb, balancer_ui_url, alerting_suffix):
        """

        :type alerting_prefix: six.text_type
        :type juggler_namespace: six.text_type
        :type namespace_id: six.text_type
        :type balancer_pb: model_pb2.Balancer
        :type location: six.text_type
        :type juggler_check_tags: List[Text]
        :type balancer_ui_url: six.text_type
        :type alerting_suffix: six.text_type
        :rtype: juggler_sdk.Check
        """
        assert balancer_pb.spec.config_transport.type == model_pb2.NANNY_STATIC_FILE

        nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        nanny_service_ui_url = 'https://nanny.yandex-team.ru/ui/#/services/catalog/{}/'.format(nanny_service_id)
        host = make_juggler_check_host(
            alerting_prefix=alerting_prefix,
            namespace_id=namespace_id,
            location=location,
            env_type=balancer_pb.spec.env_type,
            alerting_suffix=alerting_suffix,
        )

        default_meta_urls = make_juggler_check_meta_urls(
            balancer_ui_url=balancer_ui_url,
            # TODO write corresponding alert chapter in wiki
            actions_wiki_page_anchor=self.name.replace('_', '-'),
            padding_right=0
        )

        return juggler_sdk.Check(
            namespace=juggler_namespace,
            host=host,
            service=self.name,
            refresh_time=self._refresh_time,
            ttl=self._ttl,
            aggregator='more_than_limit_is_problem',
            aggregator_kwargs={
                'ignore_nodata': False,
                'mode': 'percent' if isinstance(self._crit, six.string_types) else 'normal',
                'crit_limit': self._crit,
                'warn_limit': self._warn,
            },
            children=self.make_aggregate_children(balancer_pb),
            flaps_config=juggler_sdk.FlapOptions(
                stable=60,
                critical=600,
                boost=0,
            ),
            tags=juggler_check_tags,
            meta={
                'urls': [{
                    'url': nanny_service_ui_url,
                    'type': 'nanny',
                    'title': u'⚠️Сервис в nanny'
                }] + default_meta_urls
            }
        )


class HttpCheck(JugglerCheck):
    def __init__(
        self,
        notify_group,
        depend_on_meta=False,
        warn='30',
        crit='50',
        check_local_ports=False,
        ttl=None,
        stable_time=None,
        critical_time=None,
        check_cert_expiration=True,
    ):
        """
        :type notify_group: int
        :type depend_on_meta: bool
        :type warn: Union[int, Text]
        :type crit: Union[int, Text]
        :type ttl: Optional[int]
        :type stable_time: Optional[int]
        :type critical_time: Optional[int]
        :type check_cert_expiration: bool
        """
        super(HttpCheck, self).__init__(notify_group)
        self._depend_on_meta = depend_on_meta
        self._warn = warn
        self._crit = crit
        self._check_local_ports = check_local_ports
        self._ttl = ttl
        self._stable_time = stable_time
        self._critical_time = critical_time
        self._check_cert_expiration = check_cert_expiration

    @property
    def name(self):
        return 'http_check'

    def get_ping_path(self, namespace_id, balancer_pb):
        """
        :type namespace_id: six.text_type
        :type balancer_pb: model_pb2.Balancer
        :rtype: six.text_type
        """
        if (
            balancer_pb.spec.WhichOneof('spec') == 'yandex_balancer'
            and balancer_pb.spec.yandex_balancer.config.WhichOneof('module') == 'l7_macro'
            and balancer_pb.spec.yandex_balancer.config.l7_macro.HasField('health_check_reply')
        ):
            return '/awacs-balancer-health-check'

        upstream_pb = cache.IAwacsCache.instance().get_upstream(namespace_id, 'awacs-balancer-health-check')
        if upstream_pb is None:
            raise AssertionError("Namespace has no 'awacs-balancer-health-check' upstream, each namespace must have one")

        config = upstream_pb.spec.yandex_balancer.config
        if config.HasField('regexp_section'):
            path = config.regexp_section.matcher.match_fsm.uri
        elif config.HasField('regexp_path_section'):
            path = config.regexp_path_section.pattern
        elif config.HasField('prefix_path_router_section'):
            path = config.prefix_path_router_section.route
        elif config.HasField('l7_upstream_macro'):
            path = config.l7_upstream_macro.matcher.path_re
        else:
            path = None

        if not path:
            raise AssertionError("Upstream 'awacs-balancer-health-check' is misconfigured")
        return path

    def get_ping_protocol_and_port(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: Tuple[six.text_type, Optional[int]]
        """
        if not balancer_pb.spec.WhichOneof('spec') == 'yandex_balancer':
            raise AssertionError("Balancer {!r} is not yandex_balancer".format(balancer_pb.meta.id))

        if balancer_pb.spec.yandex_balancer.config.WhichOneof('module') == 'l7_macro':
            macro = balancer_pb.spec.yandex_balancer.config.l7_macro
            if macro.HasField('http'):
                port = 80 if not macro.http.ports or 80 in macro.http.ports else macro.http.ports[0]
                return 'http', port
            elif macro.HasField('https'):
                port = 443 if not macro.https.ports or 443 in macro.https.ports else macro.https.ports[0]
                return 'https', port
            else:
                raise AssertionError("Balancer {!r} l7_macro must have 'http' or 'https' to enable alerting".format(
                    balancer_pb.meta.id,
                ))
        elif balancer_pb.spec.yandex_balancer.config.WhichOneof('module') == 'instance_macro':
            sections = balancer_pb.spec.yandex_balancer.config.instance_macro.sections
            http_section = next(iter(filter(lambda pb: pb.key == 'http_section', sections)), None)
            https_section = next(iter(filter(lambda pb: pb.key == 'https_section', sections)), None)
            if http_section is not None:
                ports = []
                if self._check_local_ports:
                    ports = [port.value for port in http_section.value.local_ports if port.WhichOneof('kind') == 'value']

                    if not ports and len(http_section.value.local_ports):
                        # balancer uses only portvars so we'll omit port value and let juggler find it
                        return 'http', None

                if not ports:
                    ports = [port.value for port in http_section.value.ports if port.WhichOneof('kind') == 'value']
                port = 80 if not ports or 80 in ports else ports[0]
                return 'http', port
            elif https_section:
                ports = []
                if self._check_local_ports:
                    ports = [port.value for port in https_section.value.local_ports if port.WhichOneof('kind') == 'value']

                    if not ports and len(https_section.value.local_ports):
                        # balancer uses only portvars so we'll omit port value and let juggler find it
                        return 'https', None

                if not ports:
                    ports = [port.value for port in https_section.value.ports if port.WhichOneof('kind') == 'value']
                port = 443 if not ports or 443 in ports else ports[0]
                return 'https', port
            else:
                raise AssertionError(
                    "Balancer {!r} instance_macro must have 'http_section' or 'https_section' to enable alerting".format(
                        balancer_pb.meta.id,
                    )
                )
        else:
            raise AssertionError("Balancer {!r} has no instance_macro nor l7_macro".format(balancer_pb.meta.id))

    def to_juggler_sdk_check(self, alerting_prefix, juggler_namespace, namespace_id, location, juggler_check_tags,
                             balancer_pb, balancer_ui_url, alerting_suffix):
        """

        :type alerting_prefix: six.text_type
        :type juggler_namespace: six.text_type
        :type namespace_id: six.text_type
        :type location: six.text_type
        :type juggler_check_tags: list[six.text_type]
        :type balancer_pb: model_pb2.Balancer
        :type balancer_ui_url: six.text_type
        :type alerting_suffix: six.text_type
        :rtype: juggler_sdk.Check
        """

        path = self.get_ping_path(namespace_id, balancer_pb)

        nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        nanny_service_ui_url = 'https://nanny.yandex-team.ru/ui/#/services/catalog/{}/'.format(nanny_service_id)
        host = make_juggler_check_host(
            alerting_prefix=alerting_prefix,
            namespace_id=namespace_id,
            location=location,
            env_type=balancer_pb.spec.env_type,
            alerting_suffix=alerting_suffix,
        )

        default_meta_urls = make_juggler_check_meta_urls(
            balancer_ui_url=balancer_ui_url,
            # TODO write corresponding alert chapter in wiki
            actions_wiki_page_anchor=self.name.replace('_', '-'),
            padding_right=0
        )

        aggregator_kwargs = {
            'ignore_nodata': True,
            'mode': 'percent' if isinstance(self._crit, six.string_types) else 'normal',
            'crit_limit': self._crit,
            'warn_limit': self._warn,
        }
        if self._depend_on_meta:
            aggregator_kwargs['unreach_mode'] = 'skip'
            aggregator_kwargs['unreach_service'] = [{
                'check': ':META',
                'hold': 600,
            }]

        protocol, port = self.get_ping_protocol_and_port(balancer_pb)

        active_kwargs = {
            'ok_codes': [200],
            'path': path,
        }
        if port is not None:
            active_kwargs['port'] = port

        if protocol == 'https':
            active_kwargs['validate_hostname'] = False
            active_kwargs['allow_self_signed'] = True
            if not self._check_cert_expiration:
                active_kwargs['warn_expire'] = 0
                active_kwargs['crit_expire'] = 0

        check = juggler_sdk.Check(
            namespace=juggler_namespace,
            host=host,
            service=self.name,
            active=protocol,
            active_kwargs=active_kwargs,
            aggregator='more_than_limit_is_problem',
            aggregator_kwargs=aggregator_kwargs,
            children=self.make_aggregate_children(balancer_pb, instance='all' if port is None else ''),
            tags=juggler_check_tags,
            ttl=self._ttl,
            meta={
                'urls': [{
                    'url': nanny_service_ui_url,
                    'type': 'nanny',
                    'title': u'⚠️Сервис в nanny'
                }] + default_meta_urls
            }
        )
        if self._stable_time is not None:
            check.flaps_config = juggler_sdk.FlapOptions(
                stable=self._stable_time,
                critical=self._critical_time,
                boost=0,
            )
        return check


class NamespaceAlertingConfig(object):
    def __init__(self, version, alerting_prefix, namespace_id):
        self.config = get_config(version)
        self.alerting_prefix = alerting_prefix
        self.namespace_id = namespace_id
        self._yasm_alert_prefix = None
        self._juggler_namespace = None
        self._juggler_checks_mark = None

    @property
    def yasm_alert_prefix(self):
        if self._yasm_alert_prefix is None:
            self._yasm_alert_prefix = self.config.build_yasm_alert_prefix(self.alerting_prefix, self.namespace_id)
        return self._yasm_alert_prefix

    @property
    def juggler_namespace(self):
        if self._juggler_namespace is None:
            self._juggler_namespace = self.config.build_juggler_namespace(self.alerting_prefix, self.namespace_id)
        return self._juggler_namespace

    @property
    def juggler_checks_mark(self):
        if self._juggler_checks_mark is None:
            self._juggler_checks_mark = self.config.build_juggler_checks_mark(self.alerting_prefix, self.namespace_id)
        return self._juggler_checks_mark


class AlertingConfig(object):
    VALID_NOTIFY_GROUPS = [BALANCER_GROUP, PLATFORM_GROUP]
    DEFAULT_MONITORING_SLUG_ALPHABET = lowercase + string.digits
    DEFAULT_MONITORING_SLUG_LENGTH = 8

    LOCATION_TO_GEO = {
        'sas': 'sas',
        'man': 'man',
        'vla': 'vla',
        'iva': 'msk',
        'myt': 'msk',
        'test_sas': 'sas',
        'man_pre': 'man',
    }

    @classmethod
    def build_juggler_namespace(cls, alerting_prefix, namespace_id):
        """
        _namespace_regex = re.compile(r'^[.\-\w_]+$')
        https://bb.yandex-team.ru/projects/JUGGLER/repos/juggler/browse/juggler/validators/system.py#130
        MAX_NAMESPACE_NAME_LEN = 128

        find parent namespace:
        https://bb.yandex-team.ru/projects/JUGGLER/repos/juggler/browse/juggler/namespaces.py#15


        :type alerting_prefix: six.text_type
        :type namespace_id: six.text_type
        :rtype: six.text_type
        """
        return u"{}.{}".format(alerting_prefix, normalize_namespace_id(namespace_id))

    @classmethod
    def build_juggler_check_namespace_tag(cls, alerting_prefix, namespace_id):
        """
        _tag_regex = re.compile(r"^[\w\-+._]+$")
        MAX_TAG_NAME_LEN = 126

        :type alerting_prefix: six.text_type
        :type namespace_id: six.text_type
        :rtype: six.text_type
        """
        return u'{}_namespace_id_{}'.format(alerting_prefix, namespace_id)

    @classmethod
    def build_juggler_check_notify_group_tag(cls, alerting_prefix, notify_group):
        """

        :type alerting_prefix: six.text_type
        :type notify_group: int
        :rtype: six.text_type
        """
        return u'{}_notify_group_{}'.format(alerting_prefix, notify_group_str(notify_group).lower())

    @classmethod
    def build_juggler_check_balancer_tag(cls, alerting_prefix, balancer_id):
        """

        :type alerting_prefix: six.text_type
        :type balancer_id: six.text_type
        :rtype: six.text_type
        """
        return u'{}_balancer_id_{}'.format(alerting_prefix, balancer_id)

    @classmethod
    def build_juggler_checks_mark(cls, alerting_prefix, namespace_id):
        """

        :type alerting_prefix: six.text_type
        :type namespace_id: six.text_type
        :rtype: six.text_type
        """
        return "{}_{}_checks".format(alerting_prefix, normalize_namespace_id(namespace_id))

    @classmethod
    def build_yasm_alert_prefix(cls, alerting_prefix, namespace_id):
        """

        :type alerting_prefix: six.text_type
        :type namespace_id: six.text_type
        :rtype: six.text_type
        """
        return u"{}.{}.".format(alerting_prefix, normalize_namespace_id(namespace_id))

    @classmethod
    def build_yasm_alert_name(cls, yasm_alert_prefix, location, env_type, yasm_alert, yasm_alert_suffix):
        """

        yasm alert name match:  "[a-zA-Z0-9@+-./_]{1,128}
        balancer match ^[A-z][A-z0-9-._]{,100}$

        1 - [dev-]awacs.{8}.         15
        2 - balancer_id              70 + 1
        3 -                          42
        max name len 128

        :type yasm_alert_prefix: six.text_type
        :type location: six.text_type
        :type yasm_alert: YasmAlert
        :type yasm_alert_suffix: six.text_type
        :rtype: six.text_type
        """
        suffix = ENV_TYPE_TO_SUFFIX[env_type]
        return u"{}{}{}{}.{}".format(yasm_alert_prefix, location.lower(), suffix, yasm_alert_suffix, yasm_alert.name)

    @classmethod
    def balancer_location_to_geo(cls, location):
        return cls.LOCATION_TO_GEO[location.lower()]

    @classmethod
    def get_juggler_raw_notify_rules(cls, alerting_settings):
        """

        :type alerting_settings: model_pb2.NamespaceSpec.AlertingSettings
        :rtype: list[(int, model_pb2.NamespaceSpec.AlertingSettings.JugglerRawNotifyRule)]
        """
        notify_rules = []

        notify_rules_field = alerting_settings.WhichOneof('notify_rules')
        if notify_rules_field is None:
            return notify_rules

        if notify_rules_field == 'juggler_raw_notify_rules':
            notify_rules = list(chain(
                [(BALANCER_GROUP, pb2) for pb2 in alerting_settings.juggler_raw_notify_rules.balancer],
                [(PLATFORM_GROUP, pb2) for pb2 in alerting_settings.juggler_raw_notify_rules.platform],
            ))
        elif notify_rules_field == 'notify_rules_disabled':
            pass
        else:
            raise NotImplementedError('NotifyRules from "{}" field not implemented!'.format(notify_rules_field))

        return notify_rules

    @classmethod
    def get_juggler_raw_downtimers(cls, alerting_settings):
        """

        :type alerting_settings: model_pb2.NamespaceSpec.AlertingSettings
        :rtype: model_pb2.NamespaceSpec.AlertingSettings.JugglerRawDowntimers
        """
        raw_downtimers = model_pb2.NamespaceSpec.AlertingSettings.JugglerRawDowntimers()

        downtimers_field = alerting_settings.WhichOneof('downtimers')

        if downtimers_field is None:
            return raw_downtimers

        if downtimers_field == 'juggler_raw_downtimers':
            staff_logins = sorted(set(alerting_settings.juggler_raw_downtimers.staff_logins))
            alerting_settings.juggler_raw_downtimers.staff_logins[:] = staff_logins

            staff_group_ids = sorted(set(alerting_settings.juggler_raw_downtimers.staff_group_ids))
            alerting_settings.juggler_raw_downtimers.staff_group_ids[:] = staff_group_ids

            raw_downtimers.CopyFrom(alerting_settings.juggler_raw_downtimers)
        else:
            raise NotImplementedError('Downtimers from "{}" field not implemented!'.format(downtimers_field))

        return raw_downtimers

    def __init__(self, version, alerts):
        """

        :type version: semantic_version.Version
        :type alerts: list[YasmAlert | JugglerCheck]
        """
        assert isinstance(version, semantic_version.Version)
        self._version = version
        self._yasm_alerts_by_groups = None
        self._juggler_checks_by_groups = None
        self._prepare_alerts(alerts)

    @property
    def version(self):
        return self._version

    @property
    def yasm_alerts_by_groups(self):
        """

        :rtype: dict[int, list[YasmAlert]]
        """
        return self._yasm_alerts_by_groups

    @property
    def juggler_checks_by_groups(self):
        """

        :rtype: dict[int, list[JugglerCheck]]
        """
        return self._juggler_checks_by_groups

    def _prepare_alerts(self, alerts):
        """

        :type alerts: list[YasmAlert | JugglerCheck]
        """
        self._yasm_alerts_by_groups = {group: [] for group in self.VALID_NOTIFY_GROUPS}
        self._juggler_checks_by_groups = {group: [] for group in self.VALID_NOTIFY_GROUPS}
        alert_names = [a.name for a in alerts]
        if len(alert_names) != len(set(alert_names)):
            raise ValueError('Alerts names are not unique: {}'.format(sorted(alert_names)))

        for alert in alerts:
            assert isinstance(alert, Alert)
            assert alert.notify_group in self.VALID_NOTIFY_GROUPS

            if isinstance(alert, YasmAlert):
                self._yasm_alerts_by_groups[alert.notify_group].append(alert)
            elif isinstance(alert, JugglerCheck):
                self._juggler_checks_by_groups[alert.notify_group].append(alert)
            else:
                raise NotImplementedError('Alert with type {} is not supported'.format(alert.__class__.__name__))

    def gen_balancer_yasm_alerts(self, alerting_prefix, yasm_alert_prefix, juggler_namespace,
                                 juggler_check_tags, itype, ctype, prj, location, balancer_ui_url,
                                 balancer_pb, namespace_id, abc_service_slug, yasm_alert_suffix,
                                 check_groups=None):
        """

        :type alerting_prefix: six.text_type
        :type yasm_alert_prefix: six.text_type
        :type juggler_namespace: six.text_type
        :type juggler_check_tags: list[six.text_type]
        :type itype: six.text_type
        :type ctype: six.text_type
        :type prj: six.text_type
        :type location: six.text_type
        :type balancer_ui_url: six.text_type
        :type balancer_pb: model_pb2.Balancer
        :type namespace_id: six.text_type
        :type abc_service_slug: six.text_type
        :type yasm_alert_suffix: six.text_type
        :type check_groups: Set[model_pb2.NamespaceSpec.AlertingSettings.NotifyRuleGroup]
        :rtype: list[dict]
        """
        check_groups = check_groups or self.VALID_NOTIFY_GROUPS
        result = []
        for notify_group, yasm_alerts in self.yasm_alerts_by_groups.items():
            if notify_group not in check_groups:
                continue

            default_tags = [
                self.build_juggler_check_notify_group_tag(alerting_prefix, notify_group)
            ]
            for yasm_alert in yasm_alerts:
                alert_name = self.build_yasm_alert_name(yasm_alert_prefix, location,
                                                        balancer_pb.spec.env_type, yasm_alert, yasm_alert_suffix)
                result.append(yasm_alert.to_json(
                    alerting_prefix=alerting_prefix,
                    alert_name=alert_name,
                    juggler_namespace=juggler_namespace,
                    juggler_check_tags=sorted(set(default_tags + juggler_check_tags)),
                    itype=itype,
                    ctype=ctype,
                    prj=prj,
                    geo=self.balancer_location_to_geo(location),
                    location=location,
                    balancer_ui_url=balancer_ui_url,
                    balancer_pb=balancer_pb,
                    namespace_id=namespace_id,
                    abc_service_slug=abc_service_slug,
                    alerting_suffix=yasm_alert_suffix,
                ))
        return result

    def gen_balancer_juggler_checks(self, alerting_prefix, juggler_namespace, juggler_check_tags, location,
                                    balancer_ui_url, namespace_id, balancer_pb, alerting_suffix,
                                    check_groups=None):
        """

        :param alerting_prefix:
        :param juggler_namespace:
        :param juggler_check_tags:
        :param location:
        :param balancer_ui_url:
        :param namespace_id:
        :param balancer_pb:
        :param alerting_suffix:
        :param check_groups:
        :type check_groups: Set[model_pb2.NamespaceSpec.AlertingSettings.NotifyRuleGroup]
        :rtype: list[juggler_sdk.Check]
        """
        check_groups = check_groups or self.VALID_NOTIFY_GROUPS
        result = []
        for notify_group, juggler_check_builders in self.juggler_checks_by_groups.items():
            if notify_group not in check_groups:
                continue

            default_tags = [
                self.build_juggler_check_notify_group_tag(alerting_prefix, notify_group)
            ]
            for juggler_check_builder in juggler_check_builders:
                result.append(juggler_check_builder.to_juggler_sdk_check(
                    alerting_prefix=alerting_prefix,
                    juggler_namespace=juggler_namespace,
                    namespace_id=namespace_id,
                    location=location,
                    juggler_check_tags=sorted(set(default_tags + juggler_check_tags)),
                    balancer_pb=balancer_pb,
                    balancer_ui_url=balancer_ui_url,
                    alerting_suffix=alerting_suffix,
                ))

        return result

    def gen_juggler_notify_rules(self, namespace_id, alerting_prefix, alerting_settings, description=None):
        """

        :type namespace_id: six.text_type
        :type alerting_prefix: six.text_type
        :type alerting_settings: model_pb2.NamespaceSpec.AlertingSettings
        :type description: six.text_type | None
        :rtype: list[tuple[juggler_client.NotifyRule, int, model_pb2.NamespaceSpec.AlertingSettings.JugglerRawNotifyRule]]
        """
        juggler_raw_notify_rules = self.get_juggler_raw_notify_rules(alerting_settings)

        juggler_notify_rules = []
        for notify_rule_group, notify_rule_pb in juggler_raw_notify_rules:
            notify_rule = juggler_client.NotifyRule(
                selector=" & ".join([
                    'tag={}'.format(
                        self.build_juggler_check_notify_group_tag(alerting_prefix, notify_rule_group)),
                    'tag={}'.format(
                        self.build_juggler_check_namespace_tag(alerting_prefix, namespace_id))
                ]),
                template_name=notify_rule_pb.template_name,
                template_kwargs=yaml.safe_load(notify_rule_pb.template_kwargs),
                description=description
            )
            juggler_notify_rules.append((notify_rule, notify_rule_group, notify_rule_pb))

        return sorted(juggler_notify_rules, key=lambda x: (x[0].selector, x[0].template_name, x[0].template_kwargs_json))


VERSION_0_0_1 = semantic_version.Version('0.0.1')
VERSION_0_0_2 = semantic_version.Version('0.0.2')
VERSION_0_0_3 = semantic_version.Version('0.0.3')
VERSION_0_0_4 = semantic_version.Version('0.0.4')
VERSION_0_0_5 = semantic_version.Version('0.0.5')
VERSION_0_0_6 = semantic_version.Version('0.0.6')
VERSION_0_0_7 = semantic_version.Version('0.0.7')
VERSION_0_1_0 = semantic_version.Version('0.1.0')
VERSION_0_1_1 = semantic_version.Version('0.1.1')
VERSION_0_1_2 = semantic_version.Version('0.1.2')
VERSION_0_1_3 = semantic_version.Version('0.1.3')
VERSION_0_1_4 = semantic_version.Version('0.1.4')
VERSION_0_2_0 = semantic_version.Version('0.2.0')
VERSION_0_2_1 = semantic_version.Version('0.2.1')
VERSION_0_2_2 = semantic_version.Version('0.2.2')
VERSION_0_2_3 = semantic_version.Version('0.2.3')

_CONFIGS = [
    AlertingConfig(
        version=VERSION_0_0_1,
        alerts=[
            CPUGuaranteeUsageYasmAlert(BALANCER_GROUP),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            LogsVolumeUsageYasmAlert(BALANCER_GROUP),

            WorkerCPUUsageYasmAlert(PLATFORM_GROUP)
        ]
    ),
    AlertingConfig(
        version=VERSION_0_0_2,
        alerts=[
            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            LogsVolumeUsageYasmAlert(BALANCER_GROUP),

            WorkerCPUUsageYasmAlert(PLATFORM_GROUP)
        ]
    ),
    AlertingConfig(
        version=VERSION_0_0_3,
        alerts=[
            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            LogsVolumeUsageYasmAlert(BALANCER_GROUP),

            WorkerCPUUsageYasmAlert(PLATFORM_GROUP),
            NannyDeployStatusJugglerCheck(PLATFORM_GROUP, warn_time=60 * 30, crit_time=60 * 60)
        ]
    ),
    AlertingConfig(
        version=VERSION_0_0_4,
        alerts=[
            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            LogsVolumeUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),

            WorkerCPUUsageWindowAverYasmAlert(PLATFORM_GROUP),

        ]
    ),
    AlertingConfig(
        version=VERSION_0_0_5,
        alerts=[
            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            LogsVolumeUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),

            WorkerCPUUsageWindowAverYasmAlert(PLATFORM_GROUP),
            FileDescriptorsUsageYasmAlert(PLATFORM_GROUP),
            FrozenThreadsYasmAlert(PLATFORM_GROUP),

        ]
    ),
    AlertingConfig(
        version=VERSION_0_0_6,
        alerts=[
            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            LogsVolumeUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),

            WorkerCPUUsageWindowAverYasmAlert(PLATFORM_GROUP),
            FileDescriptorsUsageYasmAlert(PLATFORM_GROUP),
            FrozenThreadsYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1),
        ]
    ),
    AlertingConfig(
        version=VERSION_0_0_7,
        alerts=[
            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            LogsVolumeUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),

            WorkerCPUUsageWindowAverYasmAlert(PLATFORM_GROUP),
            FileDescriptorsUsageYasmAlert(PLATFORM_GROUP),
            FrozenThreadsYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1),
        ]
    ),
    AlertingConfig(
        version=VERSION_0_1_0,
        alerts=[
            MetaJugglerCheck(PLATFORM_GROUP),

            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            LogsVolumeUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),

            WorkerCPUUsageWindowAverYasmAlert(PLATFORM_GROUP),
            FileDescriptorsUsageYasmAlert(PLATFORM_GROUP),
            FrozenThreadsYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1, depend_on_meta=True),
        ]
    ),
    AlertingConfig(
        version=VERSION_0_1_1,
        alerts=[
            MetaJugglerCheck(PLATFORM_GROUP),

            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            LogsVolumeUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),

            FileDescriptorsUsageYasmAlert(PLATFORM_GROUP),
            FrozenThreadsYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1, depend_on_meta=True),
        ]
    ),
    AlertingConfig(
        version=VERSION_0_1_2,
        alerts=[
            MetaJugglerCheck(PLATFORM_GROUP),

            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),

            LogsVolumeUsageYasmAlert(PLATFORM_GROUP),
            FileDescriptorsUsageYasmAlert(PLATFORM_GROUP),
            FrozenThreadsYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1, depend_on_meta=True),
        ]
    ),
    AlertingConfig(
        version=VERSION_0_1_3,
        alerts=[
            MetaJugglerCheck(PLATFORM_GROUP),

            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),

            LogsVolumeUsageYasmAlert(PLATFORM_GROUP),
            FileDescriptorsUsageYasmAlert(PLATFORM_GROUP),
            FrozenThreadsYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1,
                                           ttl=10000, stable_time=1200, critical_time=6000, depend_on_meta=True),
        ]
    ),
    AlertingConfig(
        version=VERSION_0_1_4,
        alerts=[
            MetaJugglerCheck(PLATFORM_GROUP),

            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),

            LogsVolumeUsageYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1,
                                           ttl=10000, stable_time=1200, critical_time=6000, depend_on_meta=True),
        ]
    ),
    AlertingConfig(
        version=VERSION_0_2_0,
        alerts=[
            MetaJugglerCheck(PLATFORM_GROUP),

            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),
            HttpCheck(BALANCER_GROUP, depend_on_meta=True, warn='30', crit='50'),

            LogsVolumeUsageYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1,
                                           ttl=10000, stable_time=1200, critical_time=6000, depend_on_meta=True),
        ],
    ),
    AlertingConfig(
        version=VERSION_0_2_1,
        alerts=[
            MetaJugglerCheck(PLATFORM_GROUP),

            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),
            HttpCheck(BALANCER_GROUP, depend_on_meta=True, warn='30', crit='50', check_local_ports=True,
                      ttl=1800, stable_time=180, critical_time=900),

            LogsVolumeUsageYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1,
                                           ttl=10000, stable_time=1200, critical_time=6000, depend_on_meta=True),
        ],
    ),
    AlertingConfig(
        version=VERSION_0_2_2,
        alerts=[
            MetaJugglerCheck(PLATFORM_GROUP),

            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),
            HttpCheck(BALANCER_GROUP, depend_on_meta=True, warn='30', crit='50', check_local_ports=True,
                      ttl=1800, stable_time=180, critical_time=900),

            LogsVolumeUsageYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1,
                                           ttl=10000, stable_time=1200, critical_time=6000, depend_on_meta=True,
                                           crit_days=2, warn_days=14,
                                           ),
        ],
    ),
    AlertingConfig(
        version=VERSION_0_2_3,
        alerts=[
            MetaJugglerCheck(PLATFORM_GROUP),

            CPULimitUsageYasmAlert(BALANCER_GROUP, flaps_stable_time=120),
            CPUWaitCoresYasmAlert(BALANCER_GROUP),
            MemoryUsageYasmAlert(BALANCER_GROUP),
            NannyDeployStatusJugglerCheck(BALANCER_GROUP, warn_time=60 * 60, crit_time=60 * 60 * 24 * 365),
            HttpCheck(BALANCER_GROUP, depend_on_meta=True, warn='30', crit='50', check_local_ports=True,
                      ttl=1800, stable_time=180, critical_time=900, check_cert_expiration=False),

            LogsVolumeUsageYasmAlert(PLATFORM_GROUP),
            CoredumpsTotalYasmAlert(PLATFORM_GROUP),
            EnddateCertificateJugglerCheck(PLATFORM_GROUP, warn=1, crit=1,
                                           ttl=10000, stable_time=1200, critical_time=6000, depend_on_meta=True,
                                           crit_days=2, warn_days=14,
                                           ),
        ],
    ),
]

CURRENT_VERSION = VERSION_0_2_3

_CONFIGS_BY_VERSIONS = {
    c.version: c for c in _CONFIGS
}


def get_config(version):
    """
    Return Alerting Config

    :type version: six.text_type | semantic_version.Version
    :rtype: AlertingConfig
    """
    if isinstance(version, six.string_types):
        version = semantic_version.Version(version)
    if version not in _CONFIGS_BY_VERSIONS:
        raise AssertionError('AlertingConfig with version "{}" does not exists'.format(version))
    return _CONFIGS_BY_VERSIONS[version]


def get_versions():
    return sorted(_CONFIGS_BY_VERSIONS.keys())
