# coding: utf-8
import collections

import inject
import six

from awacs.lib import pagination
from awacs.lib.gutils import gevent_idle_iter
from awacs.lib.strutils import flatten_full_id
from awacs.model import util, errors, cache, objects
from infra.awacs.proto import model_pb2


class IAwacsApiCache(object):
    @classmethod
    def instance(cls):
        """
        :rtype: AwacsApiCache
        """
        return inject.instance(cls)


def copy_statuses(entity_pb, balancer_state_pb):
    """
    :type entity_pb: model_pb2.Balancer | model_pb2.Upstream | model_pb2.Backend | model_pb2.EndpointSet | model_pb2.Knob | model_pb2.Certificate
    :type balancer_state_pb: model_pb2.BalancerState
    :return:
    """
    entity_pb = util.clone_pb(entity_pb)
    entity_pb.ClearField('status')
    entity_id = entity_pb.meta.id
    namespace_id = entity_pb.meta.namespace_id

    flat_entity_id = flatten_full_id(balancer_state_pb.namespace_id, (namespace_id, entity_id))
    if isinstance(entity_pb, model_pb2.Balancer):
        statuses_pb = balancer_state_pb.balancer
    elif isinstance(entity_pb, model_pb2.Upstream):
        if entity_id not in balancer_state_pb.upstreams:
            return entity_pb
        statuses_pb = balancer_state_pb.upstreams[entity_id]
    elif isinstance(entity_pb, model_pb2.Domain):
        if flat_entity_id not in balancer_state_pb.domains:
            return entity_pb
        statuses_pb = balancer_state_pb.domains[flat_entity_id]
    elif isinstance(entity_pb, model_pb2.Backend):
        if flat_entity_id not in balancer_state_pb.backends:
            return entity_pb
        statuses_pb = balancer_state_pb.backends[flat_entity_id]
    elif isinstance(entity_pb, model_pb2.EndpointSet):
        if flat_entity_id not in balancer_state_pb.endpoint_sets:
            return entity_pb
        statuses_pb = balancer_state_pb.endpoint_sets[flat_entity_id]
    elif isinstance(entity_pb, model_pb2.Knob):
        if flat_entity_id not in balancer_state_pb.knobs:
            return entity_pb
        statuses_pb = balancer_state_pb.knobs[flat_entity_id]
    elif isinstance(entity_pb, model_pb2.Certificate):
        if flat_entity_id not in balancer_state_pb.certificates:
            return entity_pb
        statuses_pb = balancer_state_pb.certificates[flat_entity_id]
    else:
        raise RuntimeError()

    if entity_pb.meta.version not in {s.revision_id for s in statuses_pb.statuses}:
        balancer_state_pb = util.clone_pb(balancer_state_pb)
        if isinstance(entity_pb, model_pb2.Balancer):
            status_pbs = balancer_state_pb.balancer.statuses
        elif isinstance(entity_pb, model_pb2.Upstream):
            status_pbs = balancer_state_pb.upstreams[entity_id].statuses
        elif isinstance(entity_pb, model_pb2.Domain):
            status_pbs = balancer_state_pb.domains[flat_entity_id].statuses
        elif isinstance(entity_pb, model_pb2.Backend):
            status_pbs = balancer_state_pb.backends[flat_entity_id].statuses
        elif isinstance(entity_pb, model_pb2.EndpointSet):
            status_pbs = balancer_state_pb.endpoint_sets[flat_entity_id].statuses
        elif isinstance(entity_pb, model_pb2.Knob):
            status_pbs = balancer_state_pb.knobs[flat_entity_id].statuses
        elif isinstance(entity_pb, model_pb2.Certificate):
            status_pbs = balancer_state_pb.certificates[flat_entity_id].statuses
        else:
            raise RuntimeError()
        state_rev_status_pb = status_pbs.add()
        state_rev_status_pb.validated.status = 'Unknown'
        state_rev_status_pb.in_progress.status = 'False'
        state_rev_status_pb.active.status = 'False'
        state_rev_status_pb.revision_id = entity_pb.meta.version
        state_rev_status_pb.ctime.CopyFrom(entity_pb.meta.mtime)

    if isinstance(entity_pb, model_pb2.Balancer):
        statuses_pb = balancer_state_pb.balancer
    elif isinstance(entity_pb, model_pb2.Upstream):
        statuses_pb = balancer_state_pb.upstreams[entity_id]
    elif isinstance(entity_pb, model_pb2.Domain):
        statuses_pb = balancer_state_pb.domains[flat_entity_id]
    elif isinstance(entity_pb, model_pb2.Backend):
        statuses_pb = balancer_state_pb.backends[flat_entity_id]
    elif isinstance(entity_pb, model_pb2.EndpointSet):
        statuses_pb = balancer_state_pb.endpoint_sets[flat_entity_id]
    elif isinstance(entity_pb, model_pb2.Knob):
        statuses_pb = balancer_state_pb.knobs[flat_entity_id]
    elif isinstance(entity_pb, model_pb2.Certificate):
        statuses_pb = balancer_state_pb.certificates[flat_entity_id]
    else:
        raise RuntimeError()

    for state_rev_status_pb in statuses_pb.statuses:
        if state_rev_status_pb.revision_id == entity_pb.meta.version:
            entity_pb.status.validated.CopyFrom(state_rev_status_pb.validated)
            entity_pb.status.in_progress.CopyFrom(state_rev_status_pb.in_progress)
            entity_pb.status.active.CopyFrom(state_rev_status_pb.active)
        balancer_rev_status_pb = entity_pb.status.revisions.add(id=state_rev_status_pb.revision_id)
        balancer_rev_status_pb.ctime.CopyFrom(state_rev_status_pb.ctime)
        balancer_rev_status_pb.validated.CopyFrom(state_rev_status_pb.validated)
        balancer_rev_status_pb.in_progress.CopyFrom(state_rev_status_pb.in_progress)
        balancer_rev_status_pb.active.CopyFrom(state_rev_status_pb.active)
    return entity_pb


def construct_per_balancer_statuses(entity_pb, balancer_state_pbs):
    """
    Constructs a list of *RevisionStatusPerBalancer, not modifying `entity_pb`.

    :type entity_pb: model_pb2.Upstream | model_pb2.Backend | model_pb2.EndpointSet | model_pb2.Knob | model_pb2.Certificate | model_pb2.WeightSection
    :param balancer_state_pbs: Balancer states to take statuses from
    :type balancer_state_pbs: list[model_pb2.BalancerState]
    :rtype: list[*RevisionStatusPerBalancer]
    """
    entity_id = entity_pb.meta.id
    namespace_id = entity_pb.meta.namespace_id

    revs = collections.defaultdict(set)
    validated = collections.defaultdict(dict)
    in_progress = collections.defaultdict(dict)
    active = collections.defaultdict(dict)

    for balancer_state_pb in gevent_idle_iter(balancer_state_pbs, idle_period=100):
        flat_entity_id = flatten_full_id(balancer_state_pb.namespace_id, (namespace_id, entity_id))
        if isinstance(entity_pb, model_pb2.Upstream):
            if balancer_state_pb.namespace_id == namespace_id and entity_id in balancer_state_pb.upstreams:
                status_pbs = balancer_state_pb.upstreams[entity_id].statuses
            else:
                continue
        elif isinstance(entity_pb, model_pb2.Backend):
            if flat_entity_id in balancer_state_pb.backends:
                status_pbs = balancer_state_pb.backends[flat_entity_id].statuses
            else:
                continue
        elif isinstance(entity_pb, model_pb2.Domain):
            if flat_entity_id in balancer_state_pb.domains:
                status_pbs = balancer_state_pb.domains[flat_entity_id].statuses
            else:
                continue
        elif isinstance(entity_pb, model_pb2.EndpointSet):
            if flat_entity_id in balancer_state_pb.backends:
                status_pbs = balancer_state_pb.endpoint_sets[flat_entity_id].statuses
            else:
                continue
        elif isinstance(entity_pb, model_pb2.Knob):
            if flat_entity_id in balancer_state_pb.knobs:
                status_pbs = balancer_state_pb.knobs[flat_entity_id].statuses
            else:
                continue
        elif isinstance(entity_pb, model_pb2.Certificate):
            if flat_entity_id in balancer_state_pb.certificates:
                status_pbs = balancer_state_pb.certificates[flat_entity_id].statuses
            else:
                continue
        elif isinstance(entity_pb, model_pb2.WeightSection):
            if flat_entity_id in balancer_state_pb.weight_sections:
                status_pbs = balancer_state_pb.weight_sections[flat_entity_id].statuses
            else:
                continue
        else:
            raise RuntimeError()

        for status_pb in status_pbs:
            rev_id = status_pb.revision_id
            revs[rev_id].add(status_pb.ctime.ToNanoseconds())
            key = balancer_state_pb.namespace_id + ':' + balancer_state_pb.balancer_id
            validated[rev_id][key] = status_pb.validated
            in_progress[rev_id][key] = status_pb.in_progress
            active[rev_id][key] = status_pb.active

    for rev_id, ctimes in revs.items():
        assert len(ctimes) == 1
        revs[rev_id] = list(ctimes)[0]

    curr_rev_id = entity_pb.meta.version
    if curr_rev_id not in revs:
        revs[curr_rev_id] = entity_pb.meta.mtime.ToNanoseconds()
        validated[curr_rev_id] = in_progress[curr_rev_id] = active[curr_rev_id] = {}

    status_pbs = []
    for rev_id, ctime in revs.items():
        if isinstance(entity_pb, model_pb2.Knob):
            message_cls = model_pb2.KnobRevisionStatusPerBalancer
        elif isinstance(entity_pb, model_pb2.Upstream):
            message_cls = model_pb2.UpstreamRevisionStatusPerBalancer
        elif isinstance(entity_pb, model_pb2.Domain):
            message_cls = model_pb2.DomainRevisionStatusPerBalancer
        elif isinstance(entity_pb, model_pb2.Backend):
            message_cls = model_pb2.BackendRevisionStatusPerBalancer
        elif isinstance(entity_pb, model_pb2.EndpointSet):
            message_cls = model_pb2.EndpointSetRevisionStatusPerBalancer
        elif isinstance(entity_pb, model_pb2.Certificate):
            message_cls = model_pb2.CertificateRevisionStatusPerBalancer
        elif isinstance(entity_pb, model_pb2.WeightSection):
            message_cls = model_pb2.WeightSectionRevisionStatusPerBalancer
        else:
            raise RuntimeError()

        rev_pb = message_cls(
            id=rev_id,
            validated=validated[rev_id],
            in_progress=in_progress[rev_id],
            active=active[rev_id],
        )
        rev_pb.ctime.FromNanoseconds(ctime)
        status_pbs.append(rev_pb)

    return status_pbs


def set_per_balancer_statuses(entity_pb, status_pbs):
    """
    :type entity_pb: model_pb2.Upstream | model_pb2.Backend | model_pb2.EndpointSet | model_pb2.Knob | model_pb2.Certificate | model_pb2.WeightSection
    :type status_pbs: list[*RevisionStatusPerBalancer]
    """
    entity_pb = util.clone_pb(entity_pb)
    status_pbs.sort(key=lambda status_pb: (status_pb.ctime.seconds, status_pb.ctime.nanos))
    del entity_pb.statuses[:]
    entity_pb.statuses.extend(status_pbs)
    return entity_pb


def copy_per_balancer_statuses(entity_pb, balancer_state_pbs):
    """
    :type entity_pb: model_pb2.Upstream | model_pb2.Backend | model_pb2.EndpointSet | model_pb2.Knob | model_pb2.Certificate | model_pb2.Domain | model_pb2.WeightSection
    :param balancer_state_pbs: Balancer states to take statuses from
    :type balancer_state_pbs: list[model_pb2.BalancerState]
    :rtype: cloned entity_pb
    """
    status_pbs = construct_per_balancer_statuses(entity_pb=entity_pb, balancer_state_pbs=balancer_state_pbs)
    return set_per_balancer_statuses(entity_pb, status_pbs)


def copy_l3_statuses(entity_pb, l3_balancer_state_pb):
    """
    :type entity_pb: model_pb2.L3Balancer | model_pb2.Backend | model_pb2.EndpointSet
    :type l3_balancer_state_pb: model_pb2.L3BalancerState
    :return:
    """
    entity_pb = util.clone_pb(entity_pb)
    entity_pb.ClearField('l3_status')
    entity_id = entity_pb.meta.id

    if isinstance(entity_pb, model_pb2.L3Balancer):
        statuses_pb = l3_balancer_state_pb.l3_balancer
    elif isinstance(entity_pb, model_pb2.Backend):
        if entity_id not in l3_balancer_state_pb.backends:
            return entity_pb
        statuses_pb = l3_balancer_state_pb.backends[entity_id]
    elif isinstance(entity_pb, model_pb2.EndpointSet):
        if entity_id not in l3_balancer_state_pb.endpoint_sets:
            return entity_pb
        statuses_pb = l3_balancer_state_pb.endpoint_sets[entity_id]
    else:
        raise RuntimeError()

    if entity_pb.meta.version not in {s.revision_id for s in statuses_pb.l3_statuses}:
        l3_balancer_state_pb = util.clone_pb(l3_balancer_state_pb)
        if isinstance(entity_pb, model_pb2.L3Balancer):
            l3_status_pbs = l3_balancer_state_pb.l3_balancer.l3_statuses
        elif isinstance(entity_pb, model_pb2.Backend):
            l3_status_pbs = l3_balancer_state_pb.backends[entity_id].l3_statuses
        elif isinstance(entity_pb, model_pb2.EndpointSet):
            l3_status_pbs = l3_balancer_state_pb.endpoint_sets[entity_id].l3_statuses
        else:
            raise RuntimeError()
        l3_state_rev_status_pb = l3_status_pbs.add()
        l3_state_rev_status_pb.validated.status = 'Unknown'
        l3_state_rev_status_pb.in_progress.status = 'False'
        l3_state_rev_status_pb.active.status = 'False'
        l3_state_rev_status_pb.revision_id = entity_pb.meta.version
        l3_state_rev_status_pb.ctime.CopyFrom(entity_pb.meta.mtime)

    if isinstance(entity_pb, model_pb2.L3Balancer):
        statuses_pb = l3_balancer_state_pb.l3_balancer
    elif isinstance(entity_pb, model_pb2.Backend):
        statuses_pb = l3_balancer_state_pb.backends[entity_id]
    elif isinstance(entity_pb, model_pb2.EndpointSet):
        statuses_pb = l3_balancer_state_pb.endpoint_sets[entity_id]
    else:
        raise RuntimeError()

    for l3_state_rev_status_pb in statuses_pb.l3_statuses:
        if l3_state_rev_status_pb.revision_id == entity_pb.meta.version:
            entity_pb.l3_status.validated.CopyFrom(l3_state_rev_status_pb.validated)
            entity_pb.l3_status.in_progress.CopyFrom(l3_state_rev_status_pb.in_progress)
            entity_pb.l3_status.active.CopyFrom(l3_state_rev_status_pb.active)
        l3_balancer_rev_status_pb = entity_pb.l3_status.revisions.add(id=l3_state_rev_status_pb.revision_id)
        l3_balancer_rev_status_pb.ctime.CopyFrom(l3_state_rev_status_pb.ctime)
        l3_balancer_rev_status_pb.validated.CopyFrom(l3_state_rev_status_pb.validated)
        l3_balancer_rev_status_pb.in_progress.CopyFrom(l3_state_rev_status_pb.in_progress)
        l3_balancer_rev_status_pb.active.CopyFrom(l3_state_rev_status_pb.active)
    return entity_pb


def copy_dns_record_statuses(entity_pb, dns_record_state_pb):
    """
    :type entity_pb: model_pb2.DnsRecord
    :type dns_record_state_pb: model_pb2.DnsRecordState
    :return:
    """
    entity_pb = util.clone_pb(entity_pb)
    entity_pb.ClearField('dns_record_status')
    entity_id = entity_pb.meta.id

    if isinstance(entity_pb, model_pb2.DnsRecord):
        statuses_pb = dns_record_state_pb.dns_record
    elif isinstance(entity_pb, model_pb2.Backend):
        if entity_id not in dns_record_state_pb.backends:
            return entity_pb
        statuses_pb = dns_record_state_pb.backends[entity_id]
    elif isinstance(entity_pb, model_pb2.EndpointSet):
        if entity_id not in dns_record_state_pb.endpoint_sets:
            return entity_pb
        statuses_pb = dns_record_state_pb.endpoint_sets[entity_id]
    else:
        raise RuntimeError()

    if entity_pb.meta.version not in {s.revision_id for s in statuses_pb.statuses}:
        dns_record_state_pb = util.clone_pb(dns_record_state_pb)
        if isinstance(entity_pb, model_pb2.DnsRecord):
            dns_record_status_pbs = dns_record_state_pb.dns_record.statuses
        elif isinstance(entity_pb, model_pb2.Backend):
            dns_record_status_pbs = dns_record_state_pb.backends[entity_id].statuses
        elif isinstance(entity_pb, model_pb2.EndpointSet):
            dns_record_status_pbs = dns_record_state_pb.endpoint_sets[entity_id].statuses
        else:
            raise RuntimeError()
        dns_record_state_rev_status_pb = dns_record_status_pbs.add()
        dns_record_state_rev_status_pb.validated.status = 'Unknown'
        dns_record_state_rev_status_pb.revision_id = entity_pb.meta.version
        dns_record_state_rev_status_pb.ctime.CopyFrom(entity_pb.meta.mtime)

    for dns_record_state_rev_status_pb in statuses_pb.statuses:
        if dns_record_state_rev_status_pb.revision_id == entity_pb.meta.version:
            entity_pb.dns_record_status.validated.CopyFrom(dns_record_state_rev_status_pb.validated)
        dns_record_rev_status_pb = entity_pb.dns_record_status.revisions.add(
            id=dns_record_state_rev_status_pb.revision_id)
        dns_record_rev_status_pb.ctime.CopyFrom(dns_record_state_rev_status_pb.ctime)
        dns_record_rev_status_pb.validated.CopyFrom(dns_record_state_rev_status_pb.validated)
    return entity_pb


def copy_l7heavy_config_statuses(entity_pb, l7heavy_config_state_pb):
    """
    :type entity_pb: model_pb2.L7HeavyConfig
    :type l7heavy_config_state_pb: model_pb2.L7HeavyConfigState
    :return:
    """
    entity_pb = util.clone_pb(entity_pb)
    entity_pb.ClearField('l7heavy_config_status')
    entity_id = entity_pb.meta.id

    if isinstance(entity_pb, model_pb2.L7HeavyConfig):
        statuses_pb = l7heavy_config_state_pb.l7heavy_config
    elif isinstance(entity_pb, model_pb2.WeightSection):
        statuses_pb = l7heavy_config_state_pb.weight_sections[entity_id]
    else:
        raise RuntimeError()

    if entity_pb.meta.version not in {s.revision_id for s in statuses_pb.statuses}:
        l7heavy_config_state_pb = util.clone_pb(l7heavy_config_state_pb)
        if isinstance(entity_pb, model_pb2.L7HeavyConfig):
            l7heavy_config_status_pbs = l7heavy_config_state_pb.l7heavy_config.statuses
        elif isinstance(entity_pb, model_pb2.WeightSection):
            l7heavy_config_status_pbs = l7heavy_config_state_pb.weight_sections[entity_id].statuses
        else:
            raise RuntimeError()

        l7heavy_config_state_rev_status_pb = l7heavy_config_status_pbs.add()
        l7heavy_config_state_rev_status_pb.validated.status = 'Unknown'
        l7heavy_config_state_rev_status_pb.in_progress.status = 'False'
        l7heavy_config_state_rev_status_pb.active.status = 'False'
        l7heavy_config_state_rev_status_pb.revision_id = entity_pb.meta.version
        l7heavy_config_state_rev_status_pb.ctime.CopyFrom(entity_pb.meta.mtime)

    if isinstance(entity_pb, model_pb2.L7HeavyConfig):
        statuses_pb = l7heavy_config_state_pb.l7heavy_config
    elif isinstance(entity_pb, model_pb2.WeightSection):
        statuses_pb = l7heavy_config_state_pb.weight_sections[entity_id]
    else:
        raise RuntimeError()

    for l7heavy_config_state_rev_status_pb in statuses_pb.statuses:
        if l7heavy_config_state_rev_status_pb.revision_id == entity_pb.meta.version:
            entity_pb.l7heavy_config_status.validated.CopyFrom(l7heavy_config_state_rev_status_pb.validated)
            entity_pb.l7heavy_config_status.in_progress.CopyFrom(l7heavy_config_state_rev_status_pb.in_progress)
            entity_pb.l7heavy_config_status.active.CopyFrom(l7heavy_config_state_rev_status_pb.active)
        l7heavy_config_rev_status_pb = entity_pb.l7heavy_config_status.revisions.add(
            id=l7heavy_config_state_rev_status_pb.revision_id)
        l7heavy_config_rev_status_pb.ctime.CopyFrom(l7heavy_config_state_rev_status_pb.ctime)
        l7heavy_config_rev_status_pb.validated.CopyFrom(l7heavy_config_state_rev_status_pb.validated)
        l7heavy_config_rev_status_pb.in_progress.CopyFrom(l7heavy_config_state_rev_status_pb.in_progress)
        l7heavy_config_rev_status_pb.active.CopyFrom(l7heavy_config_state_rev_status_pb.active)
    return entity_pb


def copy_per_l7heavy_config_statuses(entity_pb):
    """
    :type entity_pb: model_pb2.WeightSection
    :return:
    """
    entity_pb = util.clone_pb(entity_pb)
    entity_id = entity_pb.meta.id
    namespace_id = entity_pb.meta.namespace_id

    revs = collections.defaultdict(set)
    validated = collections.defaultdict(dict)
    active = collections.defaultdict(dict)
    l7hc_state_pb = objects.L7HeavyConfig.state.cache.get(namespace_id, namespace_id)
    if l7hc_state_pb is not None:
        if isinstance(entity_pb, model_pb2.WeightSection):
            l7hc_status_pbs = l7hc_state_pb.weight_sections[entity_id].statuses
        else:
            raise RuntimeError()

        for l7hc_status_pb in l7hc_status_pbs:
            rev_id = l7hc_status_pb.revision_id
            revs[rev_id].add(l7hc_status_pb.ctime.ToNanoseconds())
            key = namespace_id + ':' + l7hc_state_pb.l7heavy_config_id
            validated[rev_id][key] = l7hc_status_pb.validated
            active[rev_id][key] = l7hc_status_pb.active

    for rev_id, ctimes in revs.items():
        assert len(ctimes) == 1
        revs[rev_id] = list(ctimes)[0]

    curr_rev_id = entity_pb.meta.version
    if curr_rev_id not in revs:
        revs[curr_rev_id] = entity_pb.meta.mtime.ToNanoseconds()
        validated[curr_rev_id] = {}
        active[curr_rev_id] = {}

    l7heavy_config_statuses = []
    for rev_id, ctime in revs.items():
        if isinstance(entity_pb, model_pb2.WeightSection):
            message_cls = model_pb2.WeightSectionRevisionStatusPerL7HeavyConfig
        else:
            raise RuntimeError()

        rev_pb = message_cls(
            id=rev_id,
            validated=validated[rev_id],
            active=active[rev_id],
        )
        rev_pb.ctime.FromNanoseconds(ctime)
        l7heavy_config_statuses.append(rev_pb)

    del entity_pb.l7heavy_config_statuses[:]

    l7heavy_config_statuses = sorted(l7heavy_config_statuses, key=lambda x: x.ctime.ToNanoseconds())
    entity_pb.l7heavy_config_statuses.extend(l7heavy_config_statuses)
    return entity_pb


def copy_per_balancer_l3_statuses(entity_pb):
    """
    :type entity_pb: model_pb2.Backend | model_pb2.EndpointSet
    :return:
    """
    entity_pb = util.clone_pb(entity_pb)
    entity_id = entity_pb.meta.id
    namespace_id = entity_pb.meta.namespace_id

    c = cache.IAwacsCache.instance()

    revs = collections.defaultdict(set)
    validated = collections.defaultdict(dict)
    in_progress = collections.defaultdict(dict)
    active = collections.defaultdict(dict)
    for l3_balancer_state_pb in c.list_all_l3_balancer_states(namespace_id=namespace_id):
        if isinstance(entity_pb, model_pb2.Backend):
            if entity_id not in l3_balancer_state_pb.backends:
                continue
            l3_status_pbs = l3_balancer_state_pb.backends[entity_id].l3_statuses
        elif isinstance(entity_pb, model_pb2.EndpointSet):
            if entity_id not in l3_balancer_state_pb.endpoint_sets:
                continue
            l3_status_pbs = l3_balancer_state_pb.endpoint_sets[entity_id].l3_statuses
        else:
            raise RuntimeError()

        for l3_status_pb in l3_status_pbs:
            rev_id = l3_status_pb.revision_id
            revs[rev_id].add(l3_status_pb.ctime.ToNanoseconds())
            key = namespace_id + ':' + l3_balancer_state_pb.l3_balancer_id
            validated[rev_id][key] = l3_status_pb.validated
            in_progress[rev_id][key] = l3_status_pb.in_progress
            active[rev_id][key] = l3_status_pb.active

    for rev_id, ctimes in revs.items():
        assert len(ctimes) == 1
        revs[rev_id] = list(ctimes)[0]

    curr_rev_id = entity_pb.meta.version
    if curr_rev_id not in revs:
        revs[curr_rev_id] = entity_pb.meta.mtime.ToNanoseconds()
        validated[curr_rev_id] = in_progress[curr_rev_id] = active[curr_rev_id] = {}

    l3_statuses = []
    for rev_id, ctime in revs.items():
        if isinstance(entity_pb, model_pb2.Backend):
            message_cls = model_pb2.BackendRevisionL3StatusPerBalancer
        elif isinstance(entity_pb, model_pb2.EndpointSet):
            message_cls = model_pb2.EndpointSetRevisionL3StatusPerBalancer
        else:
            raise RuntimeError()

        rev_pb = message_cls(
            id=rev_id,
            validated=validated[rev_id],
            in_progress=in_progress[rev_id],
            active=active[rev_id],
        )
        rev_pb.ctime.FromNanoseconds(ctime)
        l3_statuses.append(rev_pb)

    del entity_pb.l3_statuses[:]

    l3_statuses = sorted(l3_statuses, key=lambda x: x.ctime.ToNanoseconds())
    entity_pb.l3_statuses.extend(l3_statuses)
    return entity_pb


def copy_per_balancer_dns_record_statuses(entity_pb):
    """
    :type entity_pb: model_pb2.Backend | model_pb2.EndpointSet
    :return:
    """
    entity_pb = util.clone_pb(entity_pb)
    entity_id = entity_pb.meta.id
    namespace_id = entity_pb.meta.namespace_id

    c = cache.IAwacsCache.instance()

    revs = collections.defaultdict(set)
    validated = collections.defaultdict(dict)
    for dns_record_state_pb in c.list_all_dns_record_states(namespace_id=namespace_id):
        if isinstance(entity_pb, model_pb2.Backend):
            dns_record_status_pbs = dns_record_state_pb.backends[entity_id].statuses
        elif isinstance(entity_pb, model_pb2.EndpointSet):
            dns_record_status_pbs = dns_record_state_pb.endpoint_sets[entity_id].statuses
        else:
            raise RuntimeError()

        for dns_record_status_pb in dns_record_status_pbs:
            rev_id = dns_record_status_pb.revision_id
            revs[rev_id].add(dns_record_status_pb.ctime.ToNanoseconds())
            key = namespace_id + ':' + dns_record_state_pb.dns_record_id
            validated[rev_id][key] = dns_record_status_pb.validated

    for rev_id, ctimes in revs.items():
        assert len(ctimes) == 1
        revs[rev_id] = list(ctimes)[0]

    curr_rev_id = entity_pb.meta.version
    if curr_rev_id not in revs:
        revs[curr_rev_id] = entity_pb.meta.mtime.ToNanoseconds()
        validated[curr_rev_id] = {}

    dns_record_statuses = []
    for rev_id, ctime in revs.items():
        if isinstance(entity_pb, model_pb2.Backend):
            message_cls = model_pb2.BackendRevisionDnsRecordStatusPerBalancer
        elif isinstance(entity_pb, model_pb2.EndpointSet):
            message_cls = model_pb2.EndpointSetRevisionDnsRecordStatusPerBalancer
        else:
            raise RuntimeError()

        rev_pb = message_cls(
            id=rev_id,
            validated=validated[rev_id],
        )
        rev_pb.ctime.FromNanoseconds(ctime)
        dns_record_statuses.append(rev_pb)

    del entity_pb.dns_record_statuses[:]

    dns_record_statuses = sorted(dns_record_statuses, key=lambda x: x.ctime.ToNanoseconds())
    entity_pb.dns_record_statuses.extend(dns_record_statuses)
    return entity_pb


class AwacsApiCache(object):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    L3BalancersQueryTarget = cache.AwacsCache.L3BalancersQueryTarget
    BalancersQueryTarget = cache.AwacsCache.BalancersQueryTarget
    BalancerOpsQueryTarget = cache.AwacsCache.BalancerOpsQueryTarget
    UpstreamsSortTarget = cache.AwacsCache.UpstreamsSortTarget
    UpstreamsQueryTarget = cache.AwacsCache.UpstreamsQueryTarget
    DomainsSortTarget = cache.AwacsCache.DomainsSortTarget
    DomainsQueryTarget = cache.AwacsCache.DomainsQueryTarget
    BackendsSortTarget = cache.AwacsCache.BackendsSortTarget
    BackendsQueryTarget = cache.AwacsCache.BackendsQueryTarget
    KnobsSortTarget = cache.AwacsCache.KnobsSortTarget
    KnobsQueryTarget = cache.AwacsCache.KnobsQueryTarget
    CertsQueryTarget = cache.AwacsCache.CertsQueryTarget
    CertsSortTarget = cache.AwacsCache.CertsSortTarget
    CertRenewalsQueryTarget = cache.AwacsCache.CertRenewalsQueryTarget
    CertRenewalsSortTarget = cache.AwacsCache.CertRenewalsSortTarget

    DEFAULT_UPSTREAMS_LIMIT = 1000
    MAX_UPSTREAMS_LIMIT = 2000

    DEFAULT_DNS_RECORDS_LIMIT = 1000
    MAX_DNS_RECORDS_LIMIT = 2000

    DEFAULT_NAME_SERVERS_LIMIT = 1000
    MAX_NAME_SERVERS_LIMIT = 2000

    DEFAULT_L3_BALANCERS_LIMIT = 1000
    MAX_L3_BALANCERS_LIMIT = 2000

    DEFAULT_BALANCERS_LIMIT = 1000
    MAX_BALANCERS_LIMIT = 2000

    DEFAULT_BACKENDS_LIMIT = 1000
    MAX_BACKENDS_LIMIT = 2000

    DEFAULT_ENDPOINT_SETS_LIMIT = 1000
    MAX_ENDPOINT_SETS_LIMIT = 2000

    DEFAULT_KNOBS_LIMIT = 1000
    MAX_KNOBS_LIMIT = 2000

    DEFAULT_WEIGHT_SECTIONS_LIMIT = 1000
    MAX_WEIGHT_SECTIONS_LIMIT = 2000

    DEFAULT_CERTS_LIMIT = 1000
    MAX_CERTS_LIMIT = 2000

    DEFAULT_DOMAINS_LIMIT = 1000
    MAX_DOMAINS_LIMIT = 2000

    DEFAULT_OP_LIMIT = 1000
    MAX_OP_LIMIT = 2000


    def get_namespace(self, namespace_id):
        return self._cache.get_namespace(namespace_id)

    def must_get_namespace(self, namespace_id):
        return self._cache.must_get_namespace(namespace_id)

    def list_balancers(self, namespace_id=None, query=None, skip=None, limit=None):
        balancer_pbs = self._cache.list_all_balancers(namespace_id=namespace_id, query=query)
        total = len(balancer_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_BALANCERS_LIMIT,
                                        max_limit=self.MAX_BALANCERS_LIMIT)
        items = []
        for balancer_pb in balancer_pbs[page]:  # type: model_pb2.Balancer
            balancer_id = balancer_pb.meta.id
            balancer_state_pb = self._cache.get_balancer_state_or_empty(
                namespace_id=balancer_pb.meta.namespace_id, balancer_id=balancer_id)
            balancer_pb = copy_statuses(balancer_pb, balancer_state_pb)
            items.append(balancer_pb)
        return pagination.SliceResult(items=items, total=total)

    def get_balancer(self, namespace_id, balancer_id):
        balancer_pb = self._cache.get_balancer(namespace_id=namespace_id, balancer_id=balancer_id)
        if balancer_pb:
            balancer_state_pb = self._cache.get_balancer_state_or_empty(
                namespace_id=namespace_id, balancer_id=balancer_id)
            balancer_pb = copy_statuses(balancer_pb, balancer_state_pb)
        return balancer_pb

    def must_get_balancer(self, namespace_id, balancer_id):
        balancer_pb = self.get_balancer(namespace_id, balancer_id)
        if not balancer_pb:
            raise errors.NotFoundError('Balancer "{}:{}" not found'.format(namespace_id, balancer_id))
        return balancer_pb

    def list_namespace_balancer_operations(self, namespace_id, query=None, skip=None, limit=None):
        balancer_op_pbs = self._cache.list_all_balancer_operations(namespace_id)

        matching_ids = {balancer_op_pb.meta.id for balancer_op_pb in balancer_op_pbs}

        query = query or {}
        id_in = query.get(self.BalancerOpsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        filtered_balancer_ops = []
        for balancer_op_pb in balancer_op_pbs:
            if balancer_op_pb.meta.id in matching_ids:
                filtered_balancer_ops.append(balancer_op_pb)

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_BALANCERS_LIMIT,
                                        max_limit=self.MAX_BALANCERS_LIMIT)
        paged_filtered_balancer_ops = filtered_balancer_ops[page]
        return pagination.SliceResult(items=paged_filtered_balancer_ops, total=len(filtered_balancer_ops))

    def get_balancer_operation(self, namespace_id, balancer_id):
        return self._cache.get_balancer_operation(namespace_id, balancer_id)

    def must_get_balancer_operation(self, namespace_id, balancer_id):
        balancer_op_pb = self.get_balancer_operation(namespace_id, balancer_id)
        if not balancer_op_pb:
            raise errors.NotFoundError('Balancer operation "{}" not found in namespace "{}"'.format(balancer_id,
                                                                                                    namespace_id))
        return balancer_op_pb

    def get_dns_record(self, namespace_id, dns_record_id):
        dns_record_pb = self._cache.get_dns_record(namespace_id=namespace_id, dns_record_id=dns_record_id)
        if dns_record_pb:
            dns_record_state_pb = self._cache.get_dns_record_state_or_empty(
                namespace_id=namespace_id, dns_record_id=dns_record_id)
            dns_record_pb = copy_dns_record_statuses(dns_record_pb, dns_record_state_pb)  # TODO
        return dns_record_pb

    def list_dns_records(self, namespace_id=None, skip=None, limit=None):
        dns_record_pbs = self._cache.list_all_dns_records(namespace_id=namespace_id)
        total = len(dns_record_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_DNS_RECORDS_LIMIT,
                                        max_limit=self.MAX_DNS_RECORDS_LIMIT)
        items = []
        for dns_record_pb in dns_record_pbs[page]:
            dns_record_id = dns_record_pb.meta.id
            dns_record_state_pb = self._cache.get_dns_record_state_or_empty(
                namespace_id=namespace_id, dns_record_id=dns_record_id)
            dns_record_pb = copy_dns_record_statuses(dns_record_pb, dns_record_state_pb)  # TODO
            items.append(dns_record_pb)
        return pagination.SliceResult(items=items, total=total)

    def list_dns_record_operations(self, namespace_id, dns_record_id=None, skip=None, limit=None):
        dns_record_op_pbs = self._cache.list_all_dns_record_operations(namespace_id=namespace_id)
        total = len(dns_record_op_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_DNS_RECORDS_LIMIT,
                                        max_limit=self.MAX_DNS_RECORDS_LIMIT)
        return pagination.SliceResult(items=dns_record_op_pbs[page], total=total)

    def list_name_servers(self, namespace_id=None, skip=None, limit=None):
        name_server_pbs = self._cache.list_all_name_servers(namespace_id=namespace_id)
        total = len(name_server_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_NAME_SERVERS_LIMIT,
                                        max_limit=self.MAX_NAME_SERVERS_LIMIT)
        items = []
        for name_server_pb in name_server_pbs[page]:
            items.append(name_server_pb)
        return pagination.SliceResult(items=items, total=total)

    def list_l3_balancers(self, namespace_id=None, query=None, skip=None, limit=None):
        l3_balancer_pbs = self._cache.list_all_l3_balancers(namespace_id=namespace_id, query=query)
        total = len(l3_balancer_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_L3_BALANCERS_LIMIT,
                                        max_limit=self.MAX_L3_BALANCERS_LIMIT)
        items = []
        for l3_balancer_pb in l3_balancer_pbs[page]:
            l3_balancer_id = l3_balancer_pb.meta.id
            l3_balancer_state_pb = self._cache.get_l3_balancer_state_or_empty(
                namespace_id=namespace_id, l3_balancer_id=l3_balancer_id)
            l3_balancer_pb = copy_l3_statuses(l3_balancer_pb, l3_balancer_state_pb)  # TODO
            items.append(l3_balancer_pb)
        return pagination.SliceResult(items=items, total=total)

    def get_l3_balancer(self, namespace_id, l3_balancer_id):
        l3_balancer_pb = self._cache.get_l3_balancer(namespace_id=namespace_id, l3_balancer_id=l3_balancer_id)
        if l3_balancer_pb:
            l3_balancer_state_pb = self._cache.get_l3_balancer_state_or_empty(
                namespace_id=namespace_id, l3_balancer_id=l3_balancer_id)
            l3_balancer_pb = copy_l3_statuses(l3_balancer_pb, l3_balancer_state_pb)  # TODO
        return l3_balancer_pb

    def must_get_l3_balancer(self, namespace_id, l3_balancer_id):
        l3_balancer_pb = self.get_l3_balancer(namespace_id, l3_balancer_id)
        if not l3_balancer_pb:
            raise errors.NotFoundError('L3 balancer "{}:{}" not found'.format(namespace_id, l3_balancer_id))
        return l3_balancer_pb

    def list_balancer_upstreams(self, namespace_id, balancer_id,
                                query=None, sort=(UpstreamsSortTarget.ID, 1), skip=None, limit=None):
        balancer_state_pb = self._cache.get_balancer_state_or_empty(namespace_id=namespace_id, balancer_id=balancer_id)

        q = {k: v for k, v in six.iteritems(query) if k not in (self.UpstreamsQueryTarget.VALIDATED_STATUS_IN,
                                                                self.UpstreamsQueryTarget.IN_PROGRESS_STATUS_IN,
                                                                self.UpstreamsQueryTarget.ACTIVE_STATUS_IN)}
        # Statuses are not filled yet, so we do not need to filter them in list_all_upstreams
        upstream_pbs = self._cache.list_all_upstreams(namespace_id, sort=sort, query=q)

        upstream_id_by_validated_status = collections.defaultdict(set)
        upstream_id_by_in_progress_status = collections.defaultdict(set)
        upstream_id_by_active_status = collections.defaultdict(set)
        for upstream_pb in upstream_pbs:
            upstream_id = upstream_pb.meta.id
            curr_upstream_version = upstream_pb.meta.version
            if upstream_id not in balancer_state_pb.upstreams:
                continue
            rev_status_pb = util.find_rev_status_by_revision_id(
                balancer_state_pb.upstreams[upstream_id].statuses, curr_upstream_version)
            if rev_status_pb:
                upstream_id_by_validated_status[rev_status_pb.validated.status].add(upstream_id)
                upstream_id_by_in_progress_status[rev_status_pb.in_progress.status].add(upstream_id)
                upstream_id_by_active_status[rev_status_pb.active.status].add(upstream_id)
            else:
                upstream_id_by_validated_status['Unknown'].add(upstream_id)
                upstream_id_by_in_progress_status['False'].add(upstream_id)
                upstream_id_by_active_status['False'].add(upstream_id)

        matching_ids = {upstream_pb.meta.id for upstream_pb in upstream_pbs}

        query = query or {}
        id_in = query.get(self.UpstreamsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        id_regexp = query.get(self.UpstreamsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {upstream_id for upstream_id in matching_ids if id_regexp.search(upstream_id)}

        validated_status_in = query.get(self.UpstreamsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_ids = matching_ids & set.union(
                *[upstream_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(self.UpstreamsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_ids = matching_ids & set.union(
                *[upstream_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(self.UpstreamsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_ids = matching_ids & set.union(
                *[upstream_id_by_active_status.get(s) or set() for s in active_status_in])

        all_items = []
        for upstream_pb in upstream_pbs:
            if upstream_pb.meta.id in matching_ids:
                all_items.append(upstream_pb)

        items = []
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_UPSTREAMS_LIMIT,
                                        max_limit=self.MAX_UPSTREAMS_LIMIT)
        for upstream_pb in all_items[page]:
            upstream_pb = copy_statuses(upstream_pb, balancer_state_pb)
            items.append(upstream_pb)
        return pagination.SliceResult(items=items, total=len(all_items))

    def list_namespace_upstreams(self, namespace_id,
                                 query=None, sort=(UpstreamsSortTarget.ID, 1),
                                 skip=None, limit=None):
        q = {k: v for k, v in six.iteritems(query) if k not in (self.UpstreamsQueryTarget.VALIDATED_STATUS_IN,
                                                                self.UpstreamsQueryTarget.IN_PROGRESS_STATUS_IN,
                                                                self.UpstreamsQueryTarget.ACTIVE_STATUS_IN)}
        upstream_pbs = self._cache.list_all_upstreams(namespace_id, sort=sort, query=q)

        # upstreams can only be used within their namespace
        # let's make use of this fact and filter balancer states by `namespace_id` before
        # we call `construct_per_balancer_statuses`
        balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=namespace_id)

        upstreams_w_status_pbs = [
            (upstream_pb, construct_per_balancer_statuses(entity_pb=upstream_pb,
                                                          balancer_state_pbs=balancer_state_pbs))
            for upstream_pb in upstream_pbs
        ]

        upstream_id_by_validated_status = collections.defaultdict(set)
        upstream_id_by_in_progress_status = collections.defaultdict(set)
        upstream_id_by_active_status = collections.defaultdict(set)
        for upstream_pb, status_pbs in upstreams_w_status_pbs:
            upstream_id = upstream_pb.meta.id
            curr_upstream_version = upstream_pb.meta.version
            rev_status_pb = util.find_rev_status_by_id(status_pbs, curr_upstream_version)
            if rev_status_pb:
                for _, validated_pb in six.iteritems(rev_status_pb.validated):
                    upstream_id_by_validated_status[validated_pb.status].add(upstream_id)
                for _, in_progress_pb in six.iteritems(rev_status_pb.in_progress):
                    upstream_id_by_in_progress_status[in_progress_pb.status].add(upstream_id)
                for _, active_pb in six.iteritems(rev_status_pb.active):
                    upstream_id_by_active_status[active_pb.status].add(upstream_id)

        matching_ids = {upstream_pb.meta.id for upstream_pb in upstream_pbs}

        query = query or {}
        id_in = query.get(self.UpstreamsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        id_regexp = query.get(self.UpstreamsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {upstream_id for upstream_id in matching_ids if id_regexp.search(upstream_id)}

        validated_status_in = query.get(self.UpstreamsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_ids = matching_ids & set.union(
                *[upstream_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(self.UpstreamsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_ids = matching_ids & set.union(
                *[upstream_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(self.UpstreamsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_ids = matching_ids & set.union(
                *[upstream_id_by_active_status.get(s) or set() for s in active_status_in])

        filtered_upstreams_w_status_pbs = []
        for upstream_pb, status_pbs in upstreams_w_status_pbs:
            if upstream_pb.meta.id in matching_ids:
                filtered_upstreams_w_status_pbs.append((upstream_pb, status_pbs))

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_UPSTREAMS_LIMIT,
                                        max_limit=self.MAX_UPSTREAMS_LIMIT)
        paged_filtered_upstreams_w_status_pbs = filtered_upstreams_w_status_pbs[page]

        item_pbs = []
        for upstream_pb, status_pbs in paged_filtered_upstreams_w_status_pbs:
            item_pbs.append(set_per_balancer_statuses(upstream_pb, status_pbs))
        return pagination.SliceResult(items=item_pbs, total=len(filtered_upstreams_w_status_pbs))

    def get_upstream(self, namespace_id, upstream_id):
        upstream_pb = self._cache.get_upstream(namespace_id, upstream_id)
        if upstream_pb:
            upstream_pb = copy_per_balancer_statuses(
                entity_pb=upstream_pb,
                balancer_state_pbs=self._cache.list_all_balancer_states(namespace_id=namespace_id))
        return upstream_pb

    def must_get_upstream(self, namespace_id, upstream_id):
        upstream_pb = self.get_upstream(namespace_id, upstream_id)
        if not upstream_pb:
            raise errors.NotFoundError('Upstream "{}" not found in namespace "{}"'.format(upstream_id, namespace_id))
        return upstream_pb

    def list_balancer_domains(self, namespace_id, balancer_id,
                              query=None, sort=(DomainsSortTarget.ID, 1), skip=None, limit=None):
        balancer_state_pb = self._cache.get_balancer_state_or_empty(namespace_id=namespace_id, balancer_id=balancer_id)
        domain_pbs = self._cache.list_all_domains(namespace_id, sort=sort)

        domain_id_by_validated_status = collections.defaultdict(set)
        domain_id_by_in_progress_status = collections.defaultdict(set)
        domain_id_by_active_status = collections.defaultdict(set)
        for domain_pb in domain_pbs:
            domain_id = domain_pb.meta.id
            curr_domain_version = domain_pb.meta.version
            if domain_id not in balancer_state_pb.domains:
                continue
            rev_status_pb = util.find_rev_status_by_revision_id(
                balancer_state_pb.domains[domain_id].statuses, curr_domain_version)
            if rev_status_pb:
                domain_id_by_validated_status[rev_status_pb.validated.status].add(domain_id)
                domain_id_by_in_progress_status[rev_status_pb.in_progress.status].add(domain_id)
                domain_id_by_active_status[rev_status_pb.active.status].add(domain_id)
            else:
                domain_id_by_validated_status['Unknown'].add(domain_id)
                domain_id_by_in_progress_status['False'].add(domain_id)
                domain_id_by_active_status['False'].add(domain_id)

        matching_ids = {domain_pb.meta.id for domain_pb in domain_pbs}

        query = query or {}
        id_in = query.get(self.DomainsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        id_regexp = query.get(self.DomainsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {domain_id for domain_id in matching_ids if id_regexp.search(domain_id)}

        validated_status_in = query.get(self.DomainsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_ids = matching_ids & set.union(
                *[domain_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(self.DomainsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_ids = matching_ids & set.union(
                *[domain_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(self.DomainsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_ids = matching_ids & set.union(
                *[domain_id_by_active_status.get(s) or set() for s in active_status_in])

        all_items = []
        for domain_pb in domain_pbs:
            if domain_pb.meta.id in matching_ids:
                all_items.append(domain_pb)

        items = []
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_DOMAINS_LIMIT,
                                        max_limit=self.MAX_DOMAINS_LIMIT)
        for domain_pb in all_items[page]:
            domain_pb = copy_statuses(domain_pb, balancer_state_pb)
            items.append(domain_pb)
        return pagination.SliceResult(items=items, total=len(all_items))

    def list_namespace_domains(self, namespace_id,
                               query=None, sort=(DomainsSortTarget.ID, 1),
                               skip=None, limit=None):
        domain_pbs = self._cache.list_all_domains(namespace_id, sort=sort)
        # domains can only be used within their namespace
        # let's make use of this fact and filter balancer states by `namespace_id` before
        # we call `construct_per_balancer_statuses`
        balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=namespace_id)

        domains_w_status_pbs = [
            (domain_pb, construct_per_balancer_statuses(entity_pb=domain_pb,
                                                        balancer_state_pbs=balancer_state_pbs))
            for domain_pb in domain_pbs
        ]

        domain_id_by_validated_status = collections.defaultdict(set)
        domain_id_by_in_progress_status = collections.defaultdict(set)
        domain_id_by_active_status = collections.defaultdict(set)
        for domain_pb, status_pbs in domains_w_status_pbs:
            domain_id = domain_pb.meta.id
            curr_domain_version = domain_pb.meta.version
            rev_status_pb = util.find_rev_status_by_id(status_pbs, curr_domain_version)
            if rev_status_pb:
                for _, validated_pb in six.iteritems(rev_status_pb.validated):
                    domain_id_by_validated_status[validated_pb.status].add(domain_id)
                for _, in_progress_pb in six.iteritems(rev_status_pb.in_progress):
                    domain_id_by_in_progress_status[in_progress_pb.status].add(domain_id)
                for _, active_pb in six.iteritems(rev_status_pb.active):
                    domain_id_by_active_status[active_pb.status].add(domain_id)

        matching_ids = {domain_pb.meta.id for domain_pb in domain_pbs}

        query = query or {}
        id_in = query.get(self.DomainsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        id_regexp = query.get(self.DomainsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {domain_id for domain_id in matching_ids if id_regexp.search(domain_id)}

        validated_status_in = query.get(self.DomainsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_ids = matching_ids & set.union(
                *[domain_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(self.DomainsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_ids = matching_ids & set.union(
                *[domain_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(self.DomainsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_ids = matching_ids & set.union(
                *[domain_id_by_active_status.get(s) or set() for s in active_status_in])

        filtered_domains_w_status_pbs = []
        for domain_pb, status_pbs in domains_w_status_pbs:
            if domain_pb.meta.id in matching_ids:
                filtered_domains_w_status_pbs.append((domain_pb, status_pbs))

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_DOMAINS_LIMIT,
                                        max_limit=self.MAX_DOMAINS_LIMIT)
        paged_filtered_domains_w_status_pbs = filtered_domains_w_status_pbs[page]

        item_pbs = []
        for domain_pb, status_pbs in paged_filtered_domains_w_status_pbs:
            item_pbs.append(set_per_balancer_statuses(domain_pb, status_pbs))
        return pagination.SliceResult(items=item_pbs, total=len(filtered_domains_w_status_pbs))

    def get_domain(self, namespace_id, domain_id):
        domain_pb = self._cache.get_domain(namespace_id, domain_id)
        if domain_pb:
            domain_pb = copy_per_balancer_statuses(
                entity_pb=domain_pb,
                balancer_state_pbs=self._cache.list_all_balancer_states(namespace_id=namespace_id))
        return domain_pb

    def must_get_domain(self, namespace_id, domain_id):
        domain_pb = self.get_domain(namespace_id, domain_id)
        if not domain_pb:
            raise errors.NotFoundError('Domain "{}" not found in namespace "{}"'.format(domain_id, namespace_id))
        return domain_pb

    def list_namespace_domain_operations(self, namespace_id, query=None, skip=None, limit=None):
        domain_op_pbs = self._cache.list_all_domain_operations(namespace_id)

        matching_ids = {domain_op_pb.meta.id for domain_op_pb in domain_op_pbs}

        query = query or {}
        id_in = query.get(self.DomainsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        id_regexp = query.get(self.DomainsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {domain_id for domain_id in matching_ids if id_regexp.search(domain_id)}

        filtered_domain_ops = []
        for domain_op_pb in domain_op_pbs:
            if domain_op_pb.meta.id in matching_ids:
                filtered_domain_ops.append(domain_op_pb)

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_DOMAINS_LIMIT,
                                        max_limit=self.MAX_DOMAINS_LIMIT)
        paged_filtered_domain_ops = filtered_domain_ops[page]
        return pagination.SliceResult(items=paged_filtered_domain_ops, total=len(filtered_domain_ops))

    def get_domain_operation(self, namespace_id, domain_id):
        return self._cache.get_domain_operation(namespace_id, domain_id)

    def must_get_domain_operation(self, namespace_id, domain_id):
        domain_op_pb = self.get_domain_operation(namespace_id, domain_id)
        if not domain_op_pb:
            raise errors.NotFoundError('Domain operation "{}" not found in namespace "{}"'.format(domain_id,
                                                                                                  namespace_id))
        return domain_op_pb

    def list_balancer_backends(self, namespace_id, balancer_id,
                               query=None, sort=(BackendsSortTarget.ID, 1), skip=None, limit=None):
        balancer_state_pb = self._cache.get_balancer_state_or_empty(namespace_id=namespace_id, balancer_id=balancer_id)
        backend_pbs = self._cache.list_all_backends(sort=sort)

        backend_id_by_validated_status = collections.defaultdict(set)
        backend_id_by_in_progress_status = collections.defaultdict(set)
        backend_id_by_active_status = collections.defaultdict(set)
        system_backend_ids = set()

        filtered_backend_pbs = []
        for backend_pb in backend_pbs:
            full_backend_id = (backend_pb.meta.namespace_id, backend_pb.meta.id)
            backend_id = flatten_full_id(namespace_id, full_backend_id)
            curr_backend_version = backend_pb.meta.version
            if backend_id not in balancer_state_pb.backends:
                continue
            filtered_backend_pbs.append(backend_pb)
            rev_status_pb = util.find_rev_status_by_revision_id(
                balancer_state_pb.backends[backend_id].statuses, curr_backend_version)
            if backend_pb.meta.is_system.value:
                system_backend_ids.add(full_backend_id)
            if rev_status_pb:
                backend_id_by_validated_status[rev_status_pb.validated.status].add(full_backend_id)
                backend_id_by_in_progress_status[rev_status_pb.in_progress.status].add(full_backend_id)
                backend_id_by_active_status[rev_status_pb.active.status].add(full_backend_id)
            else:
                backend_id_by_validated_status['Unknown'].add(full_backend_id)
                backend_id_by_in_progress_status['False'].add(full_backend_id)
                backend_id_by_active_status['False'].add(full_backend_id)

        matching_full_ids = {(backend_pb.meta.namespace_id, backend_pb.meta.id) for backend_pb in filtered_backend_pbs}

        query = query or {}
        id_in = query.get(self.BackendsQueryTarget.ID_IN, [])
        if id_in:
            matching_full_ids = {full_id for full_id in matching_full_ids if full_id[1] in set(id_in)}

        id_regexp = query.get(self.BackendsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_full_ids = {full_id for full_id in matching_full_ids if id_regexp.search(full_id[1])}

        only_system = query.get(self.BackendsQueryTarget.ONLY_SYSTEM, False)
        if only_system:
            matching_full_ids = {full_id for full_id in matching_full_ids if full_id in system_backend_ids}

        exclude_system = query.get(self.BackendsQueryTarget.EXCLUDE_SYSTEM, False)
        if exclude_system:
            matching_full_ids = {full_id for full_id in matching_full_ids if full_id not in system_backend_ids}

        validated_status_in = query.get(self.BackendsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_full_ids = matching_full_ids & set.union(
                *[backend_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(self.BackendsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_full_ids = matching_full_ids & set.union(
                *[backend_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(self.BackendsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_full_ids = matching_full_ids & set.union(
                *[backend_id_by_active_status.get(s) or set() for s in active_status_in])

        all_items = []
        for backend_pb in filtered_backend_pbs:
            if (backend_pb.meta.namespace_id, backend_pb.meta.id) in matching_full_ids:
                all_items.append(backend_pb)

        items = []
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_BACKENDS_LIMIT,
                                        max_limit=self.MAX_BACKENDS_LIMIT)
        all_balancer_state_pbs = None
        namespace_balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=namespace_id)
        for backend_pb in gevent_idle_iter(all_items[page], idle_period=30):
            if backend_pb.spec.is_global.value:
                if all_balancer_state_pbs is None:
                    # compute `all_balancer_state_pbs` only if necessary
                    all_balancer_state_pbs = self._cache.list_all_balancer_states()
                balancer_state_pbs = all_balancer_state_pbs
            else:
                balancer_state_pbs = namespace_balancer_state_pbs
            backend_pb = copy_per_balancer_statuses(entity_pb=backend_pb,
                                                    balancer_state_pbs=balancer_state_pbs)
            backend_pb = copy_statuses(backend_pb, balancer_state_pb)
            backend_pb = copy_per_balancer_l3_statuses(backend_pb)
            backend_pb = copy_per_balancer_dns_record_statuses(backend_pb)
            items.append(backend_pb)
        return pagination.SliceResult(items=items, total=len(all_items))

    def list_namespace_backends(self, namespace_id,
                                query=None, sort=(BackendsSortTarget.ID, 1), skip=None, limit=None):
        backend_pbs = self._cache.list_all_backends(namespace_id, sort=sort)
        all_balancer_state_pbs = None
        namespace_balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=namespace_id)

        annotated_backend_pbs = []
        for backend_pb in gevent_idle_iter(backend_pbs, idle_period=30):
            if backend_pb.spec.is_global.value:
                if all_balancer_state_pbs is None:
                    # compute `all_balancer_state_pbs` only if necessary
                    all_balancer_state_pbs = self._cache.list_all_balancer_states()
                balancer_state_pbs = all_balancer_state_pbs
            else:
                balancer_state_pbs = namespace_balancer_state_pbs
            annotated_backend_pbs.append(copy_per_balancer_statuses(entity_pb=backend_pb,
                                                                    balancer_state_pbs=balancer_state_pbs))
        backend_pbs = annotated_backend_pbs

        backend_pbs = list(map(copy_per_balancer_l3_statuses, backend_pbs))
        backend_pbs = list(map(copy_per_balancer_dns_record_statuses, backend_pbs))

        backend_id_by_validated_status = collections.defaultdict(set)
        backend_id_by_in_progress_status = collections.defaultdict(set)
        backend_id_by_active_status = collections.defaultdict(set)
        system_backend_ids = set()
        for backend_pb in backend_pbs:
            backend_id = backend_pb.meta.id
            curr_backend_version = backend_pb.meta.version
            rev_status_pb = util.find_rev_status_by_id(backend_pb.statuses, curr_backend_version)
            if backend_pb.meta.is_system.value:
                system_backend_ids.add(backend_id)
            if rev_status_pb:
                for _, validated_pb in six.iteritems(rev_status_pb.validated):
                    backend_id_by_validated_status[validated_pb.status].add(backend_id)
                for _, in_progress_pb in six.iteritems(rev_status_pb.in_progress):
                    backend_id_by_in_progress_status[in_progress_pb.status].add(backend_id)
                for _, active_pb in six.iteritems(rev_status_pb.active):
                    backend_id_by_active_status[active_pb.status].add(backend_id)

        matching_ids = {backend_pb.meta.id for backend_pb in backend_pbs}

        query = query or {}
        id_in = query.get(self.BackendsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        id_regexp = query.get(self.BackendsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {backend_id for backend_id in matching_ids if id_regexp.search(backend_id)}

        only_system = query.get(self.BackendsQueryTarget.ONLY_SYSTEM, False)
        if only_system:
            matching_ids = {backend_id for backend_id in matching_ids if backend_id in system_backend_ids}

        exclude_system = query.get(self.BackendsQueryTarget.EXCLUDE_SYSTEM, False)
        if exclude_system:
            matching_ids = {backend_id for backend_id in matching_ids if backend_id not in system_backend_ids}

        validated_status_in = query.get(self.BackendsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_ids = matching_ids & set.union(
                *[backend_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(self.BackendsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_ids = matching_ids & set.union(
                *[backend_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(self.BackendsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_ids = matching_ids & set.union(
                *[backend_id_by_active_status.get(s) or set() for s in active_status_in])

        all_items = []
        for backend_pb in backend_pbs:
            if backend_pb.meta.id in matching_ids:
                all_items.append(backend_pb)

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_BACKENDS_LIMIT,
                                        max_limit=self.MAX_BACKENDS_LIMIT)
        return pagination.SliceResult(items=all_items[page], total=len(all_items))

    def list_backends(self, query=None, sort=(BackendsSortTarget.ID, 1), skip=None, limit=None):
        backend_pbs = self._cache.list_all_backends(sort=sort, query=query)
        backend_pbs_by_path = {}

        backend_id_by_validated_status = collections.defaultdict(set)
        backend_id_by_in_progress_status = collections.defaultdict(set)
        backend_id_by_active_status = collections.defaultdict(set)

        query = query or {}
        statuses_copied = False
        all_balancer_state_pbs = None
        if (self.BackendsQueryTarget.VALIDATED_STATUS_IN in query or
                self.BackendsQueryTarget.IN_PROGRESS_STATUS_IN in query or
                self.BackendsQueryTarget.ACTIVE_STATUS_IN in query):

            annotated_backend_pbs = []
            for backend_pb in gevent_idle_iter(backend_pbs, idle_period=100):
                if backend_pb.spec.is_global.value:
                    if all_balancer_state_pbs is None:
                        # compute `all_balancer_state_pbs` only if necessary
                        all_balancer_state_pbs = self._cache.list_all_balancer_states()
                    balancer_state_pbs = all_balancer_state_pbs
                else:
                    balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=backend_pb.meta.namespace_id)
                annotated_backend_pbs.append(copy_per_balancer_statuses(entity_pb=backend_pb,
                                                                        balancer_state_pbs=balancer_state_pbs))
            backend_pbs = annotated_backend_pbs

            backend_pbs = list(map(copy_per_balancer_l3_statuses, backend_pbs))
            backend_pbs = list(map(copy_per_balancer_dns_record_statuses, backend_pbs))
            statuses_copied = True
            for backend_pb in gevent_idle_iter(backend_pbs, idle_period=50):
                backend_path = objects.BackendDescriptor.uid_to_zk_path(backend_pb.meta.namespace_id,
                                                                        backend_pb.meta.id)
                curr_backend_version = backend_pb.meta.version
                rev_status_pb = util.find_rev_status_by_id(backend_pb.statuses, curr_backend_version)
                if rev_status_pb:
                    for _, validated_pb in six.iteritems(rev_status_pb.validated):
                        backend_id_by_validated_status[validated_pb.status].add(backend_path)
                    for _, in_progress_pb in six.iteritems(rev_status_pb.in_progress):
                        backend_id_by_in_progress_status[in_progress_pb.status].add(backend_path)
                    for _, active_pb in six.iteritems(rev_status_pb.active):
                        backend_id_by_active_status[active_pb.status].add(backend_path)

            for backend_pb in backend_pbs:
                backend_path = objects.BackendDescriptor.uid_to_zk_path(backend_pb.meta.namespace_id,
                                                                        backend_pb.meta.id)
                backend_pbs_by_path[backend_path] = backend_pb

            matching_paths = set(backend_pbs)

            validated_status_in = query.get(self.BackendsQueryTarget.VALIDATED_STATUS_IN, [])
            if validated_status_in:
                matching_paths = matching_paths & set.union(
                    *[backend_id_by_validated_status.get(status) or set() for status in validated_status_in])

            in_progress_status_in = query.get(self.BackendsQueryTarget.IN_PROGRESS_STATUS_IN, [])
            if in_progress_status_in:
                matching_paths = matching_paths & set.union(
                    *[backend_id_by_in_progress_status.get(status) or set() for status in in_progress_status_in])

            active_status_in = query.get(self.BackendsQueryTarget.ACTIVE_STATUS_IN, [])
            if active_status_in:
                matching_paths = matching_paths & set.union(
                    *[backend_id_by_active_status.get(status) or set() for status in active_status_in])

            backend_pbs = list(map(backend_pbs_by_path.get, matching_paths))

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_BACKENDS_LIMIT,
                                        max_limit=self.MAX_BACKENDS_LIMIT)
        item_pbs = backend_pbs[page]
        if not statuses_copied:
            all_balancer_state_pbs = None

            annotated_item_pbs = []
            for item_pb in gevent_idle_iter(item_pbs, idle_period=30):
                if item_pb.spec.is_global.value:
                    if all_balancer_state_pbs is None:
                        # compute `all_balancer_state_pbs` only if necessary
                        all_balancer_state_pbs = self._cache.list_all_balancer_states()
                    balancer_state_pbs = all_balancer_state_pbs
                else:
                    balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=item_pb.meta.namespace_id)
                annotated_item_pbs.append(copy_per_balancer_statuses(entity_pb=item_pb,
                                                                     balancer_state_pbs=balancer_state_pbs))
            item_pbs = annotated_item_pbs
            item_pbs = list(map(copy_per_balancer_l3_statuses, item_pbs))
            item_pbs = list(map(copy_per_balancer_dns_record_statuses, item_pbs))
        return pagination.SliceResult(items=item_pbs, total=len(backend_pbs))

    def get_backend(self, namespace_id, backend_id):
        backend_pb = self._cache.get_backend(namespace_id, backend_id)
        if backend_pb:
            if backend_pb.spec.is_global.value:
                balancer_state_pbs = self._cache.list_all_balancer_states()
            else:
                balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=namespace_id)
            backend_pb = copy_per_balancer_statuses(entity_pb=backend_pb,
                                                    balancer_state_pbs=balancer_state_pbs)
            backend_pb = copy_per_balancer_l3_statuses(backend_pb)
            backend_pb = copy_per_balancer_dns_record_statuses(backend_pb)
        return backend_pb

    def must_get_backend(self, namespace_id, backend_id):
        backend_pb = self.get_backend(namespace_id, backend_id)
        if not backend_pb:
            raise errors.NotFoundError('Backend "{}" not found in namespace "{}"'.format(backend_id, namespace_id))
        return backend_pb

    def get_knob(self, namespace_id, knob_id):
        knob_pb = self._cache.get_knob(namespace_id, knob_id)
        if knob_pb:
            knob_pb = copy_per_balancer_statuses(
                entity_pb=knob_pb,
                balancer_state_pbs=self._cache.list_all_balancer_states(namespace_id=namespace_id))
        return knob_pb

    def must_get_knob(self, namespace_id, knob_id):
        knob_pb = self.get_knob(namespace_id, knob_id)
        if not knob_pb:
            raise errors.NotFoundError('Knob "{}" not found in namespace "{}"'.format(knob_id, namespace_id))
        return knob_pb

    def list_balancer_knobs(self, namespace_id, balancer_id,
                            query=None, sort=(KnobsSortTarget.ID, 1), skip=None, limit=None):
        balancer_state_pb = self._cache.get_balancer_state_or_empty(namespace_id=namespace_id, balancer_id=balancer_id)
        knob_pbs = self._cache.list_all_knobs(namespace_id, sort=sort)

        knob_id_by_validated_status = collections.defaultdict(set)
        knob_id_by_in_progress_status = collections.defaultdict(set)
        knob_id_by_active_status = collections.defaultdict(set)
        knob_id_by_mode = collections.defaultdict(set)
        for knob_pb in knob_pbs:
            knob_id = knob_pb.meta.id
            curr_knob_version = knob_pb.meta.version
            knob_id_by_mode[knob_pb.spec.mode].add(knob_id)
            if knob_id not in balancer_state_pb.knobs:
                continue
            rev_status_pb = util.find_rev_status_by_revision_id(balancer_state_pb.knobs[knob_id].statuses,
                                                                curr_knob_version)
            if rev_status_pb:
                knob_id_by_validated_status[rev_status_pb.validated.status].add(knob_id)
                knob_id_by_in_progress_status[rev_status_pb.in_progress.status].add(knob_id)
                knob_id_by_active_status[rev_status_pb.active.status].add(knob_id)
            else:
                knob_id_by_validated_status['Unknown'].add(knob_id)
                knob_id_by_in_progress_status['False'].add(knob_id)
                knob_id_by_active_status['False'].add(knob_id)

        matching_ids = {knob_pb.meta.id for knob_pb in knob_pbs}

        query = query or {}
        id_in = query.get(self.KnobsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        id_regexp = query.get(self.KnobsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {knob_id for knob_id in matching_ids if id_regexp.search(knob_id)}

        validated_status_in = query.get(self.KnobsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_ids = matching_ids & set.union(
                *[knob_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(self.KnobsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_ids = matching_ids & set.union(
                *[knob_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(self.KnobsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_ids = matching_ids & set.union(
                *[knob_id_by_active_status.get(s) or set() for s in active_status_in])

        if self.KnobsQueryTarget.MODE in query:
            mode = query[self.KnobsQueryTarget.MODE]
            matching_ids = matching_ids & knob_id_by_mode[mode]

        all_items = []
        for knob_pb in knob_pbs:
            if knob_pb.meta.id in matching_ids:
                all_items.append(knob_pb)

        items = []
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_KNOBS_LIMIT,
                                        max_limit=self.MAX_KNOBS_LIMIT)
        for knob_pb in all_items[page]:
            knob_pb = copy_statuses(knob_pb, balancer_state_pb)
            items.append(knob_pb)
        return pagination.SliceResult(items=items, total=len(all_items))

    def list_namespace_knobs(self, namespace_id,
                             query=None, sort=(KnobsSortTarget.ID, 1), skip=None, limit=None):
        knob_pbs = self._cache.list_all_knobs(namespace_id, sort=sort)
        balancer_state_pbs = self._cache.list_all_balancer_states()
        knob_pbs = [copy_per_balancer_statuses(entity_pb=knob_pb, balancer_state_pbs=balancer_state_pbs)
                    for knob_pb in knob_pbs]

        knob_id_by_validated_status = collections.defaultdict(set)
        knob_id_by_in_progress_status = collections.defaultdict(set)
        knob_id_by_active_status = collections.defaultdict(set)
        knob_id_by_mode = collections.defaultdict(set)
        for knob_pb in knob_pbs:
            knob_id = knob_pb.meta.id
            curr_knob_version = knob_pb.meta.version
            knob_id_by_mode[knob_pb.spec.mode].add(knob_id)
            rev_status_pb = util.find_rev_status_by_id(knob_pb.statuses, curr_knob_version)
            if rev_status_pb:
                for _, validated_pb in six.iteritems(rev_status_pb.validated):
                    knob_id_by_validated_status[validated_pb.status].add(knob_id)
                for _, in_progress_pb in six.iteritems(rev_status_pb.in_progress):
                    knob_id_by_in_progress_status[in_progress_pb.status].add(knob_id)
                for _, active_pb in six.iteritems(rev_status_pb.active):
                    knob_id_by_active_status[active_pb.status].add(knob_id)

        matching_ids = {knob_pb.meta.id for knob_pb in knob_pbs}

        query = query or {}
        id_in = query.get(self.KnobsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        id_regexp = query.get(self.KnobsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {knob_id for knob_id in matching_ids if id_regexp.search(knob_id)}

        validated_status_in = query.get(self.KnobsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_ids = matching_ids & set.union(
                *[knob_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(self.KnobsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_ids = matching_ids & set.union(
                *[knob_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(self.KnobsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_ids = matching_ids & set.union(
                *[knob_id_by_active_status.get(s) or set() for s in active_status_in])

        if self.KnobsQueryTarget.MODE in query:
            mode = query[self.KnobsQueryTarget.MODE]
            matching_ids = matching_ids & knob_id_by_mode[mode]

        all_items = []
        for knob_pb in knob_pbs:
            if knob_pb.meta.id in matching_ids:
                all_items.append(knob_pb)

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_KNOBS_LIMIT,
                                        max_limit=self.MAX_KNOBS_LIMIT)
        return pagination.SliceResult(items=all_items[page], total=len(all_items))

    def get_endpoint_set(self, namespace_id, endpoint_set_id):
        endpoint_set_pb = self._cache.get_endpoint_set(namespace_id, endpoint_set_id)
        if endpoint_set_pb:
            if endpoint_set_pb.spec.is_global.value:
                balancer_state_pbs = self._cache.list_all_balancer_states()
            else:
                balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=namespace_id)
            endpoint_set_pb = copy_per_balancer_statuses(entity_pb=endpoint_set_pb,
                                                         balancer_state_pbs=balancer_state_pbs)
            endpoint_set_pb = copy_per_balancer_l3_statuses(endpoint_set_pb)
        return endpoint_set_pb

    def must_get_endpoint_set(self, namespace_id, endpoint_set_id):
        endpoint_set_pb = self.get_endpoint_set(namespace_id, endpoint_set_id)
        if not endpoint_set_pb:
            raise errors.NotFoundError(
                'Endpoint set "{}" not found in namespace "{}"'.format(endpoint_set_id, namespace_id))
        return endpoint_set_pb

    def list_namespace_endpoint_sets(self, namespace_id, skip=None, limit=None):
        endpoint_set_pbs = self._cache.list_all_endpoint_sets(namespace_id)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_ENDPOINT_SETS_LIMIT,
                                        max_limit=self.MAX_ENDPOINT_SETS_LIMIT)
        return pagination.SliceResult(items=endpoint_set_pbs[page], total=len(endpoint_set_pbs))

    def get_cert(self, namespace_id, cert_id):
        cert_pb = self._cache.get_cert(namespace_id, cert_id)
        if cert_pb:
            cert_pb = copy_per_balancer_statuses(entity_pb=cert_pb,
                                                 balancer_state_pbs=self._cache.list_all_balancer_states())
        return cert_pb

    def must_get_cert(self, namespace_id, cert_id):
        cert_pb = self.get_cert(namespace_id, cert_id)
        if not cert_pb:
            raise errors.NotFoundError('Certificate "{}" not found in namespace "{}"'.format(cert_id, namespace_id))
        return cert_pb

    def list_namespace_certs(self, namespace_id,
                             query=None, sort=(CertsSortTarget.ID, 1),
                             skip=None, limit=None):
        cert_pbs = self._cache.list_all_certs(namespace_id, sort=sort)
        balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=namespace_id)
        certs_w_status_pbs = [
            (cert_pb, construct_per_balancer_statuses(entity_pb=cert_pb,
                                                      balancer_state_pbs=balancer_state_pbs))
            for cert_pb in cert_pbs
        ]

        cert_id_by_validated_status = collections.defaultdict(set)
        cert_id_by_in_progress_status = collections.defaultdict(set)
        cert_id_by_active_status = collections.defaultdict(set)
        for cert_pb, status_pbs in certs_w_status_pbs:
            cert_id = cert_pb.meta.id
            curr_cert_version = cert_pb.meta.version
            rev_status_pb = util.find_rev_status_by_id(status_pbs, curr_cert_version)
            if rev_status_pb:
                for _, validated_pb in six.iteritems(rev_status_pb.validated):
                    cert_id_by_validated_status[validated_pb.status].add(cert_id)
                for _, in_progress_pb in six.iteritems(rev_status_pb.in_progress):
                    cert_id_by_in_progress_status[in_progress_pb.status].add(cert_id)
                for _, active_pb in six.iteritems(rev_status_pb.active):
                    cert_id_by_active_status[active_pb.status].add(cert_id)

        matching_ids = {cert_pb.meta.id for cert_pb in cert_pbs}

        query = query or {}
        id_in = query.get(self.CertsQueryTarget.ID_IN, [])
        if id_in:
            matching_ids = set(id_in)

        id_regexp = query.get(self.CertsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {cert_id for cert_id in matching_ids if id_regexp.search(cert_id)}

        validated_status_in = query.get(self.CertsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_ids = matching_ids & set.union(
                *[cert_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(self.CertsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_ids = matching_ids & set.union(
                *[cert_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(self.CertsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_ids = matching_ids & set.union(
                *[cert_id_by_active_status.get(s) or set() for s in active_status_in])

        filtered_certs_w_status_pbs = []
        for cert_pb, status_pbs in certs_w_status_pbs:
            if cert_pb.meta.id in matching_ids:
                filtered_certs_w_status_pbs.append((cert_pb, status_pbs))

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_CERTS_LIMIT,
                                        max_limit=self.MAX_CERTS_LIMIT)
        paged_filtered_certs_w_status_pbs = filtered_certs_w_status_pbs[page]

        item_pbs = []
        for cert_pb, status_pbs in paged_filtered_certs_w_status_pbs:
            item_pbs.append(set_per_balancer_statuses(cert_pb, status_pbs))
        return pagination.SliceResult(items=item_pbs, total=len(filtered_certs_w_status_pbs))

    def list_cert_renewals(self, namespace_id=None, query=None, sort=(CertRenewalsSortTarget.ID, 1), skip=None,
                           limit=None):
        cert_renewal_pbs = self._cache.list_all_cert_renewals(namespace_id=namespace_id, sort=sort)
        matching_ids = {cert_renewal_pb.meta.id for cert_renewal_pb in cert_renewal_pbs}

        query = query or {}

        id_regexp = query.get(self.CertRenewalsQueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {cert_id for cert_id in matching_ids if id_regexp.search(cert_id)}

        filtered_certs_renewal_pbs = []
        incomplete = query.get(self.CertRenewalsQueryTarget.INCOMPLETE)
        paused = query.get(self.CertRenewalsQueryTarget.PAUSED)
        validity_not_before_gte = query.get(self.CertRenewalsQueryTarget.VALIDITY_NOT_BEFORE_GTE)
        validity_not_before_lte = query.get(self.CertRenewalsQueryTarget.VALIDITY_NOT_BEFORE_LTE)
        for cert_pb in cert_renewal_pbs:
            if cert_pb.meta.id not in matching_ids:
                continue
            if incomplete is not None and cert_pb.spec.incomplete is not incomplete:
                continue
            if paused is not None and cert_pb.meta.paused.value is not paused:
                continue
            not_before_dt = cert_pb.spec.fields.validity.not_before.ToDatetime()
            if validity_not_before_gte is not None and not_before_dt < validity_not_before_gte:
                continue
            if validity_not_before_lte is not None and not_before_dt > validity_not_before_lte:
                continue
            filtered_certs_renewal_pbs.append(cert_pb)

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_CERTS_LIMIT,
                                        max_limit=self.MAX_CERTS_LIMIT)
        paged_filtered_certs_renewal_pbs = filtered_certs_renewal_pbs[page]

        return pagination.SliceResult(items=paged_filtered_certs_renewal_pbs, total=len(filtered_certs_renewal_pbs))

    def list_namespace_operations(self, namespace_id, skip=None, limit=None):
        op_pbs = objects.NamespaceOperation.cache.list(namespace_id)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_OP_LIMIT,
                                        max_limit=self.MAX_OP_LIMIT)
        return pagination.SliceResult(items=op_pbs[page], total=len(op_pbs))

    def list_namespace_weight_sections(self, namespace_id, query=None, sort=(objects.WeightSection.cache.SortTarget.ID, 1),
                                        skip=None, limit=None):
        weight_section_pbs = objects.WeightSection.cache.list(namespace_id, sort=sort)
        balancer_state_pbs = self._cache.list_all_balancer_states(namespace_id=namespace_id)

        weight_section_pbs = list(map(copy_per_l7heavy_config_statuses, weight_section_pbs))
        weight_sections_w_status_pbs = [
            (weight_section_pb, construct_per_balancer_statuses(entity_pb=weight_section_pb,
                                                                balancer_state_pbs=balancer_state_pbs))
            for weight_section_pb in weight_section_pbs
        ]

        weight_section_id_by_validated_status = collections.defaultdict(set)
        weight_section_id_by_in_progress_status = collections.defaultdict(set)
        weight_section_id_by_active_status = collections.defaultdict(set)
        for weight_section_pb, status_pbs in weight_sections_w_status_pbs:
            weight_section_id = weight_section_pb.meta.id
            curr_weight_section_version = weight_section_pb.meta.version
            rev_status_pb = util.find_rev_status_by_id(status_pbs, curr_weight_section_version)
            if rev_status_pb:
                for _, validated_pb in six.iteritems(rev_status_pb.validated):
                    weight_section_id_by_validated_status[validated_pb.status].add(weight_section_id)
                for _, in_progress_pb in six.iteritems(rev_status_pb.in_progress):
                    weight_section_id_by_in_progress_status[in_progress_pb.status].add(weight_section_id)
                for _, active_pb in six.iteritems(rev_status_pb.active):
                    weight_section_id_by_active_status[active_pb.status].add(weight_section_id)

        matching_ids = {weight_section_pb.meta.id for weight_section_pb in weight_section_pbs}

        query = query or {}

        id_regexp = query.get(objects.WeightSection.cache.QueryTarget.ID_REGEXP)
        if id_regexp:
            matching_ids = {weight_section_id for weight_section_id in matching_ids if id_regexp.search(weight_section_id)}

        validated_status_in = query.get(objects.WeightSection.cache.QueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            matching_ids = matching_ids & set.union(
                *[weight_section_id_by_validated_status.get(s) or set() for s in validated_status_in])

        in_progress_status_in = query.get(objects.WeightSection.cache.QueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            matching_ids = matching_ids & set.union(
                *[weight_section_id_by_in_progress_status.get(s) or set() for s in in_progress_status_in])

        active_status_in = query.get(objects.WeightSection.cache.QueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            matching_ids = matching_ids & set.union(
                *[weight_section_id_by_active_status.get(s) or set() for s in active_status_in])

        filtered_weight_sections_w_status_pbs = []
        for weight_section_pb, status_pbs in weight_sections_w_status_pbs:
            if weight_section_pb.meta.id in matching_ids:
                filtered_weight_sections_w_status_pbs.append((weight_section_pb, status_pbs))

        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_WEIGHT_SECTIONS_LIMIT,
                                        max_limit=self.MAX_WEIGHT_SECTIONS_LIMIT)
        paged_filtered_weight_sections_w_status_pbs = filtered_weight_sections_w_status_pbs[page]

        item_pbs = []
        for weight_section_pb, status_pbs in paged_filtered_weight_sections_w_status_pbs:
            item_pbs.append(set_per_balancer_statuses(weight_section_pb, status_pbs))
        return pagination.SliceResult(items=item_pbs, total=len(filtered_weight_sections_w_status_pbs))
