# coding: utf-8
import random

import six

from awacs.lib.gutils import gevent_idle_iter
from awacs.lib import ctlmanager
from awacs.model import events
from infra.awacs.proto import model_pb2
from .ctl import BackendCtl


def acquire_timeout_strategy():
    while 1:
        yield random.uniform(8, 12) * ctlmanager.HOUR


def standoff_strategy():
    yield random.uniform(5, 15) * ctlmanager.MINUTE
    yield random.uniform(10, 25) * ctlmanager.MINUTE
    while 1:
        yield random.uniform(4, 12) * ctlmanager.HOUR


def has_actionable_spec(pb):
    """
    :type pb: from model_pb2.Backend
    :rtype: bool
    """
    type_ = pb.spec.selector.type
    return pb.spec.deleted or (type_ != model_pb2.BackendSelector.MANUAL and
                               type_ != model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD)


class BackendCtlManager(ctlmanager.PartyingCtlManager):
    CONTENDING_MEMBERS = 2

    def __init__(self, coord, member_id, party_suffix, cache):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type party_suffix: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        """
        super(BackendCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            member_id=member_id,
            party_suffix=party_suffix,
            name='backend-ctl-manager',
            starting_events=(events.BackendUpdate,),
            stopping_events=(events.BackendRemove,)
        )

    def _get_standoff_strategy(self, ctl_id):
        if six.PY3:
            return standoff_strategy().__next__
        else:
            return standoff_strategy().next

    def _get_acquire_timeout_strategy(self, ctl_id):
        if six.PY3:
            return acquire_timeout_strategy().__next__
        else:
            return acquire_timeout_strategy().next

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        return 'backend("{}:{}") ctl'.format(*ctl_id)

    def _list_all_ctl_ids(self):
        """
        :rtype: [(six.text_type, six.text_type)]
        """
        for backend_pb in gevent_idle_iter(self._cache.list_all_backends(), idle_period=2000):
            if has_actionable_spec(backend_pb):
                namespace_id = backend_pb.meta.namespace_id
                backend_id = backend_pb.meta.id
                yield namespace_id, backend_id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        namespace_id, backend_id = ctl_id
        return BackendCtl(namespace_id, backend_id)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: (six.text_type, six.text_type)
        """
        ctl_id = tuple(event.path.strip('/').split('/'))
        if len(ctl_id) != 2 or not ctl_id[0] or not ctl_id[1]:
            raise AssertionError('"{}" can not be parsed into a correct backend ctl id'.format(event.path))
        return ctl_id

    def _is_starting_event(self, event):
        return (super(BackendCtlManager, self)._is_starting_event(event) and
                has_actionable_spec(event.pb))

    def _is_stopping_event(self, event):
        if super(BackendCtlManager, self)._is_stopping_event(event):
            return True
        return (super(BackendCtlManager, self)._is_starting_event(event) and
                not has_actionable_spec(event.pb))


class BackendCtlManagerV2(ctlmanager.PartyingCtlManagerV2):
    CONTENDING_MEMBERS = 2

    def __init__(self, coord, member_id, party_suffix, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type party_suffix: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        """
        super(BackendCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            member_id=member_id,
            party_suffix=party_suffix,
            entity_name='backend',
            subscribed_events=(events.BackendUpdate, events.BackendRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.BackendRemove):
            return False
        return has_actionable_spec(event.pb)

    def _yield_starting_events(self):
        for backend_pb in gevent_idle_iter(self._cache.list_all_backends(), idle_period=2000):
            namespace_id = backend_pb.meta.namespace_id
            backend_id = backend_pb.meta.id
            yield events.BackendUpdate(path=self._make_event_path_from_ctl_id(namespace_id, backend_id),
                                       pb=backend_pb)

    def _create_ctl(self, ctl_id):
        return BackendCtl(*ctl_id)
